"""RunProgressDialog class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path
import re
import time

# 2. Third party modules
from matplotlib.axes import Axes
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
import psutil
from PySide2.QtCore import QTimer
from PySide2.QtGui import QFont
from PySide2.QtWidgets import QDialog, QDialogButtonBox

# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.hgs.gui import debug_control_dialog, gui_util
from xms.hgs.gui.run_progress_dialog_ui import Ui_dlg_run_progress
from xms.hgs.simulation_runner import sim_runner


def run(exe_key: str, grok_filepath: Path, parent=None, return_immediately: bool = False) -> int:
    """Runs the dialog.

    Args:
        exe_key (str): Key used to identify the executable (GROK_EXE, PHGS_EXE, or HSPLOT_EXE).
        grok_filepath (Path): Path to the .grok file.
        parent: Parent widget.
        return_immediately (bool): Used when testing to immediately return.

    Returns:
        (int): 0 (QDialog.Rejected) on reject (abort), 1 (QDialog.Accepted) on OK.
    """
    dialog = RunProgressDialog(exe_key, grok_filepath, parent, return_immediately)
    return dialog.exec()


class RunProgressDialog(XmsDlg):
    """Class that shows a progress bar while an HGS executable runs."""

    TIMER_TIC = 500  # Timer will fire every TIMER_TIC milliseconds
    # There's a progress file...
    PROGRESS_FILE = 'newton_info.dat'
    PROGRESS_PERCENT_DONE_COLUMN = 8
    # ...and a file used for the plot
    PLOT_FILE = 'water_balance.dat'
    VARIABLES_KEYWORD = "VARIABLES="
    TIME_COLUMN = 0
    TIME_STRING = 'Time'
    UNITS_KEYWORD = ' units: ('
    PLOT_MAX_POINTS = 20  # Maximum number of points to plot

    def __init__(self, exe_key: str, grok_filepath: Path, parent=None, return_immediately: bool = False) -> None:
        """Initializes the class.

        Args:
            exe_key (str): Key used to identify the executable (GROK_EXE, PHGS_EXE, or HSPLOT_EXE).
            grok_filepath (Path): Path to the .grok file.
            parent: Parent widget.
            return_immediately (bool): Used when testing to immediately return.
        """
        super().__init__(parent, 'xms.hgs.gui.run_progress_dialog')
        self._exe_key = exe_key
        self._grok_filepath = grok_filepath
        self._return_immediately = return_immediately
        self._process = None
        self._paused: bool = False
        self._timer = None
        self._stdout_file = None  # stdout gets written to this file
        self._output_file = None  # The file we read and display
        self._progress_file = None
        self._progress_line = ''
        self._plot_file = None
        self._plot_line = ''
        self._variables: list[str] = []  # All the variables except for the first one, "Time"
        self._units: dict[str, str] = {}  # Variable -> units
        self._error_units = ''
        self._progress_data_lines_reached = False
        self._zone_line_reached = False
        self._plot_data_lines_reached = False
        self._figure: Figure | None = None  # matplotlib.figure Figure
        self._canvas: FigureCanvas | None = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self._ax: Axes | None = None  # matplotlib Axes
        self._last_variable = ''
        self._values = []
        self._start_time = None
        self._plot_title = ''

        self.ui = Ui_dlg_run_progress()
        self.ui.setupUi(self)

        self._help_getter = gui_util.help_getter('xms.hgs.gui.run_progress_dialog')
        self._init_window_title()
        self._init_file_and_exe_labels()
        self._init_plot()
        self._init_output_file_text()
        self._hide_or_show_y_axis_combo()
        self._init_progress_bar()
        self._init_buttons()

        self._delete_old_files()
        self._launch_exe()
        self._start_timer()

    def showEvent(self, event):  # noqa: N802 - function name should be lowercase
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        self._restore_splitter_geometry()

    def _splitter_setting_key(self) -> str:
        """Returns the key to the splitter setting."""
        return f'{self._dlg_name}.splitter.{self.windowTitle()}'

    def _save_splitter_geometry(self) -> None:
        """Save the current position of the splitter."""
        settings = SettingsManager()
        settings.save_setting('xms.hgs', self._splitter_setting_key(), self.ui.splitter.sizes())

    def _restore_splitter_geometry(self) -> None:
        """Restore the position of the splitter."""
        splitter_sizes = self._get_splitter_sizes()
        if not splitter_sizes:
            if self._exe_key != sim_runner.PHGS_EXE:
                self.ui.splitter.setSizes([10, 400])  # No plot. Make top part minimal
            else:
                self.ui.splitter.setSizes([300, 500])
            return
        splitter_sizes = [int(size) for size in splitter_sizes]
        self.ui.splitter.setSizes(splitter_sizes)

    def _get_splitter_sizes(self):
        """Returns a list of the splitter sizes that are saved in the registry."""
        settings = SettingsManager()
        splitter_sizes = settings.get_setting('xms.hgs', self._splitter_setting_key())
        return splitter_sizes

    def _get_exe_path(self) -> Path:
        """Initializes the labels that show which executable and which file are being used."""
        return sim_runner.get_gms_hgs_executable_paths()[self._exe_key]

    def _init_window_title(self) -> None:
        """Initializes the window title."""
        self.setWindowTitle(f'Run {self._get_exe_path().name}')

    def _init_file_and_exe_labels(self) -> None:
        """Initializes the labels that show which executable and which file are being used."""
        self.ui.lbl_file.setText(f'File: {str(self._grok_filepath)}')
        self.ui.lbl_executable.setText(f'Executable: {self._get_exe_path()}')

    def _init_plot(self) -> None:
        """Initializes the plot."""
        if self._exe_key != sim_runner.PHGS_EXE:
            return

        self._figure = Figure()
        self._figure.set_layout_engine(layout='tight')
        self._canvas = FigureCanvas(self._figure)
        self._canvas.setMinimumHeight(200)  # So user can't resize it to nothing
        self.ui.vlay_top.addWidget(self._canvas)
        self.adjustSize()
        self._ax = self._figure.add_subplot(111)

    def _init_output_file_text(self) -> None:
        """Initializes the output file text edit."""
        # self.ui.txt_file.setStyleSheet('QTextEdit { font-family: Courier New; font-size: 8pt}')
        self.ui.txt_file.setFont(QFont("Courier New", 8))

    def _hide_or_show_y_axis_combo(self) -> None:
        """Hides or shows the y axis combo."""
        if self._exe_key != sim_runner.PHGS_EXE:
            self.ui.lbl_y_axis.setVisible(False)
            self.ui.cbx_y_axis.setVisible(False)

    def _init_progress_bar(self) -> None:
        """Initializes the progress bar."""
        if self._exe_key != sim_runner.PHGS_EXE:
            self.ui.pbar_progress.setVisible(False)
            self.ui.lbl_y_axis.setVisible(False)
            self.ui.cbx_y_axis.setVisible(False)
            return

        self.ui.pbar_progress.setValue(0)

    def _init_buttons(self):
        """Initializes the buttons."""
        self.ui.btn_pause.clicked.connect(self._on_btn_pause)
        self.ui.btn_open_debug_control.setVisible(self._exe_key == sim_runner.PHGS_EXE)
        self.ui.btn_open_debug_control.clicked.connect(self._on_open_debug_control)
        self.ui.buttonBox.button(QDialogButtonBox.Ok).setText('Abort')
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def _start_timer(self) -> None:
        """Starts the timer."""
        self._timer = QTimer()
        self._timer.timeout.connect(self._on_timer)
        self._timer.start(RunProgressDialog.TIMER_TIC)

    def _on_timer(self) -> None:
        """Called when timer goes off to check if process is still running (and starts file watcher if necessary)."""
        self._update()
        if self._process and self._process.poll() is not None:
            self._on_process_finished()

    def _on_btn_pause(self) -> None:
        """Pauses/resumes the run."""
        if self._process is None:
            return

        p = psutil.Process(self._process.pid)
        if not self._paused:
            p.suspend()
        else:
            p.resume()
        self._paused = not self._paused
        self.ui.btn_pause.setText('Resume' if self._paused else 'Pause')

    def _on_open_debug_control(self) -> None:
        """Opens the debug.control file in the default app."""
        debug_control_filepath = self._grok_filepath.with_name('debug.control')
        if not debug_control_filepath.is_file():
            message = f'"{str(debug_control_filepath)}" does not exist.'
            message_box.message_with_ok(parent=self, message=message, app_name='GMS')
            return

        debug_control_dialog.run(debug_control_filepath, self)

    def _on_abort(self) -> None:
        """Kills the process, stops the timer, and updates the progress bar and buttons."""
        if self._process:
            self._process.kill()
            self._on_process_finished()

    def _on_process_finished(self) -> None:
        """Updates the progress bar and enables/disables the buttons when the process finishes or is killed."""
        if self._timer:
            self._timer.stop()
            self._update()
            self._update_progress_bar(100.0)
            self._append_elapsed_time()
        self.ui.buttonBox.button(QDialogButtonBox.Ok).setText('OK')

    def _delete_old_files(self) -> None:
        """Delete any old files that should be deleted before we run grok.exe."""
        if self._exe_key != sim_runner.GROK_EXE:
            return

        files = [
            self._grok_filepath.with_name('array_sizes.default'),
            self._grok_filepath.with_name('restart_file_info.dat'),
            self._grok_filepath.with_name('parallelindx.dat'),
            self._grok_filepath.with_name(f'{self._grok_filepath.stem}.output_variable.control')
        ]
        for f in files:
            f.unlink(missing_ok=True)

    def _launch_exe(self) -> None:
        """Starts the exe in a subprocess."""
        self._start_time = time.perf_counter()
        self._stdout_file = Path(filesystem.temp_filename()).open('w')
        if not self._get_exe_path().is_file():
            self.ui.txt_file.appendPlainText(f'{str(self._get_exe_path())} not found.')
            self._on_process_finished()
            return
        self._process = sim_runner.run_hgs_exe(
            self._exe_key, self._grok_filepath, console=False, stdout=self._stdout_file
        )

    def _progress_filepath(self) -> Path:
        """Returns the path to the file we read for progress (which may or may not exist).

        Returns:
            (Path): See description.
        """
        return self._grok_filepath.with_name(f'{self._grok_filepath.stem}o.{RunProgressDialog.PROGRESS_FILE}')

    def _plot_filepath(self) -> Path:
        """Returns the path to the file we read for progress (which may or may not exist).

        Returns:
            (Path): See description.
        """
        return self._grok_filepath.with_name(f'{self._grok_filepath.stem}o.{RunProgressDialog.PLOT_FILE}')

    def _get_new_output_file_text(self) -> str:
        """Reads the file and returns the text added since the last time we read.

        Returns:
            (str): New text.
        """
        new_text = ''
        while True:
            tmp = self._output_file.readline()
            if not tmp:
                break

            new_text += tmp
        return new_text

    def _update_output_file(self) -> None:
        """Updates the output file text edit."""
        output_filepath = Path(self._stdout_file.name)
        if not output_filepath.is_file():
            return  # pragma no cover - don't know how to make this happen

        if self._output_file is None:
            # Open the file
            try:
                self._output_file = open(output_filepath, 'r')
            except FileNotFoundError:  # pragma no cover - don't know how to make this happen
                return

        new_text = self._get_new_output_file_text()
        if not new_text:
            return

        self.ui.txt_file.appendPlainText(new_text)  # We use a QPlainTextEdit for speed

    def _get_last_progress_line(self) -> str:
        """Reads the file and returns the last progress line, if it has been written, or '' if it hasn't.

        Returns:
            (str): last line.
        """
        last_progress_line = ''  # Last full line of progress data in the file
        while True:
            tmp = self._progress_file.readline()
            if not tmp:
                break

            self._progress_line += tmp
            if self._progress_line.endswith('\n'):
                if self._progress_data_lines_reached:
                    last_progress_line = self._progress_line
                elif self._progress_line.startswith('#I=IMAX#####'):
                    self._progress_data_lines_reached = True

                self._progress_line = ''

        if self._progress_data_lines_reached and last_progress_line:
            return last_progress_line
        return ''

    def _get_last_plot_line(self) -> str:
        """Reads the file and returns the last plot line, if it has been written, or '' if it hasn't.

        Returns:
            (str): last line.
        """
        last_plot_line = ''  # Last full line of plot data in the file
        while True:
            tmp = self._plot_file.readline()
            if not tmp:
                break

            self._plot_line += tmp
            if self._plot_line.endswith('\n'):
                if self._plot_data_lines_reached:
                    last_plot_line = self._plot_line
                elif self._zone_line_reached:
                    # Next line after 'zone t=' is '#I=IMAX#####' in older versions, but not in newer
                    if self._plot_line.startswith('#I=IMAX#####'):
                        self._plot_data_lines_reached = True
                    else:
                        self._plot_data_lines_reached = True
                        last_plot_line = self._plot_line
                elif self._plot_line.lower().startswith('title'):
                    pos = self._plot_line.find('=')
                    if pos > -1:
                        self._plot_title = self._plot_line[pos + 1:].strip('').replace('"', '')
                elif self._plot_line.startswith('zone t='):
                    self._zone_line_reached = True
                elif self._plot_line.startswith(RunProgressDialog.VARIABLES_KEYWORD):
                    self._read_variables()
                elif RunProgressDialog.UNITS_KEYWORD in self._plot_line:
                    self._read_units()

                self._plot_line = ''

        if self._plot_data_lines_reached and last_plot_line:
            return last_plot_line
        return ''

    def _read_units(self) -> None:
        """Reads and stores the units for the variable."""
        variable = self._plot_line[2:self._plot_line.find(RunProgressDialog.UNITS_KEYWORD)].strip()
        self._units[variable] = self._plot_line[self._plot_line.find(':'):].strip()

    def _read_variables(self) -> None:
        """Reads and stores the variable names."""
        line = self._plot_line[self._plot_line.find('='):].strip()
        self._variables = [v.replace('"', '') for v in re.split(r',(?=")', line)]  # split and remove quotes
        self._variables = self._variables[1:]  # Don't include the first variable, "Time"

    def _read_progress(self) -> float:
        """Reads the file used for progress and returns the percent done.

        Returns:
            (float): percent done.
        """
        progress_filepath = self._progress_filepath()
        if not progress_filepath.is_file():
            return -1

        if self._progress_file is None:
            # Open the file
            try:
                self._progress_file = open(progress_filepath, 'r')
            except FileNotFoundError:  # pragma no cover - don't know how to make this happen
                return -1

        line = self._get_last_progress_line()
        if not line:
            return -1

        words = line.split()
        percent_done = float(words[RunProgressDialog.PROGRESS_PERCENT_DONE_COLUMN])
        return percent_done

    def _read_plot_data(self) -> list[float]:
        """Reads the file used for the plot returns the new plot data.

        Returns:
            (tuple[float, float]): time, error.
        """
        plot_filepath = self._plot_filepath()
        if not plot_filepath.is_file():
            return []

        if self._plot_file is None:
            # Open the file
            try:
                self._plot_file = open(plot_filepath, 'r')
            except FileNotFoundError:  # pragma no cover - don't know how to make this happen
                return []

        line = self._get_last_plot_line()
        if not line:
            return []

        words = line.split()
        values = [float(word) for word in words]
        return values

    def _init_y_axis_combo_box(self) -> int:
        """Initializes the y-axis combo box and returns the current index."""
        if self._variables:
            self.ui.cbx_y_axis.addItems(self._variables)
            default_index = self._variables.index('Error percent')
            if default_index >= 0:
                self.ui.cbx_y_axis.setCurrentIndex(default_index)
            else:
                self.ui.cbx_y_axis.setCurrentIndex(0)
        return self.ui.cbx_y_axis.currentIndex()

    def _get_y_axis_variable(self) -> tuple[str, int]:
        """Returns the name and index of the y axis variable being plotted, or '', -1 if data not available yet."""
        variable_index = self.ui.cbx_y_axis.currentIndex()
        if variable_index == -1:
            variable_index = self._init_y_axis_combo_box()
        if variable_index == -1:
            return '', -1
        return self.ui.cbx_y_axis.currentText(), self.ui.cbx_y_axis.currentIndex()

    def _update_plot(self, values: list[float]) -> None:
        """Updates the plot with the new data.

        Args:
            values: The values, where the first value is Time.
        """
        # Save the values but only keep the last n values
        self._values.append(values)
        if len(self._values) > RunProgressDialog.PLOT_MAX_POINTS:
            self._values = self._values[-RunProgressDialog.PLOT_MAX_POINTS:]

        # Get variable being plotted
        variable, variable_index = self._get_y_axis_variable()
        if not variable or variable_index < 0:
            return

        # If variable being plotted changed, reset the plot
        if variable != self._last_variable:
            self._last_variable = variable
            self._ax.clear()

        # Get last n times and variable values
        times = [values[RunProgressDialog.TIME_COLUMN] for values in self._values]
        variable_values = [values[variable_index] for values in self._values]

        # Draw the plot
        self._ax.plot(times, variable_values)
        self._ax.grid(True)
        self._ax.set_title(self._plot_title)
        self._ax.set_xlabel(f'{RunProgressDialog.TIME_STRING}  {self._units["Time"]}')
        self._ax.set_ylabel(f'{variable} {self._units[variable]}')
        self._ax.relim()  # With next line, frames the lines in the plot
        self._ax.autoscale()  # With previous line, frames the lines in the plot
        self._canvas.draw()

    def _update_progress_bar(self, percent_done: float) -> None:
        """Updates the progress bar.

        Args:
            percent_done (float): The percent complete, from 0.0 to 100.0.
        """
        self.ui.pbar_progress.setValue(int(percent_done))

    def _update(self):
        """Reads the file used for progress and updates the progress bar."""
        self._update_output_file()
        if self._exe_key != sim_runner.PHGS_EXE:
            return

        values = self._read_plot_data()
        percent_done = self._read_progress()
        if not values or values[0] < 0.0:  # Time is negative
            return

        self._update_plot(values)
        self._update_progress_bar(percent_done)

    def _append_elapsed_time(self) -> None:
        """Appends the elapsed time to the text field."""
        end_time = time.perf_counter()
        self.ui.txt_file.appendPlainText(f'Elapsed time: {end_time - self._start_time:0.4f} seconds.')

    def accept(self):
        """Accepted."""
        if self.ui.buttonBox.button(QDialogButtonBox.Ok).text() == 'Abort':
            self.reject()
        else:
            self._save_splitter_geometry()
            super().accept()

    def reject(self):
        """Cancelled."""
        self._on_abort()
        self._save_splitter_geometry()
        super().reject()

    def exec(self):  # pragma no cover - don't test this
        """If testing, just accept immediately."""
        # Can't check for XmsEnvironment.xms_environ_running_tests() == 'TRUE' like we normally do because that is
        # true for most of our tests so that we can get the executable w/o GMS
        if self._return_immediately:
            self.accept()
            return QDialog.Accepted
        else:
            return super().exec_()
