"""HydrographMapAtt class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Group

# 4. Local modules
from xms.hgs.mapping import set_maker
from xms.hgs.mapping.map_att import MapAtt


class HydrographMapAtt(MapAtt):
    """A coverage attribute item (i.e. BC, observation,  hydrograph) with attributes and intersected grid components."""
    def __init__(self, feature, att_type: str, group: Group) -> None:
        """Initializes the class.

        Args:
            feature: A point, arc, or polygon.
            att_type (str): Attribute type (e.g. 'Flux nodal')
            group (Group): The BC as a generic model group.
        """
        super().__init__(feature, att_type, group)

    def write(self, section, set_id_strings: set[str] | None = None, *args, **kwargs) -> None:
        """Writes this MapAtt object to grok.

        Args:
            section (Type[SectionBase]): The section we are writing to.
            set_id_strings (set[str] | None): Set of grid component set id strings so we can ensure they are unique.
        """
        grok_filepath = Path(section.file_data.file.name)
        _, _ = set_maker.make_set(grok_filepath, self, set_id_strings, section)
        section.write_value('set hydrograph nodes', self.value('name'))

    def get_grid_component_type(self):
        """Returns the type of grid component ('point', 'face', 'segment') from the att type."""
        return 'node'  # There's only one type 'Hydrograph' and it's a point set
