"""Data class for simulation component."""

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.data_bases.sim_base_data import SimBaseData

# 4. Local modules


class SimData(SimBaseData):
    """
    Manages data file for a simulation component.

    `SimData` derives from `ComponentData` to allow storing and retrieving global and model values
    """
    def __init__(self, data_file: str | Path):
        """
        Initialize the data class.

        Args:
            data_file: The netcdf file (with path) associated with this instance data. Probably the owning
                       component's main file. If not provided, a default one will be chosen.
        """
        super().__init__(data_file)

        self.info.attrs['FILE_TYPE'] = 'HYDROAS_SIM_DATA'

    @property
    def generic_model(self) -> GenericModel:
        """
        The generic model.

        - This only gets the model's template. Values are not included.
        - This is mainly useful for models with multiple or dynamic templates. Derived models should typically just have
          a function that makes their model template and always use that to get it.
        """
        definitions = self.info.attrs.get('GENERIC_MODEL_TEMPLATE', None)
        generic_model = GenericModel(definitions=definitions)
        return generic_model

    @generic_model.setter
    def generic_model(self, generic_model: GenericModel):
        """
        The generic model.

        This only sets the model's template. Values are not included.
        """
        self.info.attrs['GENERIC_MODEL_TEMPLATE'] = generic_model.to_template().to_string()

    @property
    def model_values(self) -> str:
        """Values for model parameters."""
        if 'MODEL_VALUES' in self.info.attrs:
            return self.info.attrs['MODEL_VALUES']
        else:  # Let GenericModel define the default ones so it can change them later if necessary.
            return GenericModel().model_parameters.extract_values()

    @model_values.setter
    def model_values(self, value: str):
        """Values for model parameters."""
        self.info.attrs['MODEL_VALUES'] = value
