"""AdvancedTransportDataBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class AdvancedTransportDataBase(ListPackageData):
    """Base data class for LKT, MWT, SFT, and UZT packages."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.list_blocks = {'PACKAGEDATA': ''}  # (list blocks -> filename)

    @override
    def get_time_series_columns(self) -> list[int]:
        """Returns a list of the column indices that can contain time series.

        Returns:
            List of indices of columns that can contain time series.
        """
        return data_util.advanced_transport_time_series_columns(self)

    def block_with_aux(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    def block_with_boundnames(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    def plottable_columns(self):
        """Returns a set of columns (0-based) that can be plotted with the XySeriesEditor.

        Returns:
            See description.
        """
        return {1, 2}  # lktsetting columns

    # @overrides
    def read_csv_file_into_dataframe(self, block, filename, column_names, column_types):
        """Reads a csv file and returns a dataframe.

        Overridden to handle CONNECTIONDATA which is challenging. The IC numbers are turned into a space delimited list
        in one IC column.

        Args:
            block (str): Name of the block.
            filename (str): Filepath.
            column_names (list): Column names.
            types (dict of str -> type): Column names -> column types.

        Returns:
            (pandas.DataFrame)
        """
        return gui_util.read_csv_file_into_dataframe(filename, column_names, column_types)

    # # @overrides
    # def dataframe_to_temp_file(self, block, dataframe):
    #     """Writes the dataframe to a csv file.
    #
    #     Overridden because CONNECTIONDATA is more complicated due to IC column being a space separated list.
    #
    #     Args:
    #         block (str): Name of the block.
    #         dataframe (pandas.DataFrame: The dataframe
    #
    #     Returns:
    #         (str): Filepath of file created.
    #     """
    #     return gui_util.dataframe_to_temp_file(dataframe)

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type == 'arcs':
    #         return {'FLOW': None, 'RUNOFF': None, 'ETSW': None, 'PPTSW': None, 'ROUGHCH': None, 'ROUGHBK': None,
    #                 'CDPTH': None, 'FDPTH': None, 'AWDTH': None, 'BWDTH': None, 'HCOND1': None, 'THICKM1': None,
    #                 'ELEVUP': None, 'WIDTH1': None, 'DEPTH1': None, 'HCOND2': None, 'THICKM2': None,
    #                 'ELEVDN': None, 'WIDTH2': None, 'DEPTH2': None, }
    #     else:
    #         return {}

    def _output_variable(self) -> str:
        """Returns the name of this package's output variable.

        Returns:
            See description
        """
        if self.ftype in ['LKE6', 'MWE6', 'SFE6', 'UZE6']:
            return 'TEMPERATURE'
        return 'CONCENTRATION'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        variable = self._output_variable()
        return OptionsBlock(
            [
                CheckboxField('FLOW_PACKAGE_NAME', brief='Name of corresponding flow package', type_='str'),
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                CheckboxField(
                    'FLOW_PACKAGE_AUXILIARY_NAME',
                    brief='Name of auxiliary variable in corresponding flow package',
                    type_='str'
                ),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox(f'PRINT_{variable}', brief=f'Print {variable.lower()} to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxField(f'{variable} FILEOUT', brief=f'Save {variable.lower()} to file', type_='str'),
                CheckboxField('BUDGET FILEOUT', brief='Save budget to file', type_='str'),
                CheckboxField('BUDGETCSV FILEOUT', brief='Save budget to CSV file', type_='str'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
            ]
        )
