"""MdtData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField
from xms.mf6.mapping.map_util import Mapping


class MdtData(GriddataBase):
    """Data class to hold the info from a MDT package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'MDT6'
        self.add_block(
            'GRIDDATA', [
                'MD_TYPE_FLAG', 'MD_FRACTION', 'MD_POROSITY', 'BULK_DENSITY', 'MD_DIFF_LENGTH', 'MD_TORTUOSITY',
                'MD_DIST_COEFF', 'MD_DECAY', 'MD_DIFF_COEFF', 'AIOLD1MD', 'AIOLD2MD'
            ]
        )

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'MD_TYPE_FLAG':
                units_spec = units_util.UNITS_UNITLESS
            case 'MD_FRACTION':
                units_spec = '[?]'
            case 'MD_POROSITY':
                units_spec = units_util.UNITS_UNITLESS
            case 'BULK_DENSITY':
                units_spec = units_util.UNITS_BULK_D
            case 'MD_DIFF_LENGTH':
                units_spec = '[?]'
            case 'MD_TORTUOSITY':
                units_spec = units_util.UNITS_UNITLESS
            case 'MD_DIST_COEFF':
                units_spec = units_util.UNITS_DIST_C
            case 'MD_DECAY':
                d = self.options_block.get('FIRST_ORDER_DECAY')
                if d:
                    units_spec = '[1/T]?'
                else:
                    units_spec = '[M/L^3/T]?'
            case 'MD_DIFF_COEFF':
                units_spec = '[?]'
            case 'AIOLD1MD':
                units_spec = '[?]'
            case 'AIOLD2MD':
                units_spec = '[?]'
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name == 'MD_TYPE_FLAG'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return True  # Make them all required for now

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Matrix Diffusion Transport (MDT) Package'

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {
            'MD_TYPE': Mapping(mf_name='MD_TYPE_FLAG'),
            'MD_FRAC': Mapping(mf_name='MD_FRACTION'),
            'MD_PORO': Mapping(mf_name='MD_POROSITY'),
            'BULK_DN': Mapping(mf_name='BULK_DENSITY'),
            'MD_DIFL': Mapping(mf_name='MD_DIFF_LENGTH'),
            'MD_TORT': Mapping(mf_name='MD_TORTUOSITY'),
            'MD_DIST': Mapping(mf_name='MD_DIST_COEFF'),
            'MD_DECA': Mapping(mf_name='MD_DECAY'),
            'MD_DIFC': Mapping(mf_name='MD_DIFF_COEFF'),
            'AIOLD1MD': Mapping(mf_name='AIOLD1MD'),
            'AIOLD2MD': Mapping(mf_name='AIOLD2MD')
        }

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        del feature_type  # Unused parameter
        return {}

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SORPTION', brief='Sorption will be activated. Requires BULK_DENSITY and MD_DIST_COEFF.'),
                Checkbox('FIRST_ORDER_DECAY', brief='First-order decay will occur'),
                CheckboxField('TSHIFTMD', brief='Time elapsed since matrix diffusion began', type_='float'),
                CheckboxField(
                    'MD FILEOUT', brief='Write AIOLD1MD and AIOLD2MD coefficients to a binary file', type_='str'
                ),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'MD_TYPE_FLAG':
            txt = (
                'Matrix diffusion type flag at any location. This flag gives options at each cell on how the matrix'
                ' diffusion is treated. One value is read for every model cell. Usually, these values are read one'
                ' layer at a time; however, when the XSECTION option is specified, a single two-dimensional variable'
                ' for the cross section is read. If MD_TYPE_FLAG(N) = 0, cell N is inactive in the matrix domain'
                ' (either the cell itself is inactive or the matrix domain does not exist at this cell). If'
                ' MD_TYPE_FLAG (N) = 1, cell N is active in the matrix domain with matrix diffusion occurring at the'
                ' bottom of the cell. If MD_TYPE_FLAG (N) = 2, cell N is active in the matrix domain with matrix'
                ' diffusion occurring embedded within the cell. If MD_TYPE_FLAG (N) = 3, cell N is active in the'
                ' matrix domain with matrix diffusion occurring at the top of the cell. If MD_TYPE_FLAG (N) = 4, cell'
                ' N is active in the matrix domain with matrix diffusion occurring at both the top and bottom of the'
                ' cell. If MD_TYPE_FLAG (N) = 5, cell N is active in the matrix domain with matrix diffusion'
                ' occurring at the bottom and embedded within the cell. If MD_TYPE_FLAG (N) = 6, cell N is active in'
                ' the matrix domain with matrix diffusion occurring at the top and embedded within the cell. If'
                ' MD_TYPE_FLAG (N) = 7, cell N is active in the matrix domain with matrix diffusion occurring at the'
                ' top, bottom and embedded within the cell. Note that MD_TYPE_FLAG can be inactive for active flow'
                ' nodes. However, MD_TYPE_FLAG cannot be active for a node which is inactive for the corresponding flow'
                ' and transport simulation.'
            )
        elif tab_text == 'MD_FRACTION':
            txt = (
                'Immobile fraction at any location, i.e., the fraction of the total space that is occupied by the'
                ' matrix/low permeability domain. Note that in the USG-Transport version of MDT, the mobile fraction is'
                ' input. Input of immobile fraction in MODFLOW 6 version of MDT is required because MODFLOW 6 allows'
                ' for more than one MDT domain and each MDT domain will have its own VOLFRACMD.'
            )
        elif tab_text == 'MD_POROSITY':
            txt = 'Effective transport porosity within the matrix domain at any location.'
        elif tab_text == 'BULK_DENSITY':
            txt = 'Dry bulk density of the porous matrix of the matrix domain at any location.'
        elif tab_text == 'MD_DIFF_LENGTH':
            txt = (
                'Diffusion length for diffusive transport within the matrix domain at any location (only used for'
                ' embedded heterogeneity).'
            )
        elif tab_text == 'MD_TORTUOSITY':
            txt = 'Tortuosity of the matrix domain at any location.'
        elif tab_text == 'MD_DIST_COEFF':
            txt = 'Adsorption coefficient (Kd) of a contaminant species in the immobile domain at any location.'
        elif tab_text == 'MD_DECAY':
            txt = 'First-order decay coefficient in water and on soil (1/time) in the matrix domain at any location.'
        elif tab_text == 'MD_DIFF_COEFF':
            txt = 'Diffusion coefficient of each species component within the matrix domain at any location.'
        elif tab_text == 'AIOLD1MD':
            txt = (
                'Concentration integral (equation 8) of each species in an element associated with matrix diffusion'
                ' into an infinite acting adjacent aquitard (MD_TYPE_FLAG = 1, 3, 4, 5, 6, or 7). This value is used to'
                ' restart a simulation with prior matrix diffusion.'
            )
        elif tab_text == 'AIOLD2MD':
            txt = (
                'Concentration integral (equation 8) of each species in an element associated with matrix diffusion in'
                ' low permeability zones that are embedded in the volume element (MD_TYPE_FLAG = 2, 5, 6, or 7). This'
                ' value is used to restart a simulation with prior matrix diffusion.'
            )
        return txt
