"""ArrayLayerReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from distutils.text_file import TextFile
import os
import shlex
import sys
import typing

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.file_io import io_util


class ArrayLayerReader:
    """Reads an ArrayLayer."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()

        self.filename = ''
        self._name = ''
        self._in_file = None  # File we're reading from
        self._nvalues = -1  # Number of array values.

    def _get_factor(self, words) -> float:
        """Returns the factor specified on the line, or 1.0 if none found.

        Args:
            words (list[str]): List of words from the line after splitting.

        Returns:
            See description.
        """
        for i in range(1, len(words)):
            if words[i].upper() == 'FACTOR' and i + 1 < len(words):
                return float(words[i + 1])
        return 1.0

    def _is_binary(self, words: list[str]) -> bool:
        """Returns True if the '(BINARY)' keyword is found.

        Args:
            words (list[str]): List of words from the line after splitting.

        Returns:
            See description.
        """
        for word in words:
            if word.upper() == '(BINARY)':
                return True
        return False

    def _read_array_values(self, file):
        """Reads through the lines/values of the array.

        Args:
            file (_io.TextIOWrapper): The file object.

        Returns:
            (tuple): tuple containing:
                - all_words (list of str): List of all array words (empty if not externalizing).
                - extra_words_after_array (str): The end of the line after last array word. Typically blank.
        """
        return io_util.read_array_values(file, self._nvalues)

    def _read_internal_array(self, array_layer, factor):
        """Rewrites the array_layer so it that is in an external file.

        Args:
            array_layer (ArrayLayer): The Array object.
            factor (float): Multiplication factor of the array_layer.

        Returns:
            (str): The last line read if an internal array, or an empty string.
        """
        array_layer.storage = 'ARRAY'
        array_layer.factor = 1.0 if not factor else float(factor)

        words, line = self._read_array_values(self._in_file)
        array_layer.values = words

        return line

    def read(self, filename, nvalues, in_file: typing.TextIO | TextFile, importing) -> tuple[ArrayLayer, str]:
        """Reads the array and stores data in array.

         Rewrites it if out_file and output_dir are not None.

        Args:
            filename (str): File path of package file being read.
            nvalues (int): Total number of values in the array to read.
            in_file: The file being read.
            importing (bool): Flag to tell if we need to read external file data

        Returns:
            (tuple): tuple containing:
                - array_layer (Array): The array_layer
                - line (str): The last line read if an internal array, or an empty string.
        """
        self.filename = filename
        self._in_file = in_file
        self._nvalues = nvalues

        array_layer = ArrayLayer(name=self._name)

        last_line = ''
        # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
        # for line in self._in_file:
        while True:
            line = self._in_file.readline()
            if not line:
                break

            words = shlex.split(line, posix="win" not in sys.platform)
            if words:
                if words[0].upper() == 'INTERNAL':
                    factor = self._get_factor(words)
                    last_line = self._read_internal_array(array_layer, factor)
                    break
                elif words[0].upper() == 'OPEN/CLOSE':
                    words[1] = words[1].strip("'\"")
                    full_filename = fs.resolve_relative_path(os.path.dirname(self.filename), words[1])

                    array_layer.factor = self._get_factor(words)
                    array_layer.binary = self._is_binary(words)
                    array_layer.storage = 'ARRAY'
                    array_layer.external_filename = full_filename
                    if importing and not array_layer.binary:
                        with open(full_filename, "r") as external_array_file:
                            array_layer.values, _ = self._read_array_values(external_array_file)
                    last_line = ''
                    break
                else:  # 'constant'
                    array_layer.storage = 'CONSTANT'
                    array_layer.constant = float(words[1])

                    last_line = ''
                    break
        return array_layer, last_line
