"""DependentVariableFileReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import collections
import math
import os
from pathlib import Path
import struct
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs
from xms.datasets.dataset_writer import DatasetWriter
from xms.guipy.dialogs.feedback_thread import ExpectedError
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data import time_util
from xms.mf6.data.grid_info import DisEnum


def read(
    dis_enum: DisEnum, filename: Pathlike, dataset_name: str, ugrid_uuid: str, dset_time_units: str,
    start_date_time: str
):
    """Reads the dependent variable file and creates an XMDF dataset file.

    Args:
        dis_enum: DisEnum
        filename: File path of dependent variable file.
        dataset_name: Name that will appear in XMS.
        ugrid_uuid: UUID of the UGrid.
        dset_time_units: Units of time.
        start_date_time: Starting date/time.

    Returns:
        tuple[DatasetWriter, list[double]]: The dataset writer and the list of timestep times.
    """
    reader = DependentVariableFileReader(dis_enum)
    return reader.read(filename, dataset_name, ugrid_uuid, dset_time_units, start_date_time)


class DependentVariableFileReader:
    """Reads the Dependent Variable File (head file)."""
    def __init__(self, dis_enum, dset_uuid=None):
        """Initializes the class.

        Args:
            dis_enum (DisEnum): Tells what type of DIS/DISV/DISU we're dealing with.
            dset_uuid (str): uuid of the dataset
        """
        self._dis_enum = dis_enum

        # Determine the format of the header line depending on the DIS*
        if dis_enum == DisEnum.DIS:
            self._header_tuple = collections.namedtuple('HeaderTuple', 'kstp kper pertim totim text ncol nrow ilay')
        elif dis_enum == DisEnum.DISV:
            self._header_tuple = collections.namedtuple('HeaderTuple', 'kstp kper pertim totim text ncpl one ilay')
        elif dis_enum == DisEnum.DISU:
            self._header_tuple = collections.namedtuple('HeaderTuple', 'kstp kper pertim totim text nodes one one_')
        else:
            raise RuntimeError('Unknown dis_enum')

        self._header_fmt = '<iidd16siii'
        self.header_size = struct.calcsize(self._header_fmt)
        self.file_content = None
        self.byte = 0
        self.kstp = 0
        self.kper = 0
        self.first = True
        self.data_bytes_per_layer = 0
        self.ncpl = 0  # number of cells per layer
        self.ts_data = []
        self.totim = 0.0
        self.nbytes = 0  # Number of bytes in the file
        self.dset_uuid = dset_uuid if dset_uuid else str(uuid.uuid4())
        self._times = []

    def read(self, filename: Pathlike, dataset_name: str, ugrid_uuid: str, dset_time_units: str, start_date_time: str):
        """Reads the dependent variable file and creates an XMDF dataset file.

        Args:
            filename: File path of dependent variable file.
            dataset_name: Name that will appear in XMS.
            ugrid_uuid: UUID of the UGrid.
            dset_time_units: Units of time.
            start_date_time: Starting date/time.

        Returns:
            tuple[DatasetWriter, list[double]]: The dataset writer and the list of timestep times.
        """
        if os.path.getsize(filename) == 0:
            raise ExpectedError(f'File is empty: {filename}')

        h5_filename = _h5_filename(filename)

        # Remove the old file if it exists
        if os.path.isfile(h5_filename):
            fs.removefile(h5_filename)

        datetime = time_util.datetime_from_arbitrary_string(start_date_time)
        writer = DatasetWriter(
            h5_filename=h5_filename,
            name=dataset_name,
            geom_uuid=ugrid_uuid,
            time_units=dset_time_units,
            ref_time=datetime,
            location='cells',
            use_activity_as_null=True,
            dset_uuid=self.dset_uuid
        )
        with open(filename, mode='rb') as file:
            self.file_content = file.read()
            self.nbytes = len(self.file_content)

            header = self._read_header()
            while header:
                if self._timestep_done(header):
                    self._finalize_timestep(header, writer)

                data = self._read_data()
                self._add_data_to_timestep(data, header)
                header = self._read_header()

        self._finalize_timestep(header, writer)
        writer.active_timestep = 0
        writer.appending_finished()

        return writer, self._times

    def _read_header(self):
        """Reads and returns the header (record 1), or none if we're at the end.

        Returns:
            HeaderTuple
        """
        if self.byte >= self.nbytes:
            return None

        # Read header
        header = self._header_tuple._make(
            struct.unpack(self._header_fmt, self.file_content[self.byte:self.byte + self.header_size])
        )
        self.byte += self.header_size

        # If it's the first time, save kstp, kper so we know when we're on a new timestep. Also get ncpl which
        # should never change.
        if self.first:
            self.first = False
            self.kstp = header.kstp
            self.kper = header.kper
            if self._dis_enum == DisEnum.DIS:
                self.ncpl = header.ncol * header.nrow
            elif self._dis_enum == DisEnum.DISV:
                self.ncpl = header.ncpl
            else:
                self.ncpl = header.nodes
            self.data_bytes_per_layer = self.ncpl * 8  # 8 = size of double
        return header

    def _timestep_done(self, header):
        """Returns true if the header indicates we have moved on to the next timestep.

        Args:
            header (HeaderTuple): The header

        Returns:
            See description.
        """
        return header.kper != self.kper or header.kstp != self.kstp

    def _finalize_timestep(self, header, writer):
        """Adds the data read for this timestep to the h5 dataset.

        Args:
            header (HeaderTuple): The header.
            writer (DatasetWriter): The dataset writer.
        """
        values = np.asarray(self.ts_data, dtype='f')
        no_data = np.asarray([1e30], dtype='f')
        activity = np.asarray([int(not math.isclose(abs(v), no_data[0])) for v in values], dtype='i')
        writer.append_timestep(time=self.totim, data=values, activity=activity)
        self._times.append(self.totim)
        if header:
            self.kper = header.kper
            self.kstp = header.kstp
            self.ts_data = []

    def _add_data_to_timestep(self, data, header):
        """Adds the data to the timestep data read so far.

        Args:
            data: Data from record 2.
            header (HeaderTuple): The header
        """
        self.ts_data.extend(data)
        self.totim = header.totim

    def _read_data(self):
        """Reads and returns the data in record 2.

        Returns:
            See description.
        """
        fmt = f'<{"d" * self.ncpl}'
        data = struct.unpack(fmt, self.file_content[self.byte:self.byte + self.data_bytes_per_layer])
        self.byte += self.data_bytes_per_layer
        return data


def _h5_filename(filename: Path | str) -> str:
    """Returns the h5 filename based on the filename.

    Args:
        filename (str): File path to dependent variable file.

    Returns:
        See description.
    """
    h5_filename = str(filename) + '.h5'
    return h5_filename
