"""ImsWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.package_writer_base import PackageWriterBase
from xms.mf6.misc.settings import Settings


class ImsWriter(PackageWriterBase):
    """Writes a ims package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._complexity_option = 0  # 0 = NOT SPECIFIED, 1 = SIMPLE, 2 = MODERATE, 3 = COMPLEX
        self._presets = None
        self._file_str = ''
        self._nonlinear_written_to_file = False

    def _write_options(self, fp):
        """Writes the options block.

        Args:
            fp (_io.TextIOWrapper): The file.
        """
        fp.write('BEGIN OPTIONS\n')
        print_option = self._data.options_block.get('PRINT_OPTION')
        complexity = self._data.options_block.get('COMPLEXITY')
        csv_outer_output = self._data.options_block.get('CSV_OUTER_OUTPUT FILEOUT')
        csv_inner_output = self._data.options_block.get('CSV_INNER_OUTPUT FILEOUT')
        no_ptc = self._data.options_block.get('NO_PTC')

        if print_option and print_option.upper() != 'NONE':
            io_util.write_option(self._data.options_block.dict(), 'PRINT_OPTION', fp)
        if complexity and complexity.upper() != 'NOT SPECIFIED':
            io_util.write_option(self._data.options_block.dict(), 'COMPLEXITY', fp)
            self._complexity_option = ['NOT SPECIFIED', 'SIMPLE', 'MODERATE', 'COMPLEX'].index(complexity.upper())
        if csv_outer_output and len(csv_outer_output) > 0 and csv_outer_output.upper() != 'NONE':
            super()._write_option('CSV_OUTER_OUTPUT FILEOUT', fp, self._data.options_block.dict())
        if csv_inner_output and len(csv_inner_output) > 0 and csv_inner_output.upper() != 'NONE':
            super()._write_option('CSV_INNER_OUTPUT FILEOUT', fp, self._data.options_block.dict())
        if no_ptc:
            io_util.write_option(self._data.options_block.dict(), 'NO_PTC', fp)
        fp.write('END OPTIONS\n')

    def _write_variable_if_in_use(self, var_name):
        """Writes the class variable if the value is not equal to the default value.

        Args:
            var_name (str): class variable name
        """
        use = getattr(self._data, f'{var_name.lower()}_use')
        if use:
            val = getattr(self._data, var_name.lower())
            self._file_str += f'{io_util.mftab}{var_name} {val}\n'

    def _write_nonlinear(self, fp):
        """Writes the nonlinear block.

        Args:
            fp (_io.TextIOWrapper): The file.
        """
        self._file_str = ''
        self._write_variable_if_in_use('OUTER_DVCLOSE')
        self._write_variable_if_in_use('OUTER_MAXIMUM')
        self._write_variable_if_in_use('UNDER_RELAXATION')
        self._write_variable_if_in_use('UNDER_RELAXATION_THETA')
        self._write_variable_if_in_use('UNDER_RELAXATION_KAPPA')
        self._write_variable_if_in_use('UNDER_RELAXATION_MOMENTUM')
        self._write_variable_if_in_use('UNDER_RELAXATION_GAMMA')
        self._write_variable_if_in_use('BACKTRACKING_NUMBER')
        self._write_variable_if_in_use('BACKTRACKING_TOLERANCE')
        self._write_variable_if_in_use('BACKTRACKING_REDUCTION_FACTOR')
        self._write_variable_if_in_use('BACKTRACKING_RESIDUAL_LIMIT')
        self._write_variable_if_in_use('LINEAR_SOLVER')  # SAMG extension to MODFLOW 6

        if self._file_str:
            fp.write('\n')
            fp.write('BEGIN NONLINEAR\n')
            fp.write(self._file_str)
            fp.write('END NONLINEAR\n')
            self._nonlinear_written_to_file = True

    def _write_rclose(self):
        """Writes the INNER_RCLOSE and RCLOSE_OPTION variables."""
        if self._data.inner_rclose_use:
            self._file_str += f'{io_util.mftab}INNER_RCLOSE {self._data.inner_rclose}'
            if self._data.rclose_option_use:
                self._file_str += f' {self._data.rclose_option}'
            self._file_str += '\n'

    def _write_linear(self, fp):
        """Writes the linear block.

        Args:
            fp (_io.TextIOWrapper): The file.
        """
        self._file_str = ''
        self._write_variable_if_in_use('INNER_MAXIMUM')
        self._write_variable_if_in_use('INNER_DVCLOSE')
        self._write_rclose()
        self._write_variable_if_in_use('LINEAR_ACCELERATION')
        self._write_variable_if_in_use('RELAXATION_FACTOR')
        self._write_variable_if_in_use('PRECONDITIONER_LEVELS')
        self._write_variable_if_in_use('PRECONDITIONER_DROP_TOLERANCE')
        self._write_variable_if_in_use('NUMBER_ORTHOGONALIZATIONS')
        self._write_variable_if_in_use('SCALING_METHOD')
        self._write_variable_if_in_use('REORDERING_METHOD')

        if self._file_str:
            if not self._nonlinear_written_to_file:
                fp.write('\n')
                fp.write('BEGIN NONLINEAR\n')
                fp.write('END NONLINEAR\n')
            fp.write('\n')
            fp.write('BEGIN LINEAR\n')
            fp.write(self._file_str)
            fp.write('END LINEAR\n')

    def write_settings(self, data):
        """If writing to components area, save fname to a settings file.

        Args:
            data: The package data.
        """
        super().write_settings(data)
        if self._writer_options.dmi_sim_dir:
            settings = Settings.read_settings(data.filename)
            settings['slnmnames'] = data.slnmnames
            Settings.write_settings(data.filename, settings)

    def _write_package(self, data):
        """Writes the package file.

        Args:
            data (xms.mf6.ims_package_data.ImsData): The data.
        """
        self._data = data
        self._presets = data.get_complexity_variable_defaults()
        with open(self._data.filename, 'w') as fp:
            self._write_comments(fp)
            self._write_options(fp)
            self._write_nonlinear(fp)
            self._write_linear(fp)
