"""SfrReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.package_reader import PackageReader


class SfrReader(PackageReader):
    """Reads an SFR package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__(ftype='SFR6')
        self._block_methods = {
            'PACKAGEDATA': '_handle_list_block',
            'CONNECTIONDATA': '_handle_list_block',
            'DIVERSIONS': '_handle_list_block'
        }

    # @overrides
    def join_line(self, words, column_count, separator=io_util.mfsep):
        """Joins the words into a line and returns the line.

        Can be overridden.

        Args:
            words (list[str]): The words.
            column_count (int): Number of columns that should be written for this line.

        Returns:
            (str): The line.
        """
        if self._curr_block_name.upper() != 'PACKAGEDATA':
            return super().join_line(words, column_count, separator)

        # PACKAGEDATA block
        if words[1].upper() != 'NONE':
            cellid_len = self._data.grid_info().cell_id_column_count()
            new_words = []
            new_words.append(words[0])
            new_words.append(io_util.mfsep.join(words[1:1 + cellid_len]))
            new_words.extend(words[1 + cellid_len:])
            return super().join_line(new_words, column_count, separator=',')
        else:
            return super().join_line(words, column_count, separator=',')

    def _read_options(self, line):
        """Reads a line from the options block.

        Args:
            line (str): A line from the file.

        Returns:
            (tuple(str, str)): tuple containing:
            - key: str
            - value: str or may be list(str) or may be something else
        """
        key, value = super()._read_options(line)
        if key == 'UNIT_CONVERSION':
            # At 6.4.2, LENGTH_CONVERSION and TIME_CONVERSION replaced UNIT_CONVERSION
            del self._data.options_block.options['UNIT_CONVERSION']
            self._data.options_block.set('LENGTH_CONVERSION', True, value)
        return key, value
