"""MfsimReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import TreeNode
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.mfsim_data import MfsimData, SolutionGroup
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.package_reader import PackageReader
from xms.mf6.file_io.package_reader_base import PackageReaderBase
from xms.mf6.file_io.text_file_context_manager import TextFileContextManager


class MfsimReader(PackageReader):
    """Imports a MODFLOW 6 simulation."""
    def __init__(self):
        """Initializes the class."""
        super().__init__(ftype='MFSIM6')
        self._prev_block_number = -1  # Previous block number (e.g. 'BEGIN PERIOD 1')
        self._curr_solutiongroup = None  # Current SolutionGroup
        self.sim_node = None  # True if reading to export. Most stuff is skipped and read by calling function.
        self.node_to_data = {}  # Dict filled in _read_from_tree so that packages can later be found from tree nodes

        PackageReaderBase.importing = True

    def _read_timing(self, line):
        """Reads the timing block from the mfsim.nam file.

        Args:
            line (str): A line from the file.
        """
        if self.sim_node:  # This will be read in _read_from_tree. We just want to skip through to the solutiongroups.
            return

        words = line.split()
        if words and len(words) > 1 and words[0].upper() == 'TDIS6':
            filename = words[1]
            filename = fs.resolve_relative_path(self._data.filename, filename)
            reader = io_factory.reader_from_ftype('TDIS6')
            self._data.tdis = reader.read(filename, mfsim=self._data)

    def _read_models(self, line):
        """Reads the models block from the mfsim.nam file.

        Args:
            line (str): A line from the file.
        """
        if self.sim_node:  # This will be read in _read_from_tree. We just want to skip through to the solutiongroups.
            return

        words = line.split()
        if words and len(words) > 2:
            model_filename = fs.resolve_relative_path(self._data.filename, words[1])
            ftype = words[0].upper()
            if ftype in data_util.model_ftypes():
                reader = io_factory.reader_from_ftype(ftype)
                data = reader.read(model_filename, mfsim=self._data)
                data.mtype = words[0]
                if len(words) > 2:
                    data.pname = data.mname = words[2].strip('"\'‘’“”')
                    if not self._data.pname:
                        # set simulation name to first model name
                        self._data.pname = os.path.splitext(words[1])[0].strip('"\'‘’“”')
                self._data.models.append(data)

    def _read_exchanges(self, line):
        """Reads the exchanges block from the mfsim.nam file.

        Args:
            line (str): A line from the file.
        """
        if self.sim_node:  # This will be read in _read_from_tree. We just want to skip through to the solutiongroups.
            return

        words = line.split()
        if words and len(words) > 3:
            exgtype = words[0].upper()
            reader = io_factory.reader_from_ftype(exgtype)
            exch_file = fs.resolve_relative_path(self._data.filename, words[1])
            exg_data = reader.read(exch_file, mfsim=self._data)
            exg_data.exgtype = exgtype
            exg_data.exgmnamea = words[2]
            exg_data.exgmnameb = words[3]
            self._data.exchanges.append(exg_data)

    def _read_solutiongroup(self, line):
        """Reads the solution group block from the mfsim.nam file.

        Args:
            line (str): A line from the file.
        """
        if self._curr_block_number != self._prev_block_number:
            self._curr_solutiongroup = SolutionGroup()
            self._data.solution_groups.append(self._curr_solutiongroup)
            self._prev_block_number = self._curr_block_number

        line = io_util.remove_trailing_comment(line)
        words = line.split()
        if words and len(words) > 1:
            if words[0].upper() == 'MXITER':
                self._curr_solutiongroup.use_mxiter = True
                self._curr_solutiongroup.mxiter = int(words[1])
            elif not self.sim_node:
                # When exporting don't read the IMS. It will get read during _read_from_tree.
                slnfname = fs.resolve_relative_path(self._data.filename, words[1])
                reader = io_factory.reader_from_ftype('IMS6')
                ims_data = reader.read(slnfname, mfsim=self._data)
                self._curr_solutiongroup.ims_list.append(ims_data)
                self._curr_solutiongroup.ims_list[-1].slnmnames = [w.strip('"\'‘’“”') for w in words[2:]]

    def _on_end_file(self):
        """Called after closing the file."""
        # Set this global variable back to initial state.
        PackageReaderBase.importing = False

    def _read_from_tree(self):
        # Iterate through Project Explorer dict reading in the data and creating a MfsimData class
        self.node_to_data.clear()

        # First read TDIS as nper is required to be set when some packages are read
        for node in self.sim_node.children:
            ftype = node.unique_name  # unique_name is the ftype
            if ftype == 'TDIS6':
                data = self._read_tdis_ims_and_exchanges(ftype, node)
                self.node_to_data[node.uuid] = data
                break

        # Read everything else
        for node in self.sim_node.children:
            ftype = node.unique_name  # unique_name is the ftype
            data = None
            if ftype:
                if ftype in data_util.model_ftypes():
                    reader = io_factory.reader_from_ftype(ftype)
                    data = reader.read(node.main_file, node, mfsim=self._data, tree_node=node)
                    self._data.add_model(data)
                    self.node_to_data = {**self.node_to_data, **reader.node_to_data}  # Add model's dict to ours
                elif ftype != 'TDIS6':  # We already read TDIS above
                    data = self._read_tdis_ims_and_exchanges(ftype, node)
                if data:
                    self.node_to_data[node.uuid] = data
        return self._data

    def _read_tdis_ims_and_exchanges(self, ftype, node):
        """Reads the TDIS, IMS, or exchange packages.

        Args:
            ftype (str): The file type used in the GWT name file (e.g. 'WEL6')
            node (TreeNode): A node of the tree.

        Returns:
            The package data object.
        """
        data = None
        reader = io_factory.reader_from_ftype(ftype)
        if reader:
            data = reader.read(node.main_file, mfsim=self._data, tree_node=node)
            if not data:
                raise RuntimeError(f'Could not read ftype: "{ftype}", file: "{node.main_file}"')
            else:
                data.pname = node.name
                if ftype == 'TDIS6':
                    self._data.tdis = data
                elif ftype == 'IMS6':
                    # Note that we only support one solution group for now.
                    if self._data.solution_groups:
                        self._data.solution_groups[0].ims_list.append(data)
                elif ftype in data_util.exchange_ftypes():
                    self._data.exchanges.append(data)
        return data

    def read(self, filename, sim_node: TreeNode | None = None, query: Query | None = None) -> MfsimData:
        """Reads the mfsim.nam file.

        Args:
            filename: the mfsim.nam filename.
            sim_node: Simulation tree node.
            query: Object for communicating with GMS. Only needed to store with the sim in the xms_data variable.
        """
        self.sim_node = sim_node
        # Read mfsim.nam file. If sim_node, skips most and defers to _read_from_tree
        super().read(filename, tree_node=sim_node)
        if self._data:
            self._data.xms_data.query = query
            if sim_node:
                self._read_from_tree()
        return self._data


def model_name_files_from_mfsim_nam_file(mfsim_nam):  # noqa: C901  (too complex)
    """Reads the mfsim.nam file to find the GWF filename and returns it.

    Args:
        mfsim_nam (str): File path to the mfsim.nam file.

    Returns:
        (tuple): tuple containing:
            - model_filenames (list[str]): The full filepath of the GWF/GWT file.
            - mnames (list[str]): mname (user-assigned name) of GWF/GWT model.
            - ftypes (list[str]): ftypes of the models ('GWF6', 'GWT6').
    """
    model_filenames = []
    mnames = []
    ftypes = []
    with TextFileContextManager(mfsim_nam) as fp:
        while True:
            line = fp.readline()
            if not line:
                break

            if io_util.is_begin_line(line):
                if io_util.is_begin_x_line(line, 'MODELS'):
                    while True:
                        line2 = fp.readline()
                        if not line2:
                            break

                        if io_util.is_end_line(line2):
                            break
                        else:
                            words = line2.split()
                            if words and len(words) > 1 and words[0].upper() in data_util.model_ftypes():
                                filename = fs.resolve_relative_path(os.path.dirname(mfsim_nam), words[1])
                                model_filenames.append(filename)
                                mnames.append(words[2])
                                ftypes.append(words[0])
                    break
                else:
                    io_util.skip_block(fp)
    return model_filenames, mnames, ftypes
