"""ModelWriterBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.model_data_base import ModelDataBase
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.package_writer_base import PackageWriterBase
from xms.mf6.misc.settings import Settings


def _write_projection_file(model: ModelDataBase, dis_filepath: Path | str) -> None:
    """Writes a .prj file that has the projection, if the grid has a projection.

    Projection file is saved next to dis* file, e.g. flow.disv.prj.

    Args:
        model: The model.
        dis_filepath: Filepath of dis* file.
    """
    wkt = ''
    if not model:
        return
    elif model.projection_wkt:
        wkt = model.projection_wkt
    else:
        dogrid = model.get_dogrid()
        if dogrid and dogrid.projection and dogrid.projection.coordinate_system != 'None':
            wkt = dogrid.projection.well_known_text
    if wkt:
        prj_filepath = f'{str(dis_filepath)}.prj'  # e.g. 'model.dis.prj'
        with open(prj_filepath, 'w') as file:
            file.write(wkt)


class ModelWriterBase(PackageWriterBase):
    """Writes a GWF model."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()

        self._data = None
        self._fp = None
        self._set_pnames = set()  # set of PNAMES

    def _running_tests(self):
        """Returns True if we're running tests from GMS.

        Returns:
            (bool): See description.
        """
        xms_temp = os.environ.get('XMS_PYTHON_APP_TEMP_DIRECTORY', '')
        # testing = os.environ.get('XMS_PYTHON_APP_RUNNING_TESTS', '')
        if xms_temp:
            return os.path.isfile(os.path.join(xms_temp, 'debug_mf6_testing.dbg'))
        return False

    def _write_options(self, fp):
        """Writes the options block.

        Args:
            fp: The file object.
        """
        running_tests = self._running_tests()
        list_option_exists = self._data.options_block.has('LIST')

        if running_tests and not list_option_exists:
            # Add the list file option if we're testing so GMS can use it to compare to a baseline
            list_filename = self._default_list_filename()
            self._data.options_block.set('LIST', True, list_filename)
            list_option_exists = True

        # If LIST option, put it in output dir if needed.
        if list_option_exists and self._writer_options.use_output_dir:
            list_filename = self._data.options_block.get('LIST')
            if not list_filename:
                list_filename = self._default_list_filename()
            out_dir = os.path.basename(self._writer_options.output_dir)
            list_filename = os.path.join(out_dir, os.path.basename(list_filename)).replace('\\', '/')
            self._data.options_block.set('LIST', True, list_filename)

        super()._write_options(fp)

    def _default_list_filename(self):
        """Returns the default list filename which is the base name plus .lst.

        Returns:
            (str): See description.s
        """
        basename = os.path.splitext(os.path.basename(self._data.filename))[0]
        return basename + '.lst'

    def _get_pname(self, package):
        """Get the PNAME from the package. If we are doing DMI then we will assign a PNAME if it is empty.

        Args:
            package (package data): MF6 package data class
        """
        if self._writer_options.dmi_sim_dir:
            if not package.pname:
                base_pname = package.ftype[:-1]
                pname = base_pname
                cnt = 1
                while pname in self._set_pnames:
                    pname = f'{base_pname}-{cnt}'
                    cnt += 1
                self._set_pnames.add(pname)
                package.pname = pname
        pname = f" '{package.pname}'" if package.pname else ''
        return pname

    def _write_packages(self):
        """Writes the packages block."""
        self._fp.write('\n')
        self._fp.write('BEGIN PACKAGES\n')
        auto_file_names = []
        if not self._writer_options.dmi_sim_dir and self._writer_options.auto_file_naming:
            auto_file_names = data_util.compute_auto_names(self._data.packages, self._data.mname)
        for i, package in enumerate(self._data.packages):
            auto_file_name = auto_file_names[i] if auto_file_names else ''
            package.filename, relative_path = self._get_new_filename(package, auto_file_name)
            pname = self._get_pname(package)
            if package.ftype not in {'POBS6', 'ZONE6'}:
                self._fp.write(f'{io_util.mftab}{package.ftype} {io_util.quote(relative_path)}{pname}\n')

            if not self._writer_options.just_name_file:
                writer = io_factory.writer_from_ftype(package.ftype, self._writer_options)
                if writer:
                    if not self._data.gui_edit_active:
                        writer.write(package)
                        if package.ftype in {'DIS6', 'DISV6', 'DISU6'}:
                            _write_projection_file(self._data, package.filename)
                    else:
                        writer.write_settings(package)
        self._fp.write('END PACKAGES\n')

    def _setup_input_dir(self):
        """Creates the input directory if the use_open_close option is true."""
        if self._writer_options.use_open_close and self._writer_options.use_input_dir:
            basename = os.path.splitext(os.path.basename(self._data.filename))[0]
            path = os.path.dirname(self._data.filename)
            new_dir = os.path.join(path, f'{basename}_input')
            if os.path.isdir(new_dir):
                fs.clear_folder(new_dir)
            else:
                os.makedirs(new_dir, exist_ok=True)
            self._writer_options.open_close_dir = new_dir

    def _setup_output_dir(self):
        """Creates the output directory if the use_output_dir option is true."""
        if self._writer_options.use_output_dir:
            basename = os.path.splitext(os.path.basename(self._data.filename))[0]
            path = os.path.dirname(self._data.filename)
            new_dir = os.path.join(path, f'{basename}_output')
            if not os.path.isdir(new_dir):
                os.makedirs(new_dir, exist_ok=True)
            # if os.path.isdir(new_dir):
            #     fs.clear_folder(new_dir)
            # else:
            #     os.makedirs(new_dir, exist_ok=True)
            self._writer_options.output_dir = new_dir
            # if testing then create this file
            if self._writer_options.running_io_tests:
                keep_file = os.path.join(new_dir, '.keep')
                with open(keep_file, 'w') as self._fp:
                    pass

    # def _add_dot_keep_file(self):
    #     """Puts a .keep file in the input folder if it was empty (for testing purposes)."""
    #     if self._writer_options.running_io_tests and self._writer_options.use_open_close:
    #         for _, _, files in os.walk(self._writer_options.open_close_dir):
    #             if not files:
    #                 keep_file = os.path.join(self._writer_options.open_close_dir, '.keep')
    #                 with open(keep_file, 'w') as self._fp:
    #                     pass

    def _create_pname_set(self):
        # build a set of PNAMES
        self._set_pnames = set()
        for package in self._data.packages:
            if package.pname:
                self._set_pnames.add(package.pname)

    # @overrides
    def write_settings(self, data):
        """If writing to components area, save fname to a settings file.

        Args:
            data: The package data.
        """
        if self._writer_options.dmi_sim_dir and self._data:
            settings = Settings.read_settings(data.filename)
            packages = {}
            for package in self._data.packages:
                if package.tree_node:
                    packages[package.fname] = package.tree_node.uuid
                else:
                    uuid = io_util.uuid_from_path(package.filename)
                    packages[package.fname] = uuid
            if packages:
                settings['PACKAGES'] = packages
            Settings.write_settings(data.filename, settings)

    def _write_package(self, data):
        """Writes the mf6 simulation.

        Args:
            data (MfsimData): the modflow simulation
        """
        self._data = data
        self._create_pname_set()
        if not self._data.gui_edit_active:
            self._setup_input_dir()
            self._setup_output_dir()

        with open(self._data.filename, 'w') as self._fp:
            self._write_options(self._fp)
            self._write_packages()
