"""GriddataDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QWidget
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.griddata_widget import GriddataWidget


class GriddataDialog(PackageDialogBase):
    """A dialog used for the DIS, IC, NPF, STO and Zone packages."""
    def __init__(self, dlg_input: DialogInput, read_only_tabs: list[str] | None = None, parent: QWidget | None = None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input: Information needed by the dialog.
            read_only_tabs: List of tabs that should be read only.
            parent: The parent window.
        """
        super().__init__(dlg_input, parent)
        self._read_only_tabs = read_only_tabs
        self.griddata_widget: GriddataWidget | None = None
        self.setup_ui()

    @override
    def clear_sections(self) -> None:
        """Delete all section widgets."""
        self.griddata_widget = None
        super().clear_sections()

    @override
    def setup_ui(self) -> None:
        """Sets up everything dealing with sections."""
        self.griddata_widget = GriddataWidget(self.dlg_input, self._read_only_tabs, self)
        super().setup_ui()

    @override
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections should be set here.
        """
        raise NotImplementedError()

    @override
    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'GRIDDATA':
            self.setup_griddata_section()
        else:
            super().setup_section(section_name)

    @override
    def setup_signals(self):
        """Sets up any needed signals."""
        super().setup_signals()

    def setup_griddata_section(self):
        """Sets up the GRIDDATA section."""
        name = 'GRIDDATA'
        self.add_group_box_to_scroll_area(name)

        self.uix[name]['griddata_widget'] = self.griddata_widget
        self.uix[name]['layout'].addWidget(self.griddata_widget)
        self.griddata_widget.setup()

    @override
    def do_enabling(self):
        """Enables and disables the widgets appropriately."""
        super().do_enabling()
        self.griddata_widget.do_enabling()

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.griddata_widget.accept()
