"""GUI utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import csv
import os
from pathlib import Path
import re
from typing import Any

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QSignalBlocker, Qt
from PySide2.QtGui import QPalette
from PySide2.QtWidgets import (QDialog, QFileDialog, QHBoxLayout, QLabel, QSizePolicy, QSpinBox, QTabWidget, QWidget)

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import CoverageItem, DatasetItem
from xms.api.tree import TreeNode
from xms.guipy import settings
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.file_io import io_util
from xms.mf6.gui.qx_spin_box_style import QxSpinBoxStyle
from xms.mf6.gui.treeitem_selector_with_check_box_ui import Ui_dlg_treeitem_selector_with_check_box
from xms.mf6.gui.treeitem_selector_with_combo_box_ui import Ui_dlg_treeitem_selector_with_combo_box
from xms.mf6.misc import util


class SignalBlocker:
    """Context manager for QSignalBlocker, which apparently has its own in Pyside6."""
    def __init__(self, widget: QWidget):
        """Initializer.

        Args:
            widget: A widget.
        """
        self._widget = widget
        self._blocker: QSignalBlocker | None = None

    def __enter__(self):
        """Context manager enter.

        Returns:
            Self.
        """
        self._blocker = QSignalBlocker(self._widget)
        return self

    def __exit__(self, exc_type, exc_value, exc_tb) -> None:
        """Context manager exit.

        Args:
            exc_type: Exception type.
            exc_value: Exception value.
            exc_tb: Exception traceback.
        """
        self._blocker.unblock()


def handle_read_only_notice(locked, txt_read_only):
    """Shows or hides the read only message at the top of the dialog."""
    txt_read_only.setVisible(locked)
    if locked:
        txt_read_only.setStyleSheet("QLabel { background-color : rgb(255, 255, 128); color : rgb(0, 0, 0); }")


def open_file_in_default_app(filename, *args):
    """Opens the file in whatever app is associated with the file extension.

    *args exists because we pass this function as an argument to ListDialog,
    and we also sometimes pass ObsDialog.run_dialog_on_file, and it
    takes two args, so this one needs to also take two args.

    Args:
        filename (str): The filepath.
    """
    os.startfile(filename, 'open')


def set_vertical_header_menu_method(table, method):
    """Set the method to be called when right-clicking on the vertical header.

    Args:
        table: The table. Something derived from QTableView.
        method: Method to call when right-clicking in the header column.
    """
    table.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
    table.verticalHeader().customContextMenuRequested.connect(method)


def set_horizontal_header_menu_method(table, method):
    """Set the method to be called when right-clicking on the vertical header.

    Args:
        table: The table. Something derived from QTableView.
        method: Method to call when right-clicking in the header column.
    """
    table.horizontalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
    table.horizontalHeader().customContextMenuRequested.connect(method)


def set_table_menu_method(table, method):
    """Set the method to be called when right-clicking in the table.

    Args:
        table: The table. Something derived from QTableView.
        method: Method to call when right-clicking anywhere else.
    """
    table.setContextMenuPolicy(Qt.CustomContextMenu)
    table.customContextMenuRequested.connect(method)


def get_selected_rows_or_warn(q_table):
    """Get the selected rows and warn the user if the selection is disjoint.

    If the selection is disjoint, warns the user and returns False.

    Returns:
        (tuple): tuple containing:
            - (bool): True or False.
            - (int): the minimum selected row.
            - (int): the maximum selected row.
    """
    selected_list = q_table.selectedIndexes()

    # Get the count of unique rows and the span from the selection
    unique_rows = get_unique_selected_rows(selected_list)
    if not unique_rows:
        return False, 0, 0
    minrow = min(unique_rows)
    maxrow = max(unique_rows)
    span = maxrow - minrow + 1

    if warn_if_disjoint_selection(q_table, len(unique_rows), span):
        return False, minrow, maxrow
    return True, minrow, maxrow


def get_unique_selected_rows(selected_list) -> list[int]:
    """Returns a list of selected row indices.

     Selection may be disjoint.

    Args:
        selected_list (list of QModelIndex): The selected list.

    Returns:
        See description.
    """
    row_set = set()
    for index in selected_list:
        row_set.add(index.row())
    return list(row_set)


def warn_if_disjoint_selection(parent, unique_rows, span):
    """Warns the user with a message box if unique_rows is not equal to span.

    Args:
        parent (Something derived from QWidget): The parent window.
        unique_rows (int): Number of unique rows selected.
        span (int): The maximum row selected minus the minimum row selected.

    Returns:
        bool: True if unique_rows != span, else False.
    """
    if unique_rows != span:
        message_box.message_with_ok(parent=parent, message='Cannot process disjoint selections.')
        return True
    return False


def read_csv_file_into_dataframe(
    filename, column_names, column_types, usecols: list | None = None, separator=io_util.mfsep
):
    """Reads the file into a pandas dataframe.

    Args:
        filename (str): Path of the external file.
        column_names (list of str): List of column names.
        column_types (dict of numpy number types) : Dictionary of column name and numpy type.
        usecols: pandas.read_csv() usecols arg. List of int or list of str. Causes other columns to be ignored.
        separator (str): Separator character.

    Returns:
         (DataFrame): The pandas DataFrame.
    """
    try:
        # start_time = time.time()

        # list_data_regex = r"\s*[,\s]\s*|[#!/]"  # Regex used to parse list data
        # # "Comment lines and blanks lines are also allowed within most blocks
        # # and within most input files. Valid comment characters include “#” “!”,
        # # and “//”. Comments can also be placed at the end of some input lines,
        # # after the required information. ... Comments included at the end of
        # # the line must be separated from the rest of the line by at least one
        # # space." mf6io.pdf page 3.
        # #
        # # "One or more spaces, or a single comma optionally combined with
        # #  spaces, must separate adjacent values." mf6io.pdf page 19.
        #
        # data_frame = pandas.read_csv(filename, header=None, names=column_names, index_col=False,
        #                              dtype=column_types, float_precision='high',
        #                              sep=list_data_regex, engine='python')

        # data_frame = pandas.read_csv(filename, header=None, names=column_names, index_col=False,
        #                              usecols=column_names, dtype=column_types, sep=io_util.mfsep,
        #                              float_precision='high')
        if filename:
            data_frame = pd.read_csv(
                filename,
                header=None,
                names=column_names,
                index_col=False,
                dtype=column_types,
                sep=separator,
                float_precision='high',
                quotechar='\'',
                usecols=usecols
            )
        else:
            data_frame = empty_dataframe(column_names, list(column_types.values()), index=None)

        data_frame.index = data_frame.index + 1  # Start index at 1, not 0
        # data_frame.fillna(value='', inplace=True)

        # print('Time to read from disk: {}'.format(time.time() - start_time))

    except Exception as error:
        raise error

    return data_frame


def default_dataframe(numeric_type, shape, default_value):
    """Creates a dataframe sized appropriately for the grid and DIS package.

    Args:
        numeric_type (str): 'int' or 'float'
        shape (GridInfo): dimensions of the grid
        default_value: The default value to assign to fill the dataframe with.

    Returns:
        dataframe (DataFrame): The pandas DataFrame.
    """
    dtype = np.int64 if numeric_type == 'int' else np.float64
    narray = np.empty(shape, dtype=dtype)
    narray.fill(default_value)

    data_frame = pd.DataFrame(data=narray)
    data_frame.index += 1
    data_frame.columns += 1
    return data_frame


def empty_dataframe(columns, dtypes, index=None):
    """Creates and returns an empty dataframe with the given column types.

    From https://stackoverflow.com/questions/36462257/create-empty-dataframe-in-pandas-specifying-column-types

    Args:
        columns (list of str): List of column names.
        dtypes (list of dtypes): List of column types.
        index: Optional index

    Returns:
        The new dataframe.
    """
    assert len(columns) == len(dtypes)
    df = pd.DataFrame(index=index)
    for c, d in zip(columns, dtypes):
        df[c] = pd.Series(dtype=d)
    return df


def change_columns_in_file(old_columns, new_columns, default_value, old_filename, new_filename):
    """Reads a file and writes a new file, fixing the columns from old_columns to new_columns.

    Args:
        old_columns (list or str): The columns in the old file.
        new_columns (list or str): The columns in the new file.
        default_value (str): Default value to add to new columns.
        old_filename (str): The old filename.
        new_filename (str): The new filename.
    """
    if not os.path.isfile(old_filename):
        return

    with open(old_filename, 'r') as infile, open(new_filename, 'w', newline='') as outfile:
        writer = csv.DictWriter(
            outfile, fieldnames=new_columns, restval=default_value, extrasaction='ignore', delimiter=io_util.mfsep
        )
        for row in csv.DictReader(infile, fieldnames=old_columns, delimiter=io_util.mfsep):
            writer.writerow(row)


def array_dataframe_to_temp_file(dataframe, array):
    """Writes the dataframe to disk and saves the filename into array.

    Also deletes any old file that may be saved with array.

    Args:
        dataframe (DataFrame): The DataFrame.
        array (Array): The array
    """
    temp_filename = dataframe_to_temp_file(dataframe)

    # Remove old temp file if one exists and update with new temp filename
    if array.temp_external_filename:
        os.remove(array.temp_external_filename)
    array.temp_external_filename = temp_filename


def dataframe_to_temp_file(dataframe, separator=io_util.mfsep):
    """Saves the dataframe to a temporary file and returns the filename.

    Args:
        dataframe (DataFrame): The dataframe.
        separator (str): Separator character.

    Returns:
        (str): The temporary filename.
    """
    # Create a new temp file
    dataframe.fillna(value=0.0, inplace=True)  # Otherwise nan becomes nothing and we are off when we read it back in
    with open(io_util.get_temp_filename(), mode='wt') as temp:
        dataframe_to_csv(dataframe, temp.name, separator=separator)
        temp.close()
        return temp.name


def dataframe_to_csv(df: pd.DataFrame, filepath: Pathlike, separator: str = io_util.mfsep, append: bool = False):
    """Wrapped DataFrame.to_csv() so we could use it with or without a file but with the same args.

    Args:
        df: The dataframe.
        filepath: Filepath. If empty, the csv string is returned.
        separator: Separator character.
        append: True to append to an existing file.

    Returns:
        (str): The csv string if filepath is empty, otherwise ''.
    """
    if filepath:
        mode = 'a' if append else 'w'
        df.to_csv(filepath, mode=mode, encoding='utf-8', index=False, header=False, sep=separator, quotechar='\'')
        return ''
    else:
        return df.to_csv(encoding='utf-8', index=False, header=False, sep=separator, quotechar='\'')


def apply_filter_on_selected_cells(dlg_input, grid_info, df):
    """Returns a dataframe filtered to only show selected cells, if there are any.

    Args:
        dlg_input (DialogInput): Information needed by the dialog.
        grid_info (GridInfo): Information about the grid.
        df: The dataframe.

    Returns:
        See description.
    """
    eval_str = ''
    if dlg_input.selected_cells and len(df.index) > 0:
        # with open('C:/temp/debug.txt', 'a') as file:
        #     file.write(f'grid_info: {grid_info}\n')
        gi = grid_info  # For convenience
        if dlg_input.data.ftype == 'SFR6':  # SFR now always has a CELLID column with 3, 2, or 1 integer in it
            if 'CELLIDX' not in df:
                # Append a CELLIDX column onto the dataframe
                cell_idxs = cell_idxs_from_cellids(grid_info, df['CELLID'].tolist())
                df['CELLIDX'] = cell_idxs
            return df[df['CELLIDX'].isin(dlg_input.selected_cells)]
        else:
            if gi.dis_enum == DisEnum.DIS and 'ROW' in df.columns:
                # eval_str = '((LAY - 1) * (@gi.nrow * @gi.ncol)) + ((ROW - 1) * @gi.ncol) + (COL - 1)'
                eval_str = f'((LAY - 1) * ({gi.nrow * gi.ncol})) + ((ROW - 1) * {gi.ncol}) + (COL - 1)'
            elif gi.dis_enum == DisEnum.DISV and 'CELL2D' in df.columns:
                # eval_str = '((LAY - 1) * @gi.ncpl) + CELL2D - 1'
                eval_str = f'((LAY - 1) * {gi.ncpl}) + CELL2D - 1'
            elif 'CELLID' in df.columns:
                eval_str = '(CELLID - 1)'

    if eval_str:
        cell_idxs = df.eval(eval_str)
        return df[cell_idxs.isin(dlg_input.selected_cells)]
    else:
        return df


def set_vertical_stretch(widget, stretch):
    """Sets the vertical stretch on the widget.

    Convenience function to avoid duplicating code.

    Args:
        widget: The widget.
        stretch (int): The stretch factor.
    """
    size = QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)
    size.setVerticalStretch(stretch)
    widget.setSizePolicy(size)


def setup_stress_period_spin_box_layout(parent, uix, nper, signal_method):
    """Sets up the stress period spinbox and create/delete toolbar.

    Combines common code in PeriodListWidget and PeriodArrayWidget.

    Args:
        parent: Parent widget.
        uix: Dict of ui items
        nper (int): Number of stress periods.
        signal_method: Method to call when the stress period changes.

    Returns:
        The layout.
    """
    # Create layout
    hlayout = QHBoxLayout()

    # Create label and add to layout
    lbl = uix['txt_period'] = QLabel('Period:')
    hlayout.addWidget(lbl)

    # Create spn control and add to layout
    spn = uix['spn_period'] = QSpinBox(parent)
    spn.setAccessibleName('Period')
    spn.setStyle(QxSpinBoxStyle())
    hlayout.addWidget(spn)
    spn.setMinimum(1)
    spn.setMaximum(nper)
    spn.setValue(1)
    spn.setKeyboardTracking(False)

    # Make it so the width doesn't change. Calling setFixedWidth() seems to be only way that works
    # spn.setStyleSheet('QSpinBox {min-width: 100px; min-height: 20px;}')
    # spn.setSizePolicy(QSizePolicy.Fixed, spn.sizePolicy().verticalPolicy())
    min_width = 80  # This works well on my machine
    spn.setFixedWidth(max(min_width, spn.sizeHint().width()))  # Use the size hint if it's wider

    # Have spin control only signal when done editing
    spn.valueChanged.connect(signal_method, Qt.UniqueConnection)
    return hlayout


def set_label_styles_for_warning_message(qlabel):
    """Sets the label text and background color and other styles to indicate a warning.

    Args:
        qlabel (QLabel): The label.
    """
    qlabel.setWordWrap(True)
    qlabel.setOpenExternalLinks(True)
    qlabel.setTextFormat(Qt.RichText)
    qlabel.setTextInteractionFlags(Qt.TextBrowserInteraction)
    qlabel.setStyleSheet("QLabel { background-color : rgb(200, 200, 150); color : rgb(0, 0, 0); }")


def column_info_tuple_from_dict(columns):
    """Converts columns dict into a tuple with: name list, types dict, and defaults dict.

    Args:
        columns (dict): A dict of column names -> tuple with type and default.

    Returns:
        (tuple): tuple containing:
            - names (list): Column names.
            - types (dict of str -> type): Column names -> column types.
            - default (dict of str -> value): Column names -> default values.
    """
    names = []
    types = {}
    defaults = {}

    for key, item in columns.items():
        names.append(key)
        types[key] = item[0]
        defaults[key] = item[1]

    return names, types, defaults


def get_icon_path(node: TreeNode) -> str:
    """Method used by TreeItemSelectorDlg to display icons.

    Args:
        node (TreeNode): A tree node.

    Returns:
        (str): Icon path.
    """
    if node is None:
        return ''

    path = ''
    if node.item_typename == 'TI_COMPONENT':
        if node.unique_name == 'GWF6':
            path = ':/resources/icons/mf6_gwf_model.svg'
        elif node.unique_name == 'GWT6':
            path = ':/resources/icons/mf6_gwt_model.svg'
        elif node.unique_name == 'GWE6':
            path = ':/resources/icons/mf6_gwe_model.svg'
        else:
            path = ':/resources/icons/package.svg'
    elif node.item_typename == 'TI_DYN_SIM_FOLDER':
        path = ':/resources/icons/mf6_simulations.svg'
    elif node.item_typename == 'TI_DYN_SIM':
        path = ':/resources/icons/mf6_simulation.svg'
    elif node.item_typename == 'TI_ROOT':
        if node.name == 'UGrid Data':
            path = ':/resources/icons/folder.svg'
    elif node.item_typename == 'TI_ROOT_SIMULATION':
        path = ':/resources/icons/simulation_data.svg'
    elif node.item_typename == 'TI_SCALAR_DSET':
        if node.name in {'Point Z', 'Cell Top Z', 'Cell Bottom Z'}:
            path = ':/resources/icons/elevation_data_inactive.svg'
        else:
            if node.data_location == 'CELL':
                path = ':/resources/icons/dataset_cells_inactive.svg'
            elif node.data_location == 'NODE':
                path = ':/resources/icons/dataset_points_inactive.svg'
    elif node.item_typename == 'TI_SOLUTION_FOLDER':
        path = ':/resources/icons/folder_locked.svg'
    # TreeNode doesn't have type of UGrid constraint so we can't do any better yet.
    # elif node.item_typename == 'TI_UGRID':
    #     path = ''
    return path


def set_read_only_and_grey(widget, on):
    """Sets the widget to be read-only and changes the color to grey, or the reverse.

    Args:
        widget: The widget.
        on (bool): Whether we are setting or unsetting.
    """
    widget.setReadOnly(on)
    if on:
        read_only_palette = QPalette()
        read_only_palette.setColor(QPalette.Base, widget.palette().color(QPalette.Window))
        widget.setPalette(read_only_palette)
    else:
        my_type = type(widget)
        default_widget = my_type()
        widget.setPalette(default_widget.palette())


def tab_index_from_text(tab_widget: QTabWidget, tab_text: str) -> int:
    """Returns the index of the tab with the given text, or -1 if not found.

    Args:
        tab_widget (QTabWidget): The tab widget.
        tab_text (str): Title of the tab in question.
    """
    tab_index = -1
    for index in range(tab_widget.count()):
        if tab_widget.tabText(0) == tab_text:
            tab_index = index
            break
    return tab_index


def _create_select_dataset_dialog(
    parent: QWidget, tree: TreeNode, selected_uuid: str, cell_count: int, ui_class=None
) -> TreeItemSelectorDlg:
    """Runs the dialog to let the user select a dataset and returns the selected dataset uuid, or ''.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.
        ui_class: The ui class to use.

    Returns:
        See description.
    """
    def callback(node: TreeNode) -> bool:
        # Callback method to see if dataset should be shown in the dialog
        return node.num_vals == cell_count

    dialog = TreeItemSelectorDlg(
        title='Select Dataset',
        target_type=DatasetItem,
        pe_tree=tree,
        override_icon=get_icon_path,
        previous_selection=selected_uuid,
        parent=parent,
        selectable_callback=callback,
        ui_class=ui_class,
    )
    return dialog


def select_dataset_dialog(
    parent: QWidget,
    tree: TreeNode,
    selected_uuid: str,
    cell_count: int,
) -> None | str:
    """Runs the dialog to let the user select a dataset.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.

    Returns:
        The uuid.
    """
    dialog = _create_select_dataset_dialog(parent, tree, selected_uuid, cell_count, None)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_item_uuid()
    return None


def select_dataset_dialog_combo_box(
    parent: QWidget,
    tree: TreeNode,
    selected_uuid: str,
    cell_count: int,
    cbx_items: dict,
    cbx_start: str,
) -> tuple[str | None, Any]:
    """Runs the dialog to let the user select a dataset, adding a combo box at the top.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.
        cbx_items: Dict of strings -> int to put in the combo box.
        cbx_start: Item to show in combo box to start.

    Returns:
        Tuple of uuid and QComboBox.currentData().
    """
    dialog = _create_select_dataset_dialog(
        parent, tree, selected_uuid, cell_count, Ui_dlg_treeitem_selector_with_combo_box()
    )
    dialog.ui.lbl_combobox.setText('Column:')
    for key, value in cbx_items.items():
        dialog.ui.cbx_combobox.addItem(key, value)
    dialog.ui.cbx_combobox.setCurrentText(cbx_start)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_item_uuid(), dialog.ui.cbx_combobox.currentData()
    return None, None


def select_dataset_dialog_check_box(
    parent: QWidget, tree: TreeNode, selected_uuid: str, cell_count: int, chk_box_text: str,
    chk_box_state: Qt.CheckState
) -> tuple[str | None, Qt.CheckState]:
    """Runs the dialog to let the user select a dataset, adding a checkbox at the bottom.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.
        chk_box_text: QCheckBox text.
        chk_box_state: Starting state of the QCheckBox.

    Returns:
        Tuple of uuid and QCheckBox.checkState().
    """
    dialog = _create_select_dataset_dialog(
        parent, tree, selected_uuid, cell_count, Ui_dlg_treeitem_selector_with_check_box()
    )
    dialog.ui.checkBox.setText(chk_box_text)
    dialog.ui.checkBox.setCheckState(chk_box_state)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_item_uuid(), dialog.ui.checkBox.checkState()
    return None, None


def select_coverages_dialog(parent: QWidget, tree: TreeNode, selected_uuid: str) -> list[str]:
    """Run the TreeItemSelectorDlg to select coverages.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.

    Returns:
        List of the uuids selected.
    """
    dialog = TreeItemSelectorDlg(
        title='Select Coverage(s)',
        target_type=CoverageItem,
        pe_tree=tree,
        previous_selection=selected_uuid,
        parent=parent,
        allow_multi_select=True
    )
    uuids = []
    if dialog.exec() == QDialog.Accepted:
        uuids = dialog.get_selected_item_uuid()
    return uuids


def _correct_path(filepath: Path | str) -> str:
    """Return a valid path for open/save file dialogs: filepath if it exists, else an existing directory.

    Args:
        filepath: A filepath.

    Returns:
        See description.
    """
    filepath = Path(filepath)
    start_dir = Path(settings.get_file_browser_directory())
    if util.null_path(filepath):
        filepath = start_dir
    else:
        if io_util.is_filename(filepath):  # Just a file name?
            filepath = start_dir / filepath  # Add start_dir
        if not filepath.is_file():  # File doesn't exist?
            existing_parent = io_util.find_existing_parent(filepath.parent)
            if existing_parent:
                filepath = existing_parent / filepath.name  # Use an existing parent
            else:
                filepath = start_dir / filepath.name
    return str(filepath)


def run_open_file_dialog(
    parent: QWidget, caption: str, filepath: Path | str, filter_str: str, sel_filter: str = ''
) -> str:
    """Wrap QFileDialog.getOpenFileName(), adding getting existing dir and using get/set_file_browser_directory().

    Args:
        parent: The parent widget.
        caption: The dialog caption.
        filepath: The initial file path.
        filter_str: The filter string, e.g. 'CSV (Comma delimited) Files (*.csv);;All Files (*.*)'
        sel_filter: The selected filter.

    Return:
        The path selected from the dialog.
    """
    filepath_str = _correct_path(filepath)

    # Run the dialog
    filepath_str, _ = QFileDialog.getOpenFileName(parent, caption, filepath_str, filter_str, sel_filter)

    # Save the directory
    if filepath_str:
        dir_ = Path(filepath_str).parent
        settings.save_file_browser_directory(str(dir_))
    return filepath_str


def run_save_file_dialog(
    parent: QWidget, caption: str, filepath: Path | str, filter_str: str, sel_filter: str = ''
) -> str:
    """Wraps QFileDialog.getSaveFileName(), adding getting existing dir and using get/set_file_browser_directory().

    Args:
        parent: The parent widget.
        caption: The dialog caption.
        filepath: The initial file path.
        filter_str: The filter string, e.g. 'CSV (Comma delimited) Files (*.csv);;All Files (*.*)'
        sel_filter: The selected filter.

    Return:
        The path selected from the dialog.
    """
    filepath_str = _correct_path(filepath)

    # Run the dialog
    filepath_str, _ = QFileDialog.getSaveFileName(parent, caption, filepath_str, filter_str, sel_filter)

    # Save the directory
    if filepath_str:
        dir_ = Path(filepath_str).parent
        settings.save_file_browser_directory(str(dir_))
    return filepath_str


def help_getter(key: str) -> HelpGetter:
    """Return a HelpUrl set up using DialogInput.

    Args:
        key: The second part of the wiki help line on the above page (after the '|').

    Returns:
        See description.
    """
    return HelpGetter(key=key, default=util.wiki_mf6, dialog_help_url=util.wiki_dialog_help)


def new_table_view() -> QxTableView:
    """Return a new table view styled how we like it.

    Returns:
        The table view.
    """
    return widget_builder.new_styled_table_view(row_height=0)


def cell_idxs_from_cellids(grid_info: GridInfo, cellids: list) -> list[int]:
    """Return list of cell indices (0-based) given list of modflow cellids (can be tuples depending on DIS*).

    Args:
        grid_info: Grid information.
        cellids: List of cell ids, which can be ints (DISU), or list/tuples of ints (DIS, DISV).

    Returns:
        See description.
    """
    return [grid_info.cell_index_from_modflow_cellid([int(num) for num in cellid.split()]) for cellid in cellids]


def unique_name_no_spaces(names: set[str], name: str, sep_char: str = '_') -> str:
    """Return a name that is not in names and does not contain spaces.

    Args:
        names: A set of names.
        name: A candidate name.
        sep_char: Separator character, like '-' or '_'.

    Returns:
        See description.
    """
    # If name is good already, just return it
    if name not in names and ' ' not in name:
        return name

    # See if we can just replace spaces with dashes to make it unique
    if ' ' in name and name.replace(' ', sep_char) not in names:
        return name.replace(' ', sep_char)

    # Replace spaces with underscore
    name = name.replace(' ', sep_char)

    # Create regex that matches strings ending in sep_char followed by one or more digits, like 'bob-33'
    pattern = fr"{sep_char}\d+$"
    regex = re.compile(pattern)

    # Create unique name by having it end with char and a number
    while name in names:
        if regex.search(name):
            name = _increment_trailing_number(name, sep_char)
        else:
            name = f'{name}{sep_char}2'  # Append string with char and number to start the pattern
    return name


def _increment_trailing_number(text: str, sep_char: str) -> str:
    """Increments the number at the end of a string that follows an underscore.

    Example: "bob_33" becomes "bob_34"

    Args:
        text: The string to modify, e.g., "name_123".
        sep_char: The separator character, like '-' or '_'.

    Returns:
        The modified string with the incremented number, or the original
        string if the pattern isn't found.
    """
    # Pattern: Captures the number at the end preceded by the separator character.
    pattern = fr"{sep_char}(\d+)$"

    def replacer(match):
        """Replacement function for re.sub()."""
        number_str = match.group(1)
        incremented_number = int(number_str) + 1
        return f'{sep_char}{str(incremented_number)}'

    return re.sub(pattern, replacer, text)
