"""EvtListDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QHBoxLayout, QLabel, QLineEdit, QPushButton, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.mf6.gui import get_int_dialog, gui_util
from xms.mf6.gui.package_dialog import PackageDialog


class EvtListDialog(PackageDialog):
    """A dialog for the EVT package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

    # @overrides
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'DIMENSIONS', 'PERIODS']
        self.default_sections = ['PERIODS']

    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'DIMENSIONS':
            self.setup_dimensions_section()
        else:
            super().setup_section(section_name)

    def setup_dimensions_section(self):
        """Sets up the DIMENSIONS section."""
        name = 'DIMENSIONS'
        self.add_group_box_to_scroll_area(name)

        vlayout = self.uix[name]['layout']

        # NSEG
        w = self.uix['widget_nseg'] = QWidget()
        lay_nseg = QHBoxLayout()
        lay_nseg.setContentsMargins(0, 0, 0, 0)
        w.setLayout(lay_nseg)
        ww = self.uix['txt_nseg'] = QLabel('Number of ET segments (NSEG)')
        lay_nseg.addWidget(ww)
        ww = self.uix['edt_nseg'] = QLineEdit()
        ww.setText(str(self.dlg_input.data.nseg))

        # Make it read only
        gui_util.set_read_only_and_grey(ww, True)

        lay_nseg.addWidget(ww)
        ww = self.uix['btn_nseg'] = QPushButton('Change...')
        lay_nseg.addWidget(ww)
        lay_nseg.addStretch()
        vlayout.addWidget(w)
        self.uix['btn_nseg'].clicked.connect(self.on_btn_nseg)

    def change_nseg(self, nseg):
        """Updates NSEG (EVT package) in all stress periods.

        Args:
            nseg (int): New nseg.
        """
        self.uix['edt_nseg'].setText(str(nseg))
        old_columns, _, _ = self.dlg_input.data.get_column_info('')
        self.dlg_input.data.nseg = nseg
        new_columns, _, _ = self.dlg_input.data.get_column_info('')
        use_aux = 'chk_auxiliary' in self.options_gui.uix and self.options_gui.uix['chk_auxiliary'].isChecked()
        self.sp_widget.change_columns(old_columns, new_columns, use_aux=use_aux)

    def change_surf_rate_specified(self, on):
        """Adds or removes the petm0 column to or from all stress periods."""
        old_columns, _, _ = self.dlg_input.data.get_column_info('')
        self.dlg_input.data.options_block.set('SURF_RATE_SPECIFIED', on, None)
        new_columns, _, _ = self.dlg_input.data.get_column_info('')
        use_aux = 'chk_auxiliary' in self.options_gui.uix and self.options_gui.uix['chk_auxiliary'].isChecked()
        self.sp_widget.change_columns(old_columns, new_columns, use_aux=use_aux)

    def on_chk_surf_rate_specified(self):
        """Called when the surf rate checkbox is clicked."""
        is_checked = self.options_gui.uix['chk_surf_rate_specified'].isChecked()
        if is_checked:
            message = 'This will add the petm0 column to all periods. OK?'
        else:
            message = 'This will remove the petm0 column from all periods. OK?'

        rv = message_box.message_with_ok_cancel(parent=self, message=message)
        if rv:
            self.change_surf_rate_specified(is_checked)
        else:
            self.options_gui.uix['chk_surf_rate_specified'].setChecked(not is_checked)

    def on_btn_nseg(self):
        """Handles changing nseg (EVT package)."""
        nseg = int(self.uix['edt_nseg'].text())
        rv, new_nseg = get_int_dialog.run('NSEG', 'New number of ET segments (NSEG):', nseg, 1, 100, self)
        if rv and new_nseg != nseg:
            self.change_nseg(new_nseg)
