"""NewSimDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QPoint
from PySide2.QtWidgets import QSizePolicy, QToolTip

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import UGridItem
from xms.api.tree import tree_util, TreeNode
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_project_explorer_selector_widget import QxProjectExplorerSelectorWidget

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.gui.new_sim_dialog_ui import Ui_NewSimDialog


class NewSimDialog(XmsDlg):
    """A dialog that appears when creating a new simulation."""
    def __init__(self, project_tree, new_uuid: str, parent=None, model_str='', help_id: str = ''):
        """Initializes the class, sets up the ui.

        Args:
            project_tree: Dict of the Project Explorer tree.
            new_uuid: Uuid of the new item being created.
            parent (Something derived from QWidget): The parent window.
            model_str (str): If coming from Add Package > GWF (or GWT), is 'GWF' or 'GWT' respectively
            help_id: The second part of the wiki help line on the above page (after the '|').
        """
        super().__init__(parent, 'xms.mf6.gui.new_sim_dialog')

        self._project_tree = project_tree
        self._new_uuid = new_uuid
        self.help_getter = gui_util.help_getter(help_id)
        self.setWindowTitle('New MODFLOW 6 Simulation')

        self.ui = Ui_NewSimDialog()
        self.ui.setupUi(self)

        self._setup_tree_widget(project_tree)

        self.ui.edt_simulation_name.installEventFilter(self)
        self.model_str = model_str
        new_node = tree_util.find_tree_node_by_uuid(self._project_tree, self._new_uuid)
        if model_str == 'GWF':
            self.ui.txt_simulation_name.setText('Model name:')
            self.ui.edt_simulation_name.setText(_find_good_tree_name(new_node, 'flow'))
            self.setWindowTitle('New Groundwater Flow (GWF) Model')
            self.ui.txt_auto_sim_packages.setVisible(False)
            self.ui.grp_gwt.setVisible(False)
            self.ui.grp_gwe.setVisible(False)
            self.ui.grp_gwf.setCheckable(False)
        elif model_str == 'GWT':
            self.ui.txt_simulation_name.setText('Model name:')
            self.ui.edt_simulation_name.setText(_find_good_tree_name(new_node, 'trans'))
            self.setWindowTitle('New Groundwater Transport (GWT) Model')
            self.ui.txt_auto_sim_packages.setVisible(False)
            self.ui.grp_gwf.setVisible(False)
            self.ui.grp_gwe.setVisible(False)
            self.ui.grp_gwt.setChecked(True)
            self.ui.grp_gwt.setCheckable(False)
        elif model_str == 'GWE':
            self.ui.txt_simulation_name.setText('Model name:')
            self.ui.edt_simulation_name.setText(_find_good_tree_name(new_node, 'energy'))
            self.setWindowTitle('New Groundwater Energy Transport (GWE) Model')
            self.ui.txt_auto_sim_packages.setVisible(False)
            self.ui.grp_gwf.setVisible(False)
            self.ui.grp_gwt.setVisible(False)
            self.ui.grp_gwe.setChecked(True)
            self.ui.grp_gwe.setCheckable(False)
        else:
            self.ui.edt_simulation_name.setText(_find_good_tree_name(new_node, 'sim'))
            self.setWindowTitle('New MODFLOW 6 Simulation')

        self.ui.buttonBox.helpRequested.connect(self.help_requested)

        self.data = {}  # Stores the dialog state for use after dialog is closed on OK.

    def _setup_tree_widget(self, project_tree) -> None:
        """Set up the tree widget.

        Args:
            project_tree: Dict of the Project Explorer tree.
        """
        # If there's only one ugrid, select it by default
        items = tree_util.descendants_of_type(project_tree, UGridItem)
        previous_selection = ''
        if items and len(items) == 1:
            previous_selection = items[0].uuid

        # Swap out the generic QTreeWidget for the QxProjectExplorerSelectorWidget
        self.ui_tree_ugrids = QxProjectExplorerSelectorWidget(
            root_node=project_tree, selectable_item_type=UGridItem, parent=self, previous_selection=previous_selection
        )
        widget_builder.replace_widgets(self.ui.vlay_left.layout(), self.ui.tree_ugrids, self.ui_tree_ugrids)
        self.ui_tree_ugrids.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Minimum)
        self.ui_tree_ugrids.setMinimumWidth(250)

    def eventFilter(self, obj, event):  # noqa N802 (function name 'eventFilter' should be lowercase)
        """Prevent user from entering forbidden characters into the sim name.

        Args:
            obj: Qt object associated with the event.
            event: The QEvent object.

        Returns:
            (bool): True if the event was handled.
        """
        if obj == self.ui.edt_simulation_name and event.type() == QEvent.KeyPress:
            bad_chars = '/\\!*?"<>| '  # Project Explorer names can't have these
            letter = event.text()
            if letter and letter in bad_chars:
                bad_chars_msg = '/ \\ ! * ? " < > |'
                msg = f"The name cannot contain spaces or any of the following characters:\n\t{bad_chars_msg}"
                QToolTip.showText(self.ui.edt_simulation_name.mapToGlobal(QPoint()), msg)
                return True
            return False
        return False

    def add_package(self, packages, checkbox, ftype):
        """Helper function to append a package to the list if the checkbox is checked.

        Args:
            packages (set(str)): List of packages.
            checkbox (QCheckBox): The QCheckbox.
            ftype (`str`): The MODFLOW ftype.
        """
        if checkbox.isChecked():
            packages.add(ftype)

    def update_data(self):
        """Return a list of strings indicating which packages are selected.

        Returns:
            (dict): The dialog data.
        """
        self.data = {
            'name': self.ui.edt_simulation_name.text(),
            'ugrid_uuid': self.ui_tree_ugrids.get_selected_item_uuid(),
        }
        if self.model_str == 'GWF' or (not self.model_str and self.ui.grp_gwf.isChecked()):
            packages = {'GWF6', 'DIS6', 'IC6', 'NPF6', 'OC6'}  # Required
            self.add_package(packages, self.ui.chk_buy, 'BUY6')
            self.add_package(packages, self.ui.chk_chd, 'CHD6')
            self.add_package(packages, self.ui.chk_csub, 'CSUB6')
            self.add_package(packages, self.ui.chk_drn, 'DRN6')
            self.add_package(packages, self.ui.chk_evt, 'EVT6')
            self.add_package(packages, self.ui.chk_evta, 'EVTA6')
            self.add_package(packages, self.ui.chk_ghb, 'GHB6')
            self.add_package(packages, self.ui.chk_gnc, 'GNC6')
            self.add_package(packages, self.ui.chk_hfb, 'HFB6')
            self.add_package(packages, self.ui.chk_lak, 'LAK6')
            self.add_package(packages, self.ui.chk_maw, 'MAW6')
            self.add_package(packages, self.ui.chk_mvr, 'MVR6')
            self.add_package(packages, self.ui.chk_obs, 'OBS6')
            self.add_package(packages, self.ui.chk_rch, 'RCH6')
            self.add_package(packages, self.ui.chk_rcha, 'RCHA6')
            self.add_package(packages, self.ui.chk_riv, 'RIV6')
            self.add_package(packages, self.ui.chk_sfr, 'SFR6')
            self.add_package(packages, self.ui.chk_sto, 'STO6')
            self.add_package(packages, self.ui.chk_swi, 'SWI6')
            self.add_package(packages, self.ui.chk_uzf, 'UZF6')
            self.add_package(packages, self.ui.chk_vsc, 'VSC6')
            self.add_package(packages, self.ui.chk_wel, 'WEL6')
            self.add_package(packages, self.ui.chk_pobs, 'POBS6')
            self.add_package(packages, self.ui.chk_zones, 'ZONE6')
            self.data['gwf_ftypes'] = packages

        if self.model_str == 'GWT' or (not self.model_str and self.ui.grp_gwt.isChecked()):
            packages = {'GWT6', 'DIS6', 'IC6', 'MST6', 'OC6'}  # Required
            self.add_package(packages, self.ui.chk_adv, 'ADV6')
            self.add_package(packages, self.ui.chk_cnc, 'CNC6')
            self.add_package(packages, self.ui.chk_dsp, 'DSP6')
            self.add_package(packages, self.ui.chk_fmi, 'FMI6')
            self.add_package(packages, self.ui.chk_ist, 'IST6')
            self.add_package(packages, self.ui.chk_lkt, 'LKT6')
            self.add_package(packages, self.ui.chk_mdt, 'MDT6')
            self.add_package(packages, self.ui.chk_mvt, 'MVT6')
            self.add_package(packages, self.ui.chk_mwt, 'MWT6')
            self.add_package(packages, self.ui.chk_obs_gwt, 'OBS6')
            self.add_package(packages, self.ui.chk_sft, 'SFT6')
            self.add_package(packages, self.ui.chk_src, 'SRC6')
            self.add_package(packages, self.ui.chk_ssm, 'SSM6')
            self.add_package(packages, self.ui.chk_uzt, 'UZT6')
            self.add_package(packages, self.ui.chk_pobs_gwt, 'POBS6')
            self.data['gwt_ftypes'] = packages

        if self.model_str == 'GWE' or (not self.model_str and self.ui.grp_gwe.isChecked()):
            packages = {'GWE6', 'DIS6', 'IC6', 'OC6'}  # Required
            self.add_package(packages, self.ui.chk_adv_gwe, 'ADV6')
            self.add_package(packages, self.ui.chk_cnd, 'CND6')
            self.add_package(packages, self.ui.chk_ctp, 'CTP6')
            self.add_package(packages, self.ui.chk_esl, 'ESL6')
            self.add_package(packages, self.ui.chk_est, 'EST6')
            self.add_package(packages, self.ui.chk_fmi_gwe, 'FMI6')
            self.add_package(packages, self.ui.chk_lke, 'LKE6')
            self.add_package(packages, self.ui.chk_mve, 'MVE6')
            self.add_package(packages, self.ui.chk_mwe, 'MWE6')
            self.add_package(packages, self.ui.chk_obs_gwe, 'OBS6')
            self.add_package(packages, self.ui.chk_sfe, 'SFE6')
            self.add_package(packages, self.ui.chk_ssm_gwe, 'SSM6')
            self.add_package(packages, self.ui.chk_uze, 'UZE6')
            self.add_package(packages, self.ui.chk_pobs_gwe, 'POBS6')
            self.data['gwe_ftypes'] = packages

    def _non_unique_name_warning(self) -> str:
        """Return a warning message if the name is not unique among its siblings.

        Returns:
            See description.
        """
        name = self.ui.edt_simulation_name.text()
        name = ''.join(name.split())  # Remove whitespace
        node = tree_util.find_tree_node_by_uuid(self._project_tree, self._new_uuid)
        for child in node.parent.children:
            if child != node and child.name == name:
                if self.model_str:
                    return f'Name "{name}" matches an existing model. Please provide a unique name.'
                else:
                    return f'Name "{name}" matches an existing simulation. Please provide a unique name.'
        return ''

    def _grid_missing_tops_or_bottoms_warning(self) -> str:
        """Return an error message if the grid lacks 'Cell Top Z' or 'Cell Bottom Z' datasets, else return ''.

        Returns:
            See description.
        """
        uuid = self.ui_tree_ugrids.get_selected_item_uuid()
        grid_node = tree_util.find_tree_node_by_uuid(self._project_tree, uuid)
        cell_tops_node = tree_util.first_descendant_with_name(grid_node, 'Cell Top Z')
        cell_bottoms_node = tree_util.first_descendant_with_name(grid_node, 'Cell Bottom Z')
        name = grid_node.name
        msg = ''
        if not cell_tops_node or not cell_bottoms_node:
            msg = f'Grid "{name}" cannot be used because it is missing "Cell Top Z" or "Cell Bottom Z" datasets.'
        return msg

    def _validate(self) -> str:
        """Check for errors and return an error message if there's a problem, or '' if all is OK.

        Returns:
            See description.
        """
        # No UGrid selected
        uuid = self.ui_tree_ugrids.get_selected_item_uuid()
        if not uuid:
            return 'Please select a UGrid to create a new simulation.'

        # Name is the same as a sibling
        msg = self._non_unique_name_warning()
        if msg:
            return msg

        # UGrid missing cell tops or bottoms datasets
        msg = self._grid_missing_tops_or_bottoms_warning()
        if msg:
            return msg

        return ''

    def accept(self):
        """Called when the OK button is clicked. Makes sure a Ugrid is selected."""
        msg = self._validate()
        if msg:
            message_box.message_with_ok(parent=self, message=msg)
            return

        self.update_data()
        super().accept()


def _find_good_tree_name(node: TreeNode, starting_name: str) -> str:
    """Return a tree name that is unique amongst it's siblings, e.g. 'flow_2', 'sim_2' etc.

    Args:
        node: Tree node of the node whose name we are dealing with.
        starting_name: The name we build on.

    Returns:
        See description.
    """
    if not node:
        return ''
    new_pattern = f'{starting_name}_$*$'
    candidate = starting_name
    i = 2
    done = False
    while not done:
        done = True
        for child in node.parent.children:
            if child == node:
                continue
            if child.name == candidate:
                candidate = new_pattern.replace('_$*$', f'_{i}')
                i += 1
                done = False
                break
    return candidate
