"""Settings class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules
import orjson

# 3. Aquaveo modules
from xms.guipy import file_io_util

# 4. Local modules
from xms.mf6.misc import log_util, util


class Settings:
    """Class to handle settings files, which are json files saved next to a main_file."""
    def __init__(self):
        """Initializes the class."""

    @classmethod
    def settings_filename(cls, main_file: Path | str, prefix: str = 'settings') -> str:
        """Returns the filepath to the settings file.

        Args:
            main_file: mfsim.nam file path.
            prefix: Prefix part of settings file name.

        Returns:
            See description.
        """
        if util.null_path(main_file):
            return ''
        return str(Path(main_file).parent / f'{prefix}.json')

    @classmethod
    def read_settings(cls, main_file: Path | str, prefix: str = 'settings') -> dict:
        """Returns settings dict from disk or the default if not found.

        Args:
            main_file: main_file file path.
            prefix: Prefix part of settings file name.

        Returns:
            (dict[str, obj]): See description.
        """
        settings_filename = Settings.settings_filename(main_file, prefix)
        settings = {}
        if os.path.isfile(settings_filename):
            with open(settings_filename, 'rb') as file:
                try:
                    settings = orjson.loads(file.read())
                except orjson.JSONDecodeError as error:
                    logger = log_util.get_logger()
                    logger.error(str(error))
        return settings

    @classmethod
    def write_settings(cls, main_file: Path | str, settings: dict, prefix: str = 'settings'):
        """Writes the settings dict to disk.

        Args:
            main_file: main_file file path.
            settings (dict[str, obj]): Dict of settings.
            prefix: Prefix part of settings file name.
        """
        settings_filename = Settings.settings_filename(main_file, prefix)
        if not settings_filename:
            return
        file_io_util.write_json_file(settings, settings_filename)

    @classmethod
    def get(cls, main_file: Path | str, variable: str, default=None, prefix: str = 'settings'):
        """Returns the variable or None if not found.

        Args:
            main_file: main_file file path.
            variable: Name of the variable to get the value of.
            default: Default value to return if the variable is not found.
            prefix: Prefix part of settings file name.

        Returns:
            See description.
        """
        settings = cls.read_settings(main_file, prefix)
        value = default
        if variable in settings:
            value = settings[variable]
        return value

    @classmethod
    def set(cls, main_file: Path | str, variable: str, value, prefix: str = 'settings'):
        """Sets the last used directory.

        Args:
            main_file: main_file file path.
            variable: Name of the variable to get the value of.
            value: The value of the variable to set.
            prefix: Prefix part of settings file name.
        """
        settings = cls.read_settings(main_file, prefix)
        settings[variable] = value
        cls.write_settings(main_file, settings, prefix)
