"""SimRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import Enum
import os
import subprocess
import sys

# 2. Third party modules
from mf6_mdt_exe import get_executable_paths as get_executable_paths_mdt
from PySide2.QtWidgets import QWidget
from typing_extensions import override

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.api.tree import tree_util, TreeNode
from xms.components.bases.run_base import RunBase
from xms.core.filesystem import filesystem as fs
from xms.executables.modflow6 import get_executable_paths as get_executable_paths_mf6
from xms.executables.modflow6_samg import get_executable_paths as get_executable_paths_samg
from xms.executables.pest_gms import get_executable_paths as get_executable_paths_pest
from xms.guipy.dialogs import message_box
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.mf6.components import dmi_util
from xms.mf6.data import data_util
from xms.mf6.file_io import mfsim_reader, solution_reader
from xms.mf6.file_io.gwf import ims_reader

# Type aliases
BackupDispOptsList = list[str]  # ['{map cov uuid} {obs_category} "{filepath}"']


class ExeKey(str, Enum):  # StrEnum is not available in Python 3.10
    """Executable keyword text as found in the modflow6.xml file."""
    MODFLOW_6 = 'MODFLOW 6'
    ZONEBUDGET_6 = 'ZONEBUDGET 6'
    MODFLOW_6_SAMG = 'MODFLOW 6 (SAMG)'
    MODFLOW_6_MDT = 'MODFLOW 6 (MDT)'
    PEST_FOR_MODFLOW_6 = 'PEST for MODFLOW 6'


class ExeFile(str, Enum):
    """Executable file names."""
    MF6 = 'mf6.exe'
    ZBUD6 = 'zbud6.exe'
    MF6_SAMG = 'mf6.exe'
    MF6_MDT = 'mf6_mdt.exe'
    PEST = 'pest.exe'


def _need_samg_exe(sim_node):
    """Returns True if SAMG is on in the IMS package; otherwise returns False.

    Args:
        sim_node (TreeNode): a Query object to communicate with GMS.

    Returns:
        (bool): See description.
    """
    ims_mainfile = dmi_util.package_mainfile_from_sim_node(sim_node, 'IMS6')
    if ims_mainfile:
        samg_setting = ims_reader.samg_setting(ims_mainfile)
        return samg_setting
    return False


def _need_mdt_exe(sim_node):
    """Returns True if a MDT package is defined; otherwise returns False.

    Args:
        sim_node (TreeNode): a Query object to communicate with GMS.

    Returns:
        (bool): See description.
    """
    mdt_mainfile = dmi_util.package_mainfile_from_sim_node(sim_node, 'MDT6')
    return bool(mdt_mainfile)


def get_modflow_exe_string(sim_node):
    """Returns the string associated with the executable.

    Args:
        sim_node (TreeNode): a Query object to communicate with GMS.

    Returns:
        (str):'MODFLOW 6 (SAMG)', 'MODFLOW 6 (MDT)', or 'MODFLOW 6'
    """
    if _need_samg_exe(sim_node):
        return ExeKey.MODFLOW_6_SAMG
    elif _need_mdt_exe(sim_node):
        return ExeKey.MODFLOW_6_MDT
    else:
        return ExeKey.MODFLOW_6


def _get_installed_exe_path(exe_key: str):
    """Searches for the installed MODFLOW 6 executable.

    Args:
        exe_key: Defines which executable to get.

    Returns:
        (str): The path to the executable.
    """
    match exe_key:
        case ExeKey.MODFLOW_6:
            paths = get_executable_paths_mf6()
            exe_filename = ExeFile.MF6
        case ExeKey.ZONEBUDGET_6:
            paths = get_executable_paths_mf6()
            exe_filename = ExeFile.ZBUD6
        case ExeKey.MODFLOW_6_SAMG:
            paths = get_executable_paths_samg()
            exe_filename = ExeFile.MF6_SAMG
        case ExeKey.MODFLOW_6_MDT:
            paths = get_executable_paths_mdt()
            exe_filename = ExeFile.MF6_MDT
        case ExeKey.PEST_FOR_MODFLOW_6:
            paths = get_executable_paths_pest()
            exe_filename = ExeFile.PEST
        case _:
            raise RuntimeError('Executable file not recognized')

    for path in paths:
        if path.lower().endswith(exe_filename):
            return path
    return ''


def _add_exe_path(settings_manager, package, key: str, exes_dict):
    """Adds the executable path to the exes_dict."""
    val = settings_manager.get_setting(package, f'MODFLOW 6 - {key}', '')
    if val == '' and XmEnv.xms_environ_running_tests() == 'TRUE':
        val = _get_installed_exe_path(key)
    exes_dict[key] = os.path.abspath(val)


def _exe_key_dict() -> dict[ExeKey, ExeFile]:
    """Returns a dict of the exe keys and the exe file names."""
    return {
        ExeKey.MODFLOW_6: ExeFile.MF6,
        ExeKey.ZONEBUDGET_6: ExeFile.ZBUD6,
        ExeKey.MODFLOW_6_SAMG: ExeFile.MF6_SAMG,
        ExeKey.MODFLOW_6_MDT: ExeFile.MF6_MDT,
        ExeKey.PEST_FOR_MODFLOW_6: ExeFile.PEST,
    }


def get_gms_mf6_executable_paths():
    """Returns the path to the MODFLOW 6 executables as specified by the GMS preferences.

    If testing and we can't get the path from the GMS preferences, we look for the installed executables.

    Returns:
        (dict[str, str]): See description.
    """
    exes = {}
    settings = SettingsManager(python_path=False)
    package = 'Model Executables 64 bit'
    for exe_key in ExeKey:
        _add_exe_path(settings, package, exe_key, exes)
    return exes


def _output_dir_from_model_filename(filename):
    basename = os.path.basename(os.path.splitext(filename)[0])
    output_dir = f'{basename}_output'
    return output_dir


def _get_model_nodes(sim_node: TreeNode) -> list[TreeNode]:
    """Returns a list of the model uuids.

    Args:
        sim_node: Simulation tree node.
        query: a Query object to communicate with GMS.

    Returns:
        (list[str]): See description
    """
    model_nodes = []
    xms_types = ['TI_COMPONENT']
    for ftype in data_util.model_ftypes():
        items = tree_util.descendants_of_type(sim_node, xms_types=xms_types, unique_name=ftype, model_name='MODFLOW 6')
        model_nodes.extend(items)
    return model_nodes


def _get_model_uuids(model_nodes: list[TreeNode], mnames: list[str]) -> list[str]:
    """Returns a list of the model uuids.

    Args:
        model_nodes: Model tree nodes.
        mnames (list[str]): List of model mnames.

    Returns:
        (list[str]): See description
    """
    model_uuids = [''] * len(mnames)
    for i, mname in enumerate(mnames):
        for node in model_nodes:
            if mname == node.name:
                model_uuids[i] = node.uuid
                break
    return model_uuids


def _clear_output_folder(filelocation):
    """Clears the output folder in anticipation of a model run.

    Args:
        filelocation (str): The location of input files for the simulation.
    """
    mfsim_nam = os.path.join(filelocation, 'mfsim.nam')
    model_filenames, _, _ = mfsim_reader.model_name_files_from_mfsim_nam_file(mfsim_nam)
    for filename in model_filenames:
        output_dir = _output_dir_from_model_filename(filename)
        fs.make_or_clear_dir(os.path.join(filelocation, output_dir))


class SimRunner(RunBase):
    """Class that handles running MODFLOW."""
    def __init__(self, dummy_mainfile=''):
        """Initializes the class.

        Args:
            dummy_mainfile (str): Unused. Just to keep constructor consistent with component classes.
        """
        # del dummy_mainfile  # Unused parameter
        super().__init__()
        self.pest_coverages = None  # This is a member only to help us with testing

    def read_solution(self, query, params, win_cont: QWidget):
        """Reads the MODFLOW Solution.

        Args:
            query: a Query object to communicate with GMS.
            params: Generic map of parameters. Contains the structures for various components that
             are required for adding vertices to the Query Context with Add().

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        rv = solution_reader.read(query, params, win_cont)
        return rv

    @override
    def get_executables(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (data_objects.parameters.Simulation): The Simulation you want to load the solution for.
            query: a Query object to communicate with GMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (list of xmsapi.dmi.ExecutableCommand):
                The executable objects to run and the action requests that go with it.
        """
        mfsim_nam = os.path.join(filelocation, 'mfsim.nam')
        if not os.path.isfile(mfsim_nam):
            message_box.message_with_ok(
                parent=None,
                message=f'No simulation found at "{mfsim_nam}". Have you saved the simulation?',
            )
            return []

        _clear_output_folder(filelocation)

        script_path = os.path.normpath(__file__)

        sim_uuid = query.current_item_uuid()
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
        exe_string = get_modflow_exe_string(sim_node)

        cmd = ExecutableCommand(
            executable=script_path,
            model='MODFLOW 6',
            executable_order=0,
            display_name=exe_string,
            run_weight=100,
            progress_script='xml_entry_points/sim_progress.py',
            executable_is_script=True
        )
        exe_file = get_gms_mf6_executable_paths()[exe_string]
        cmd.add_commandline_arg(exe_file)

        load_sol = self.get_solution_load_actions(sim, query, filelocation)
        cmd.add_solution_file(load_sol[0])
        return [cmd]

    @override
    def get_solution_load_actions(self, sim, query, filelocation: str):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (data_objects.parameters.Simulation): The Simulation you want to load the solution for.
            query: a Query object to communicate with GMS.
            filelocation: The location of input files for the simulation.

        Returns:
            (list of xmsapi.dmi.ExecutableCommand):
                The executable objects to run and the action requests that go with it.
        """
        mfsim_nam = os.path.join(filelocation, 'mfsim.nam')
        if not os.path.isfile(mfsim_nam):
            message_box.message_with_ok(parent=None, message=f'Could not find the simulation at "{mfsim_nam}".')
            return []

        model_filenames, mnames, model_ftypes = mfsim_reader.model_name_files_from_mfsim_nam_file(mfsim_nam)
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        model_nodes = _get_model_nodes(sim_node)
        model_uuids = _get_model_uuids(model_nodes, mnames)

        load_sol = ActionRequest(
            main_file=os.path.join(filelocation, 'mfsim.nam'),
            modality='MODAL',
            class_name=self.__class__.__name__,
            module_name=self.__module__,
            method_name='read_solution',
            parameters={
                'model_files': model_filenames,
                'model_names': mnames,
                'model_ftypes': model_ftypes,
                'model_uuids': model_uuids,
                'run_dir': filelocation,
            }
        )
        return [load_sol]


def launch_mf6_executable(executable) -> int:
    """Run executable and look for failure.

    Args:
        executable: The path to the executable.

    Returns:
        The exit code for the process.
    """
    output_file = XmEnv.xms_environ_stdout_file()
    with open(output_file, 'wb') as f:
        process = subprocess.Popen(executable, stdout=f, stderr=f)
        process_exit_code = process.wait()

    # Check if "solver is not licensed" is found in the last part of the output file
    if os.path.isfile(output_file):
        with open(output_file, 'rb') as f:
            f.seek(0, os.SEEK_END)
            file_size = f.tell()
            seek_offset = min(file_size, 256)
            f.seek(-seek_offset, os.SEEK_END)
            last_of_output = f.read(seek_offset).decode('utf-8', errors='ignore')
        if "solver is not licensed" in last_of_output:
            process_exit_code = -1

    return process_exit_code


if __name__ == "__main__":  # pragma no cover - can't run this from the tests, and it must be here
    exit_code = launch_mf6_executable(sys.argv[1])
    exit(exit_code)
