"""Testing utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pytest

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.testing.test_util import read_test_sims


def _read(test_dir: Path, tmp_path: Path) -> list[MfsimData]:
    """Helper func so stuff will fit on one line."""
    return read_test_sims(test_dir, '', copy=True, tmp_path=tmp_path)


@pytest.fixture()
def ates_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwe-ates', tmp_path)[0]


@pytest.fixture()
def barends_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwe-barends', tmp_path)[0]


@pytest.fixture()
def case7_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Test fixture that makes a copy of the whole sim folder, reads the whole sim, and returns the sim."""
    return _read(test_files / 'test_sims' / 'gms_projects' / 'swi_case7' / 'case7_data' / 'Components', tmp_path)[0]


@pytest.fixture()
def csub_p04_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwf-csub-p04', tmp_path)[0]


@pytest.fixture()
def danckwerts_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwe-danckwerts', tmp_path)[0]


@pytest.fixture()
def moc3d_p02tg_mfsim(test_files: Path, tmp_path: Path, chklog) -> list[MfsimData]:
    """Test fixture that makes a copy of the whole sim folder, reads the whole sim, and returns the sim.

    - Steady state
    - DISV
    - OBS (in GWT sim)
    - WEL (in GWF sim)
    """
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwt-moc3d-p02tg', tmp_path)


@pytest.fixture()
def olele_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Test fixture that makes a copy of the whole sim folder, reads the whole sim, and returns the sim."""
    return _read(test_files / 'tools' / 'swi_tools' / 'olele' / 'olele_data' / 'Components', tmp_path)[0]


@pytest.fixture()
def prudic_mfsims(test_files: Path, tmp_path: Path, chklog) -> list[MfsimData]:
    """Copy tree, read, and return the sims."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwt-prudic2004t2', tmp_path)


@pytest.fixture()
def saltlake_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwt-saltlake', tmp_path)[0]


@pytest.fixture()
def spbc_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwf-spbc', tmp_path)[0]


@pytest.fixture()
def ss_dis_hfb_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ss_dis_hfb', tmp_path)[0]


@pytest.fixture()
def trans_dis_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'trans_dis', tmp_path)[0]


@pytest.fixture(scope='session')
def trans_dis_mfsim_session(test_files: Path, tmp_path_factory: pytest.TempPathFactory) -> MfsimData:
    """Copy tree, read, and return the sim."""
    temp_sim_dir = tmp_path_factory.mktemp("sim")
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'trans_dis', temp_sim_dir)[0]


@pytest.fixture()
def vsc_mfsim(test_files: Path, tmp_path: Path, chklog) -> MfsimData:
    """Copy tree, read, and return the sim."""
    return _read(test_files / 'test_sims' / 'usgs_examples' / 'ex-gwe-vsc', tmp_path)[0]
