"""Module for reading and writing model control."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['read_control', 'write_control']

# 1. Standard Python modules
from collections import namedtuple
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Section
from xms.guipy.time_format import datetime_to_string, string_to_datetime
from xms.ptmio.pcf.control_reader import read_control as ptm_io_read
from xms.ptmio.pcf.control_writer import write_control as ptm_io_write
from xms.ptmio.pcf.program_control import (
    AdvectionMethod, CentroidMethod, EulerianMethod, EulerianTransportMethod, FlowFormat, MeshFormat, NumericalScheme,
    ProgramControl, SedimentFormat, VelocityMethod
)

# 4. Local modules
from xms.ptm.model.model import simulation_model, StopMode

MappingTuple = namedtuple('MappingTuple', ['control', 'model'])
numerical_scheme_mapping = [
    MappingTuple(NumericalScheme.two, '2'),
    MappingTuple(NumericalScheme.four, '4'),
]
eulerian_transport_method_mapping = [
    MappingTuple(EulerianTransportMethod.soulsby_van_rijn, 'Soulsby-Van Rijn'),
    MappingTuple(EulerianTransportMethod.van_rijn, 'Van Rijn'),
    MappingTuple(EulerianTransportMethod.lund, 'Lund'),
    MappingTuple(EulerianTransportMethod.camenen_larson, 'Camenen Larson'),
]

velocity_method_mapping = [
    MappingTuple(VelocityMethod.two_d_log, '2D Logarithmic'),
    MappingTuple(VelocityMethod.two_d_two_point, '2D Two-Point'),
    MappingTuple(VelocityMethod.two_d_uniform, '2D Uniform'),
    MappingTuple(VelocityMethod.three_d_sigma, '3DS'),
    MappingTuple(VelocityMethod.three_d_z, '3DZ'),
]

eulerian_method_mapping = [
    MappingTuple(EulerianMethod.ptm, 'PTM'),
    MappingTuple(EulerianMethod.van_rijn, 'Van Rijn'),
]

centroid_method_mapping = [
    MappingTuple(CentroidMethod.rouse, 'Rouse'),
    MappingTuple(CentroidMethod.van_rijn, 'Van Rijn'),
]

advection_method_mapping = [
    MappingTuple(AdvectionMethod.one_d, '1D'),
    MappingTuple(AdvectionMethod.two_d, '2D'),
    MappingTuple(AdvectionMethod.three_d, '3D'),
    MappingTuple(AdvectionMethod.q_three_d, 'Q3D'),
]

mesh_format_mapping = [
    MappingTuple(MeshFormat.adcirc, 'ADCIRC'),
    MappingTuple(MeshFormat.cms_2d, 'CMS-Flow 2D'),
]

sediment_format_mapping = [
    MappingTuple(SedimentFormat.adcirc, 'ADCIRC'),
    MappingTuple(SedimentFormat.m2d, 'M2D'),
    MappingTuple(SedimentFormat.xmdf_dataset, 'XMDF Dataset'),
    MappingTuple(SedimentFormat.uniform, 'Uniform'),
]

flow_format_mapping = [
    MappingTuple(FlowFormat.adcirc_ascii, 'ADCIRC ASCII'),
    MappingTuple(FlowFormat.adcirc_xmdf, 'ADCIRC XMDF'),
    MappingTuple(FlowFormat.cmsflow_2d_single, 'CMS-Flow 2D Single'),
    MappingTuple(FlowFormat.cmsflow_2d_multi, 'CMS-Flow 2D Multi'),
    MappingTuple(FlowFormat.adh, 'ADH'),
]


def read_control(where: str | Path | TextIO) -> Section:
    """
    Read a model control from a file.

    Args:
        where: Where to read from.

    Returns:
        The read model control.
    """
    control = ptm_io_read(where)

    section = simulation_model()

    group = section.group('time')
    group.parameter('duration').value = control.duration
    group.parameter('flow_update').value = control.flow_update
    group.parameter('grid_update').value = control.grid_update
    group.parameter('last_step_trap').value = control.last_step_trap
    group.parameter('mapping_inc').value = control.mapping_inc
    group.parameter('output_inc').value = control.output_inc
    if control.start_flow is not None:
        group.parameter('start_flow').value = datetime_to_string(control.start_flow)
        group.parameter('have_start_flow').value = True
    else:
        group.parameter('have_start_flow').value = False
    group.parameter('start_run').value = datetime_to_string(control.start_run)
    group.parameter('start_trap').value = datetime_to_string(control.start_trap)
    group.parameter('start_waves').value = datetime_to_string(control.start_waves)
    if control.stop_run is not None:
        group.parameter('stop_run').value = datetime_to_string(control.stop_run)
        group.parameter('stop_mode').value = StopMode.date_time
    else:
        group.parameter('stop_mode').value = StopMode.duration
    group.parameter('stop_trap').value = datetime_to_string(control.stop_trap)
    group.parameter('time_step').value = control.time_step
    group.parameter('wave_step').value = control.wave_step

    group = section.group('files')
    group.parameter('bc_file').value = control.bc_file
    group.parameter('flow_file_xmdf').value = control.flow_file_xmdf
    group.parameter('flow_format').value = _file_to_model(control.flow_format, flow_format_mapping)
    group.parameter('mesh_file').value = control.mesh_file
    group.parameter('mesh_format').value = _file_to_model(control.mesh_format, mesh_format_mapping)
    group.parameter('neighbor_file').value = control.neighbor_file
    group.parameter('output_prefix').value = control.output_prefix
    group.parameter('sediment_file').value = control.sediment_file
    group.parameter('sediment_format').value = _file_to_model(control.sediment_format, sediment_format_mapping)
    group.parameter('source_file').value = control.source_file
    group.parameter('trap_file').value = control.trap_file
    group.parameter('xmdf_vel_path').value = control.xmdf_vel_path
    group.parameter('xmdf_wse_path').value = control.xmdf_wse_path

    group = section.group('computations')
    group.parameter('advection_method').value = _file_to_model(control.advection_method, advection_method_mapping)
    group.parameter('bed_interaction').value = control.bed_interaction
    group.parameter('bed_porosity').value = control.bed_porosity
    group.parameter('bedforms').value = control.bedforms
    group.parameter('centroid_method').value = _file_to_model(control.centroid_method, centroid_method_mapping)
    group.parameter('currents').value = control.currents
    if control.by_weight:
        group.parameter('distribution').value = 'By weight'
    else:
        group.parameter('distribution').value = 'By grain size'
    group.parameter('etmin').value = control.etmin
    group.parameter('eulerian_method').value = _file_to_model(control.eulerian_method, eulerian_method_mapping)
    group.parameter('eulerian_transport_method').value = _file_to_model(
        control.eulerian_transport_method, eulerian_transport_method_mapping
    )
    group.parameter('evmin').value = control.evmin
    group.parameter('hiding_exposure').value = control.hiding_exposure
    group.parameter('ket').value = control.ket
    group.parameter('kev').value = control.kev
    group.parameter('kew').value = control.kew
    group.parameter('min_depth').value = control.min_depth
    group.parameter('morphology').value = control.morphology
    group.parameter('neutrally_buoyant').value = control.neutrally_buoyant
    group.parameter('numerical_scheme').value = _file_to_model(control.numerical_scheme, numerical_scheme_mapping)
    group.parameter('residence_calc').value = control.residence_calc
    group.parameter('rhos').value = control.rhos
    group.parameter('salinity').value = control.salinity
    group.parameter('source_to_datum').value = control.source_to_datum
    group.parameter('temperature').value = control.temperature
    group.parameter('turbulent_shear').value = control.turbulent_shear
    group.parameter('velocity_method').value = _file_to_model(control.velocity_method, velocity_method_mapping)
    group.parameter('wave_mass_transport').value = control.wave_mass_transport

    group = section.group('output')
    group.parameter('bed_level_change_mapping').value = control.bed_level_change_mapping
    group.parameter('bed_level_mapping').value = control.bed_level_mapping
    group.parameter('bedform_mapping').value = control.bedform_mapping
    group.parameter('density_output').value = control.density_output
    group.parameter('fall_velocity_output').value = control.fall_velocity_output
    group.parameter('flow_mapping').value = control.flow_mapping
    group.parameter('flow_output').value = control.flow_output
    group.parameter('grain_size_output').value = control.grain_size_output
    group.parameter('height_output').value = control.height_output
    group.parameter('mobility_mapping').value = control.mobility_mapping
    group.parameter('mobility_output').value = control.mobility_output
    group.parameter('parcel_mass_output').value = control.parcel_mass_output
    group.parameter('population_record').value = control.population_record
    group.parameter('shear_stress_mapping').value = control.shear_stress_mapping
    group.parameter('source_output').value = control.source_output
    group.parameter('state_output').value = control.state_output
    group.parameter('tau_cr_output').value = control.tau_cr_output
    group.parameter('tecplot_maps').value = control.tecplot_maps
    group.parameter('tecplot_parcels').value = control.tecplot_parcels
    group.parameter('paths').value = control.paths
    group.parameter('transport_mapping').value = control.transport_mapping
    group.parameter('wave_mapping').value = control.wave_mapping
    group.parameter('xmdf_compressed').value = control.xmdf_compressed

    return section


def write_control(section: Section, where: str | Path | TextIO):
    """
    Write a model control to a file.

    Args:
        section: The model control section to write.
        where: Where to write it.
    """
    control = ProgramControl()

    group = section.group('time')
    control.flow_update = group.parameter('flow_update').value
    control.grid_update = group.parameter('grid_update').value
    control.last_step_trap = group.parameter('last_step_trap').value
    control.mapping_inc = group.parameter('mapping_inc').value
    control.output_inc = group.parameter('output_inc').value
    control.start_run = string_to_datetime(group.parameter('start_run').value)
    control.start_trap = string_to_datetime(group.parameter('start_trap').value)
    control.start_waves = string_to_datetime(group.parameter('start_waves').value)
    control.stop_trap = string_to_datetime(group.parameter('stop_trap').value)
    control.time_step = group.parameter('time_step').value
    control.wave_step = group.parameter('wave_step').value
    if group.parameter('stop_mode').value == StopMode.duration:
        control.duration = group.parameter('duration').value
    else:
        control.stop_run = string_to_datetime(group.parameter('stop_run').value)
    if group.parameter('have_start_flow').value is True:
        control.start_flow = string_to_datetime(group.parameter('start_flow').value)
    else:
        control.start_flow = None

    group = section.group('files')
    control.bc_file = group.parameter('bc_file').value
    control.flow_file_xmdf = group.parameter('flow_file_xmdf').value
    control.flow_format = _model_to_file(group.parameter('flow_format').value, flow_format_mapping)
    control.mesh_file = group.parameter('mesh_file').value
    control.mesh_format = _model_to_file(group.parameter('mesh_format').value, mesh_format_mapping)
    control.neighbor_file = group.parameter('neighbor_file').value
    control.output_prefix = group.parameter('output_prefix').value
    control.sediment_file = group.parameter('sediment_file').value
    control.sediment_format = _model_to_file(group.parameter('sediment_format').value, sediment_format_mapping)
    control.source_file = group.parameter('source_file').value
    control.trap_file = group.parameter('trap_file').value
    control.xmdf_vel_path = group.parameter('xmdf_vel_path').value
    control.xmdf_wse_path = group.parameter('xmdf_wse_path').value

    group = section.group('computations')
    control.advection_method = _model_to_file(group.parameter('advection_method').value, advection_method_mapping)
    control.bed_interaction = group.parameter('bed_interaction').value
    control.bed_porosity = group.parameter('bed_porosity').value
    control.bedforms = group.parameter('bedforms').value
    control.centroid_method = _model_to_file(group.parameter('centroid_method').value, centroid_method_mapping)
    control.currents = group.parameter('currents').value
    control.etmin = group.parameter('etmin').value
    control.eulerian_method = _model_to_file(group.parameter('eulerian_method').value, eulerian_method_mapping)
    control.eulerian_transport_method = _model_to_file(
        group.parameter('eulerian_transport_method').value, eulerian_transport_method_mapping
    )
    control.evmin = group.parameter('evmin').value
    control.hiding_exposure = group.parameter('hiding_exposure').value
    control.ket = group.parameter('ket').value
    control.kev = group.parameter('kev').value
    control.kew = group.parameter('kew').value
    control.min_depth = group.parameter('min_depth').value
    control.morphology = group.parameter('morphology').value
    control.neutrally_buoyant = group.parameter('neutrally_buoyant').value
    control.numerical_scheme = _model_to_file(group.parameter('numerical_scheme').value, numerical_scheme_mapping)
    control.residence_calc = group.parameter('residence_calc').value
    control.rhos = group.parameter('rhos').value
    control.salinity = group.parameter('salinity').value
    control.source_to_datum = group.parameter('source_to_datum').value
    control.temperature = group.parameter('temperature').value
    control.turbulent_shear = group.parameter('turbulent_shear').value
    control.velocity_method = _model_to_file(group.parameter('velocity_method').value, velocity_method_mapping)
    control.wave_mass_transport = group.parameter('wave_mass_transport').value
    control.by_weight = group.parameter('distribution').value == 'By weight'

    group = section.group('output')
    control.bed_level_change_mapping = group.parameter('bed_level_change_mapping').value
    control.bed_level_mapping = group.parameter('bed_level_mapping').value
    control.bedform_mapping = group.parameter('bedform_mapping').value
    control.density_output = group.parameter('density_output').value
    control.fall_velocity_output = group.parameter('fall_velocity_output').value
    control.flow_mapping = group.parameter('flow_mapping').value
    control.flow_output = group.parameter('flow_output').value
    control.grain_size_output = group.parameter('grain_size_output').value
    control.height_output = group.parameter('height_output').value
    control.mobility_mapping = group.parameter('mobility_mapping').value
    control.mobility_output = group.parameter('mobility_output').value
    control.parcel_mass_output = group.parameter('parcel_mass_output').value
    control.population_record = group.parameter('population_record').value
    control.shear_stress_mapping = group.parameter('shear_stress_mapping').value
    control.source_output = group.parameter('source_output').value
    control.state_output = group.parameter('state_output').value
    control.tau_cr_output = group.parameter('tau_cr_output').value
    control.tecplot_maps = group.parameter('tecplot_maps').value
    control.tecplot_parcels = group.parameter('tecplot_parcels').value
    control.paths = group.parameter('paths').value
    control.transport_mapping = group.parameter('transport_mapping').value
    control.wave_mapping = group.parameter('wave_mapping').value
    control.xmdf_compressed = group.parameter('xmdf_compressed').value

    ptm_io_write(control, where)


def _file_to_model(value, mapping: list[MappingTuple]):
    """Convert a file value to its model value using the provided mapping."""
    for item in mapping:
        if item.control == value:
            return item.model
    raise AssertionError('Unsupported value.')


def _model_to_file(value, mapping: list[MappingTuple]):
    """Convert a model value to its file value using the provided mapping."""
    for item in mapping:
        if item.model == value:
            return item.control
    raise AssertionError('Unsupported value.')
