"""Writer for fort.14 geometry files."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Sequence, TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.grid.ugrid import UGrid

# 4. Local modules
from .geometry import Boundary, Geometry


def write_geometry(geometry: Geometry, path: Path | str):
    """
    Write a fort14 geometry file to disk.

    Args:
        geometry: The geometry to write.
        path: Where to write it.
    """
    with open(path, 'w') as f:
        _write_header(geometry.ugrid, f)
        _write_points(geometry.ugrid, f)
        _write_connectivity(geometry.ugrid, f)
        _write_open_boundaries(geometry.open_boundaries, f)
        _write_closed_boundaries(geometry.closed_boundaries, f)


def _write_header(ugrid: UGrid, file: TextIO):
    """
    Write the header section of the file.

    Args:
        ugrid: Domain to write the header for.
        file: Where to write it to.
    """
    file.write('ADCIRC boundary conditions file\n')
    file.write(f'{ugrid.cell_count} {ugrid.point_count}\n')


def _write_points(ugrid: UGrid, file: TextIO):
    """
    Write the location section of the file.

    Args:
        ugrid: Domain to write the locations for.
        file: Where to write it to.
    """
    for index, location in enumerate(ugrid.locations, start=1):
        file.write(f'{index} {location[0]} {location[1]} {location[2]}\n')


def _write_connectivity(ugrid: UGrid, file: TextIO):
    """
    Write the connectivity section of the file.

    Args:
        ugrid: Domain to write connectivity for.
        file: Where to write it to.
    """
    for cell_index in range(ugrid.cell_count):
        cell_points = ugrid.get_cell_points(cell_index)
        points = ' '.join([str(p + 1) for p in cell_points])
        file.write(f'{cell_index + 1} {len(cell_points)} {points}\n')


def _write_open_boundaries(boundaries: Sequence[Boundary], file: TextIO):
    """
    Write the open boundaries part of the file.

    Args:
        boundaries: Open boundaries to write.
        file: Where to write them to.
    """
    num_nodes = sum(len(arc.nodes) for arc in boundaries)
    file.write(f'{len(boundaries)} ! Number of open boundaries\n')
    file.write(f'{num_nodes} ! Total number of open boundary nodes\n')
    for number, boundary in enumerate(boundaries, start=1):
        file.write(f'{len(boundary.nodes)} ! Number of nodes for open boundary {number}\n')
        for node in boundary.nodes:
            file.write(f'{node}\n')


def _write_closed_boundaries(boundaries: Sequence[Boundary], file: TextIO):
    """
    Write the closed boundaries section of the file.

    Args:
        boundaries: Closed boundaries to write.
        file: Where to write them to.
    """
    num_nodes = sum(len(arc.nodes) for arc in boundaries)
    file.write(f'{len(boundaries)} ! Number of land boundaries\n')
    file.write(f'{num_nodes} ! Total number of land boundary nodes\n')
    for number, boundary in enumerate(boundaries, start=1):
        file.write(
            f'{len(boundary.nodes)} {int(boundary.boundary_type)} ! Number of nodes for land boundary {number}\n'
        )
        for node in boundary.nodes:
            file.write(f'{node}\n')
