"""Module for the trap reader."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['read_traps']

# 1. Standard Python modules
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.ptmio.file_reader import FileReader
from xms.ptmio.trap.traps import Direction, LineTrap, PolygonTrap, Traps


def read_traps(where: str | Path | TextIO) -> Traps:
    """
    Read the traps out of a file.

    On failure, raises xms.ptmio.file_reader.ParseError.

    Args:
        where: Where to read traps from.

    Returns:
        The traps read from the file.
    """
    with FileReader(where) as reader:
        traps = _read_traps(reader)
    return traps


def _read_traps(reader: FileReader):
    """Read all the traps."""
    traps = Traps()

    num_line_traps = reader.read_int()
    reader.next_line()
    for _ in range(num_line_traps):
        trap = _read_line_trap(reader)
        traps.line_traps.append(trap)

    num_polygon_traps = reader.read_int()
    reader.next_line()
    for _ in range(num_polygon_traps):
        trap = _read_polygon_trap(reader)
        traps.polygon_traps.append(trap)

    return traps


def _read_line_trap(reader: FileReader) -> LineTrap:
    """Read a single line trap."""
    trap_id = reader.read_int()
    trap_label = reader.read_remainder()
    reader.next_line()

    direction = reader.read_int()
    if direction == -1:
        direction = Direction.decreasing_x
    elif direction == 0:
        direction = Direction.any
    elif direction == 1:
        direction = Direction.increasing_x
    else:
        raise reader.error('Direction must be -1, 0, or 1.')

    bottom, top = reader.read_float(count=2)

    is_open = reader.read_int()
    if is_open == 0:
        is_open = False
    elif is_open == 1:
        is_open = True
    else:
        raise reader.error('Open flag must be 0 or 1.')

    single = reader.read_int()
    if single == 0:
        single = False
    elif single == 1:
        single = True
    else:
        raise reader.error('Single flag must be 0 or 1.')
    reader.next_line()  # Has to come after error checking so the reader has the right field for reporting.

    x1, y1 = reader.read_float(count=2)
    reader.next_line()
    x2, y2 = reader.read_float(count=2)
    reader.next_line()

    trap = LineTrap(
        trap_id=trap_id,
        label=trap_label,
        direction=direction,
        bottom=bottom,
        top=top,
        is_open=is_open,
        is_single=single,
        start=(x1, y1),
        end=(x2, y2)
    )
    return trap


def _read_polygon_trap(reader: FileReader):
    """Read a single polygon trap."""
    trap_id = reader.read_int()
    trap_label = reader.read_remainder()
    reader.next_line()

    num_points = reader.read_int()
    bottom, top = reader.read_float(count=2)

    is_open = reader.read_int()
    if is_open == 0:
        is_open = False
    elif is_open == 1:
        is_open = True
    else:
        raise reader.error('Open flag must be 0 or 1.')

    single = reader.read_int()
    if single == 0:
        single = False
    elif single == 1:
        single = True
    else:
        raise reader.error('Single flag must be 0 or 1.')
    reader.next_line()  # Has to come after error checking so the reader has the right field for reporting.

    points = []
    for _ in range(num_points):
        reader.read_int()
        x, y = reader.read_float(count=2)
        reader.next_line()
        points.append((x, y))

    trap = PolygonTrap(
        trap_id=trap_id,
        label=trap_label,
        bottom=bottom,
        top=top,
        is_open=is_open,
        is_single=single,
        points=points,
    )
    return trap


# def _write_polygon_traps(traps: list[PolygonTrap], where: TextIO):
#     """"""
#     where.write(f'{len(traps)} ! Number of polygon traps\n')
#
#     for trap in traps:
#         _write_polygon_trap(trap, where)
#
#
# def _write_polygon_trap(trap: PolygonTrap, where: TextIO):
#     open_flag = 1 if trap.is_open else 0
#     single_flag = 1 if trap.is_single else 0
#     num_points = len(trap.points)
#
#     where.write(f'{trap.trap_id} {trap.label}\n')
#     where.write(f'  {num_points} {trap.bottom} {trap.top} {open_flag} {single_flag}\n')
#
#     for number, (x, y) in enumerate(trap.points, start=1):
#         where.write(f'{number} {x} {y}\n')
