"""BcData class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bvdd

PT_WELL = 'well'
PT_CELLHEAD = 'cellhead'
PT_CELLGHB = 'cellghb'
PT_TYPES = {PT_WELL, PT_CELLHEAD, PT_CELLGHB}
ARC_NOFLOW = 'noflow'
ARC_WALLHEAD = 'wallhead'
ARC_WALLGHB = 'wallghb'
ARC_WALLUF = 'walluf'
ARC_TYPES = {ARC_NOFLOW, ARC_WALLHEAD, ARC_WALLGHB, ARC_WALLUF}


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    gm = GenericModel(exclusive_point_conditions=True, exclusive_arc_conditions=True, exclusive_polygon_conditions=True)
    _arc_bc_data_def(gm)
    _pt_bc_data_def(gm)
    return gm


def _arc_bc_data_def(gm):
    """Create parameters for arcs in the GenericModel class.

    Args:
        gm (GenericModel): Generic model class
    """
    ap = gm.arc_parameters

    gp = ap.add_group(group_name='noflow', label='noflow')
    gp.add_option(name='section', label='Section', default='ol', options=['ol', 'gw', 'ol_gw'])
    gp.add_text(name='label', label='No flow label')

    gp = ap.add_group(group_name='wallhead', label='wallhead')
    gp.add_option(name='section', label='Section', default='ol', options=['ol', 'gw', 'ol_gw'])
    gp.add_text(name='label', label='Head label')
    uni_parent = gp.add_boolean(name='uniform', label='Uniform BC value <uniform>', default=True)
    bvdd.add_bc_val_to_group(gp, prefix='upstream_', label_replace='Upstream boundary condition')
    lab = 'Downstream boundary condition'
    bvdd.add_bc_val_to_group(gp, prefix='downstream_', label_replace=lab, parent=uni_parent)

    gp = ap.add_group(group_name='wallghb', label='wallghb')
    gp.add_text(name='label', label='General head label')
    gp.add_float(name='ghb_factor', label='GHB coefficient', default=0.0)
    uni_parent = gp.add_boolean(name='uniform', label='Uniform BC value <uniform>', default=True)
    bvdd.add_bc_val_to_group(gp, prefix='upstream_', label_replace='Upstream boundary condition')
    lab = 'Downstream boundary condition'
    bvdd.add_bc_val_to_group(gp, prefix='downstream_', label_replace=lab, parent=uni_parent)

    gp = ap.add_group(group_name='walluf', label='walluf')
    gp.add_text(name='label', label='Wall uniform flow label')
    gp.add_float(name='uniform_flow', label='Uniform flow', default=0.0)


def _pt_bc_data_def(gm):
    """Create parameters for points in the GenericModel class.

    Args:
        gm (GenericModel): Generic model class
    """
    pp = gm.point_parameters

    gp = pp.add_group(group_name='well', label='well')
    gp.add_text(name='label', label='Well label <label>')
    gp.add_integer(name='well_id', label='Well id <wellid>', default=-1)
    bvdd.add_bc_val_to_group(gp)
    opts = ['INST-VAL', 'PER-AVE', 'PER-CUM']
    gp.add_option(name='well_flow_type', label='Well flow type <type>', default=opts[0], options=opts)

    gp = pp.add_group(group_name='cellhead', label='cellhead')
    gp.add_text(name='label', label='Head label <label>')
    gp.add_integer(name='bc_id', label='BC id <bcid>', default=-1)
    bvdd.add_bc_val_to_group(gp)

    gp = pp.add_group(group_name='cellghb', label='cellghb')
    gp.add_text(name='label', label='GHB label <label>')
    gp.add_float(name='ghb_factor', label='GHB coefficient <value>', default=0.0)
    bvdd.add_bc_val_to_group(gp)
