"""Impoundment data."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import html

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bvdd
from xms.rsm.data import monitor_data_def as mdd


class _ImpoundmentDataDef:
    """A class to define the impoundment data for the generic model."""
    def __init__(self):
        self.generic_model = GenericModel(exclusive_polygon_conditions=False)
        self._pp = self.generic_model.polygon_parameters
        self._poly_data_def()

    def _poly_data_def(self):
        """Defines the polygon data for the generic model."""
        self._impoundment()
        self._boundary_conditions()
        self._monitor()

    def _impoundment(self):
        pp = self._pp
        grp = pp.add_group(group_name='impoundment', label='Impoundment')
        grp.add_text(name='label', label='Label (<label>)')
        label = html.escape('Head (<head>)')
        grp.add_float(name='head', label=label, default=0.0)
        grp.add_float(name='bottom', label='Bottom (<bottom>)', default=0.0)
        label = 'Correction factor for "open water" evapotranspiration (<owCoeff>)'
        grp.add_float(name='ow_coeff', label=label, default=0.0)
        label = 'Correction factor for "shallow water" evapotranspiration (<swCoeff>)'
        grp.add_float(name='sw_coeff', label=label, default=0.0)
        label = 'Depth where evapotranspiration switches from "open" to "shallow" (<swDepth>)'
        grp.add_float(name='sw_depth', label=label, default=0.0)
        label = 'Vertical conductivity for cells under impoundment (<kOverDelV>)'
        grp.add_float(name='k_over_delv', label=label, default=0.0)
        label = 'Confined storage coefficient (<SCconf>)'
        grp.add_float(name='sc_conf', label=label, default=0.0)
        label = 'Horizontal conductivity for cells adjacent to impoundment (<kOverDelH>)'
        grp.add_float(name='k_over_delh', label=label, default=0.0)

    def _boundary_conditions(self):
        pp = self._pp
        grp = pp.add_group(group_name='bc', label='Impoundment Boundary Conditions')
        grp.add_text(name='label', label='Label (<label>)')
        grp.add_integer(name='bc_id', label='Boundary condition id (<id>)', default=-1)
        grp.add_text(name='tag', label='Tag (<tag>)', default='')
        opts = ['impoundmenthead', 'impoundmentsource']
        grp.add_option(name='bc_type', label='BC type', default=opts[0], options=opts)
        bvdd.add_bc_val_to_group(group=grp)

    def _monitor(self):
        pp = self._pp
        outputs = [
            ('impoundmentsto', 'Monitor storage (<impoundmentsto>)'),
            ('initimpoundmentsto', 'Monitor intial storage (<initimpoundmentsto>)'),
            ('head', 'Monitor head (<head>)'),
            ('impoundmentrecharge', 'Monitor recharge (<impoundmentrecharge>)'),
            ('impoundmentrunoff', 'Monitor runoff (<impoundmentrunoff>)'),
            ('impoundmentwsupply', 'Monitor water supply (<impoundmentwsupply>)'),
            ('seepageflow', 'Monitor seepage flow (<seepageflow>)'),
            ('rain', 'Monitor rain (<rain>)'),
            ('refet', 'Monitor evapotranspiration (<refet>)'),
        ]
        for name, label in outputs:
            gp = pp.add_group(group_name=name, label=label)
            mdd.add_output_to_group(gp)


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    dd = _ImpoundmentDataDef()
    return dd.generic_model
