"""Module for exporting an RSM simulation with feedback."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, XmsEnvironment as XmEnv
import xms.components.display.display_options_io as dio
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.rsm.components.mapped_bc_component import MappedBcComponent
from xms.rsm.dmi.xms_data import XmsData
from xms.rsm.file_io import util
from xms.rsm.file_io.mesh_bc_snapper import MeshBcSnapper


class BcSnapRunner(ProcessFeedbackThread):
    """Class for exporting an RSM simulation in a worker thread."""
    def __init__(self, parent, xms_data):
        """Constructor.

        Args:
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
            xms_data (XmsData): The XMS data object
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._xms_data = xms_data
        self._disp_wkt = self._xms_data.display_wkt
        self._bc_snapper = None
        self._do_comp = None
        self._actions = []
        self._pt_files = {}
        self._arc_files = {}
        self._comp_main_file = ''
        self._comp_path = ''
        self._logger = util.get_logger()

    def _create_snap(self):
        """Create the snap component."""
        self._logger.info('Establishing communication with SMS.')
        self._xms_data.remove_existing_snapped_components()
        self._error_check()
        self._bc_snapper = MeshBcSnapper(self._xms_data)
        self._bc_snapper.generate_snap()
        self._create_component_folder_and_copy_display_options()
        self._create_drawing()
        self._create_new_component()
        self._xms_data.add_component_to_xms(self._do_comp, self._actions)

    def _error_check(self):
        """Makes sure there is a ugrid and bc coverages."""
        if self._xms_data.do_ugrid is None:
            self._logger.error('UGrid must be linked to simulation to generate snap.')
            raise RuntimeError
        if len(self._xms_data.bc_coverages) < 1:
            self._logger.error('No BC Coverages to map.')
            raise RuntimeError

    def _create_new_component(self):
        """Create the new component for the snapped bcs."""
        comp_name = 'Snapped BCs'
        comp_uuid = os.path.basename(os.path.dirname(self._comp_main_file))
        self._do_comp = Component(
            main_file=self._comp_main_file,
            name=comp_name,
            model_name='RSM',
            unique_name='MappedBcComponent',
            comp_uuid=comp_uuid
        )
        comp = MappedBcComponent(self._comp_main_file)
        self._actions = [
            ActionRequest(
                modality='NO_DIALOG',
                main_file=self._comp_main_file,
                class_name='MappedBcComponent',
                module_name=comp.module_name,
                method_name='get_initial_display_options'
            )
        ]

    def _new_uuid(self):
        """Generates a new UUID for the component."""
        return str(uuid.uuid4())

    def _create_component_folder_and_copy_display_options(self):
        """Creates the folder for the mapped bc component and copies the display options from the bc coverage."""
        self._logger.info('Creating component folder')
        bc_comp = self._xms_data.bc_coverages[0][1]
        bc_comp_path = os.path.dirname(bc_comp.main_file)
        new_uuid = self._new_uuid()
        self._comp_path = os.path.join(os.path.dirname(bc_comp_path), new_uuid)

        os.mkdir(self._comp_path)
        disp_files = ['pt_display.json', 'arc_display.json']
        for disp in disp_files:
            comp_disp = os.path.join(bc_comp_path, disp)
            snap_disp = os.path.join(self._comp_path, disp)

            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            json_dict = dio.read_display_options_from_json(comp_disp)
            json_dict['uuid'] = self._new_uuid()
            json_dict['comp_uuid'] = new_uuid
            json_dict['is_ids'] = 0
            categories.from_dict(json_dict)
            categories.projection = {'wkt': self._disp_wkt}
            if disp == 'pt_display.json':
                for c in categories.categories:
                    c.options.size += 2
                    self._pt_files[c.description] = c.file
            else:
                for c in categories.categories:
                    c.options.width += 2
                    self._arc_files[c.description] = c.file

            dio.write_display_options_to_json(snap_disp, categories)

        self._comp_main_file = os.path.join(self._comp_path, disp_files[0])

    def _create_drawing(self):
        """Creates the drawing elements for the component."""
        pt_display = {}
        arc_display = {}
        pt_grid_cell_snap = self._bc_snapper.pt_grid_cell_snap
        arc_grid_pt_snap = self._bc_snapper.arc_grid_pt_snap
        for cov, _ in self._xms_data.bc_coverages:
            for _, data in pt_grid_cell_snap[cov.uuid].items():
                comp_type = data['comp_type']
                locs = pt_display.get('comp_type', [])
                locs.extend(data['location'])
                pt_display[comp_type] = locs
            for _, data in arc_grid_pt_snap[cov.uuid].items():
                comp_type = data['comp_type']
                locs = arc_display.get(comp_type, [])
                locs.append(data['locations'])
                arc_display[comp_type] = locs

        comp_path = os.path.dirname(self._comp_main_file)
        for comp_type, locs in pt_display.items():
            fname = os.path.join(comp_path, self._pt_files[comp_type])
            dio.write_display_option_point_locations(fname, locs)
        for comp_type, locs in arc_display.items():
            fname = os.path.join(comp_path, self._arc_files[comp_type])
            dio.write_display_option_line_locations(fname, locs)

    def _do_work(self):
        """Thread worker method."""
        try:
            self._create_snap()
        except Exception:
            self._logger.exception('Error creating snapped boundary conditions.')


def generate_snap(parent, sim, query):
    """Export a simulation with a feedback dialog.

    Args:
        parent (QWidget): The window container.
        sim (SimComponent): the simlution
        query (Query): XMS interprocess communicator
    """
    ensure_qapplication_exists()
    xms_data = XmsData(query=query, sim_component=sim)
    worker = BcSnapRunner(parent, xms_data)

    display_text = {
        'title': 'RSM Generate Snap Preview',
        'working_prompt': 'Generating RSM Snap Preview. Please wait...',
        'warning_prompt': 'Warning(s) encountered while generating Snap Preview. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while generating Snap Preview. Review log output for more details.',
        'success_prompt': 'Successfully generated Snap Preview',
        'note': '',
        'auto_load': 'Close this dialog automatically when generation is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms.rsm', worker=worker, parent=parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    feedback_dlg.exec()
