"""Class for writing canal coverage data to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bc_val_dd
from xms.rsm.file_io import util


class BcValWriter:
    """Writer class for the RSM control file."""
    def __init__(self, xml_parent, group, csv_writer, rule_curve_label_id, feature_id):
        """Constructor.

        Args:
            xml_parent (xml.etree.cElementTree.Element): xml parent element
            group (xms.gmi.data.generic_model.Group): The generic model group
            csv_writer (CsvWriter): csv writer
            rule_curve_label_id (dict): dictionary mapping rule curve labels to ids
            feature_id (str): string identifier of the feature and the coverage
        """
        self._logger = util.get_logger()
        self._xml_parent = xml_parent
        self._group = group
        self._csv_writer = csv_writer
        self._rule_curve_label_id = rule_curve_label_id
        self._feature_id = feature_id
        self._bc_val_method = {
            bc_val_dd.BC_VAL_FLT: self._add_bc_val_constant,
            bc_val_dd.BC_VAL_CURVE: self._add_bc_val_curve,
            bc_val_dd.BC_VAL_DSS: self._add_bc_val_dss,
            bc_val_dd.BC_VAL_RC: self._add_bc_val_rule_curve,
        }
        self._bc_val_prefix = ''
        self._bc_val_type = None
        self._bc_val_flt = None
        self._bc_val_flt_type = None
        self._bc_val_xy = None
        self._bc_val_dss_file = None
        self._bc_val_dss_path = None
        self._bc_val_dss_mult = 1.0
        self._bc_val_dss_units = None
        self._bc_val_dss_type = None
        self._bc_val_rc = None

    def write(self):
        """Write the bc value to the xml parent."""
        self._load_bc_data()
        self._bc_val_method[self._bc_val_type]()

    def set_bc_val_prefix(self, prefix):
        """Set the bc value prefix.

        Args:
            prefix (str): The prefix to set.
        """
        self._bc_val_prefix = prefix

    def _load_bc_data(self):
        """Load the bc data."""
        prefix = self._bc_val_prefix
        self._bc_val_type = self._group.parameter(f'{prefix}bc_val_type').value
        self._bc_val_flt = self._group.parameter(f'{prefix}bc_val_flt').value
        par_name = f'{prefix}bc_val_flt_type'
        if par_name in self._group.parameter_names:  # this is only used for <well>
            self._bc_val_flt_type = self._group.parameter(par_name).value
        par_name = f'{prefix}bc_val_curve'
        if par_name in self._group.parameter_names:  # you can turn off the curve option
            self._bc_val_xy = self._group.parameter(par_name).value
        self._bc_val_dss_file = self._group.parameter(f'{prefix}bc_val_dss').value
        self._bc_val_dss_path = self._group.parameter(f'{prefix}bc_val_dss_path').value
        self._bc_val_dss_mult = self._group.parameter(f'{prefix}bc_val_dss_mult').value
        self._bc_val_dss_units = self._group.parameter(f'{prefix}bc_val_dss_units').value
        self._bc_val_dss_type = self._group.parameter(f'{prefix}bc_val_dss_type').value
        par_name = f'{prefix}bc_val_rc'
        if par_name in self._group.parameter_names:  # you can turn off the RC option
            self._bc_val_rc = self._group.parameter(par_name).value

    def _add_bc_val_constant(self):
        """Add the constant value to the xml parent."""
        atts = {'value': f'{self._bc_val_flt}'}
        if self._bc_val_flt_type:
            atts['type'] = self._bc_val_flt_type
        Et.SubElement(self._xml_parent, 'const', atts)

    def _add_bc_val_curve(self):
        """Add the curve value to the xml parent."""
        atts = self._csv_writer.write_csv_file(self._bc_val_xy)
        Et.SubElement(self._xml_parent, 'csv', atts)

    def _add_bc_val_dss(self):
        """Add the dss value to the xml parent."""
        atts = {
            'file': self._bc_val_dss_file,
            'pn': self._bc_val_dss_path,
        }
        if self._bc_val_dss_mult != 1.0:
            atts['mult'] = f'{self._bc_val_dss_mult}'
        if self._bc_val_dss_units:
            atts['units'] = self._bc_val_dss_units
        if self._bc_val_dss_type != 'Not specified':
            atts['type'] = self._bc_val_dss_type
        Et.SubElement(self._xml_parent, 'dss', atts)

    def _add_bc_val_rule_curve(self):
        """Add the rule curve value to the xml parent."""
        if self._bc_val_rc not in self._rule_curve_label_id:
            msg = f'Rule curve "{self._bc_val_rc}" not found. BC associated with {self._feature_id}.'
            self._logger.error(msg)
            return
        rc_id = self._rule_curve_label_id[self._bc_val_rc]
        Et.SubElement(self._xml_parent, 'rc', {'id': f'{rc_id}'})
