"""Module for the mapped upwind solver coverage's data manager."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional, Sequence

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules

# 4. Local modules
from xms.schism.data.base_data import BaseData

_SOLVER_DS = 'solver'


def make_mapping_dataset(nodes: Optional[Sequence[int]] = None) -> xr.Dataset:
    """
    Make a dataset for the mapping from cell index to feature type index.

    Args:
        nodes: Sequence of feature type indices. Parallel to the cell list for the geometry.

    Returns:
        A dataset for the mapping.
    """
    nodes = nodes if nodes is not None else []
    dataset = {'solver': np.array(nodes, dtype=np.int32)}
    return xr.Dataset(data_vars=dataset)


class MappedUpwindSolverCoverageData(BaseData):
    """Manages a mapped BC coverage's data."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """
        Initialize the object.

        Args:
            main_file: Path to the component's main file.
        """
        super().__init__(main_file)

        self.info.attrs['SCHISM_FILE_TYPE'] = 'MAPPED_UPWIND_SOLVER'

        self.commit()

    @property
    def solver(self) -> Sequence[int]:
        """Solver dataset for each cell."""
        return self._get_dataset(_SOLVER_DS)['solver'].values

    @solver.setter
    def solver(self, value: Sequence[int]):
        ds = make_mapping_dataset(value)
        self._set_dataset(ds, _SOLVER_DS)

    @property
    def domain_hash(self) -> str:
        """The hash of the domain the mapped coverage was made for."""
        return self.info.attrs.get('domain_hash', '')

    @domain_hash.setter
    def domain_hash(self, value: str):
        """The hash of the domain the mapped coverage was made for."""
        self.info.attrs['domain_hash'] = value

    @property
    def _dataset_names(self):
        return {_SOLVER_DS}

    @property
    def _main_file_name(self) -> str:
        """What to name the component's main file."""
        return 'schism_mapped_upwind_solver_coverage.nc'

    def _create_dataset(self, name: str):
        """Create a dataset."""
        if name == _SOLVER_DS:
            return make_mapping_dataset()
        raise AssertionError('Unknown dataset')  # pragma: nocover
