"""Worker thread for the solution importer."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules


class HydraulicLinesWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing Floodway evaluation lines."""
    def __init__(self, parent, generator):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): The Qt parent
            generator (:obj:`GenerateFloodwayEvaluationLines`): Computes the floodway evaluation lines
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.srh')
        self._generator = generator
        self.coverage = None

    def _do_work(self):
        """Thread runner that computes the floodway evaluation lines."""
        # Look for a mapped BC coverage first.
        self._logger.info('computing hydraulic evaluation lines...')
        try:
            self._generator.generate_floodway_evaluation_lines()
        except Exception:  # pragma no cover - hard to test exceptions using QueryPlayback
            self._logger.exception('Error(s) encountered while computing hydraulic evaluation lines.')

    def add_coverage(self):
        """Send imported data to XMS."""
        if self._generator.new_xs_cov:
            self._generator.query.add_coverage(self._generator.new_xs_cov)


def run_generate_floodway_evaluation_lines_with_feedback(parent, generator):
    """Entry point to read a TUFLOWFV solution.

    Args:
        parent (:obj:`QWidget`): The Qt parent
        generator (:obj:`GenerateFloodwayEvaluationLines`): Computes the floodway evaluation lines
    """
    worker = HydraulicLinesWorkerThread(parent, generator)
    display_text = {
        'title': 'Computing Hydraulic Evaluation Lines',
        'working_prompt': 'Computing hydraulic evaluation lines. Please wait...',
        'warning_prompt':
            'Warning(s) encountered computing hydraulic evaluation lines.'
            ' Review log output for more details.',
        'error_prompt':
            'Error(s) encountered computing hydraulic evaluation lines. '
            'Review log output for more details.',
        'success_prompt': 'Successfully computed hydraulic evaluation lines',
        'note': '',
        'auto_load': 'Auto load coverage into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():  # Send even if errors because we may have a partial solution
        worker.add_coverage()
