"""BcDataParam class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules
from xms.srh.data.par.bc_data_arcs import BcDataArcs
from xms.srh.data.par.bc_data_bcdata import BcDataBcData
from xms.srh.data.par.bc_data_culvert import BcDataCulvert
from xms.srh.data.par.bc_data_culvert_hy8 import BcDataCulvertHy8
from xms.srh.data.par.bc_data_exit_h import BcDataExitH
from xms.srh.data.par.bc_data_exit_q import BcDataExitQ
from xms.srh.data.par.bc_data_flow_direction import BcDataFlowDirection
from xms.srh.data.par.bc_data_gate import BcDataGate
from xms.srh.data.par.bc_data_inlet_q import BcDataInletQ
from xms.srh.data.par.bc_data_inlet_sc import BcDataInletSc
from xms.srh.data.par.bc_data_internal_sink import BcDataInternalSink
from xms.srh.data.par.bc_data_line_selector import BcDataLineSelector
from xms.srh.data.par.bc_data_link import BcDataLink
from xms.srh.data.par.bc_data_pressure import BcDataPressure
from xms.srh.data.par.bc_data_sed_inflow import BcDataSedInflow
from xms.srh.data.par.bc_data_wall import BcDataWall
from xms.srh.data.par.bc_data_weir import BcDataWeir


class BcDataParam(param.Parameterized):
    """Bc data class."""
    bc_type = param.ObjectSelector(
        label='BC Type',
        default='Wall (no-slip boundary)',
        objects=[
            'Inlet-Q (subcritical inflow)', 'Exit-H (subcritical outflow)', 'Exit-Q (known-Q outflow)',
            'Inlet-SC (supercritical inflow)', 'Exit-EX (supercritical outflow)', 'Wall (no-slip boundary)',
            'Symmetry (slip boundary)', 'Internal sink', 'Culvert HY-8', 'Culvert', 'Weir', 'Pressure', 'Gate', 'Link',
            'Bc Data'
        ],
        doc='option for specifying the WSE',
        precedence=1,
    )
    label = param.String(label='Label', default='', doc='Label for the BC line', precedence=1.1)
    inlet_q = param.ClassSelector(
        label='',
        class_=BcDataInletQ,
        doc='Inlet-Q bc parameters',
        precedence=2,
    )
    exit_h = param.ClassSelector(
        label='',
        class_=BcDataExitH,
        doc='Exit-H bc parameters',
        precedence=3,
    )
    exit_q = param.ClassSelector(
        label='',
        class_=BcDataExitQ,
        doc='Exit-Q bc parameters',
        precedence=4,
    )
    inlet_sc = param.ClassSelector(
        label='',
        class_=BcDataInletSc,
        doc='Inlet SC bc parameters',
        precedence=5,
    )
    wall = param.ClassSelector(
        label='',
        class_=BcDataWall,
        doc='Wall bc parameters',
        precedence=6,
    )
    internal_sink = param.ClassSelector(
        label='',
        class_=BcDataInternalSink,
        doc='Wall bc parameters',
        precedence=7,
    )
    arcs = param.ClassSelector(
        label='',
        class_=BcDataArcs,
        doc='Culvert bc parameters',
        precedence=8,
    )
    hy8_culvert = param.ClassSelector(
        label='',
        class_=BcDataCulvertHy8,
        doc='Wall bc parameters',
        precedence=9,
    )
    culvert = param.ClassSelector(
        label='',
        class_=BcDataCulvert,
        doc='Culvert bc parameters',
        precedence=10,
    )
    weir = param.ClassSelector(
        label='',
        class_=BcDataWeir,
        doc='Weir bc parameters',
        precedence=11,
    )
    pressure = param.ClassSelector(
        label='',
        class_=BcDataPressure,
        doc='Pressure bc parameters',
        precedence=12,
    )
    gate = param.ClassSelector(
        label='',
        class_=BcDataGate,
        doc='Gate bc parameters',
        precedence=13,
    )
    link = param.ClassSelector(
        label='',
        class_=BcDataLink,
        doc='Link bc parameters',
        precedence=14,
    )
    sediment_inflow = param.ClassSelector(
        label='',
        class_=BcDataSedInflow,
        doc='sediment inflow',
        precedence=15,
    )
    bc_data = param.ClassSelector(
        label='',
        class_=BcDataBcData,
        doc='bc data',
        precedence=16,
    )
    bc_data_lines = param.ClassSelector(
        label='',
        class_=BcDataLineSelector,
        doc='bc data lines for structures',
        precedence=17,
    )
    flow_direction = param.ClassSelector(
        label='',
        class_=BcDataFlowDirection,
        doc='Structure flow direction',
        precedence=18,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.inlet_q = BcDataInletQ()
        self.exit_h = BcDataExitH()
        self.exit_q = BcDataExitQ()
        self.inlet_sc = BcDataInletSc()
        self.wall = BcDataWall()
        self.internal_sink = BcDataInternalSink()
        self.arcs = BcDataArcs()
        self.hy8_culvert = BcDataCulvertHy8()
        self.culvert = BcDataCulvert()
        self.weir = BcDataWeir()
        self.pressure = BcDataPressure()
        self.gate = BcDataGate()
        self.link = BcDataLink()
        self.sediment_inflow = BcDataSedInflow()
        self.bc_data = BcDataBcData()
        self.bc_data_lines = BcDataLineSelector()
        self.flow_direction = BcDataFlowDirection()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['bc_type'],
            depends={
                'Inlet-Q (subcritical inflow)': ['label', 'inlet_q', 'sediment_inflow'],
                'Exit-H (subcritical outflow)': ['label', 'exit_h'],
                'Exit-Q (known-Q outflow)': ['label', 'exit_q'],
                'Inlet-SC (supercritical inflow)': ['label', 'inlet_sc', 'sediment_inflow'],
                'Wall (no-slip boundary)': ['wall'],
                'Internal sink': ['label', 'internal_sink'],
                'Culvert HY-8': ['label', 'arcs', 'hy8_culvert', 'bc_data_lines', 'flow_direction'],
                'Culvert': ['label', 'arcs', 'culvert'],
                'Weir': ['label', 'arcs', 'weir', 'bc_data_lines', 'flow_direction'],
                'Pressure': ['label', 'arcs', 'pressure', 'bc_data_lines', 'flow_direction'],
                'Gate': ['label', 'arcs', 'gate', 'bc_data_lines', 'flow_direction'],
                'Link': ['label', 'arcs', 'link', 'bc_data_lines', 'flow_direction'],
                'Bc Data': ['bc_data'],
            },
        )

    def bc_type_index(self):
        """Returns the index of the bc_type.

        Returns:
            (int): index of bc_type
        """
        return self.param.bc_type.objects.index(self.bc_type)

    @param.depends('bc_type', watch=True)
    def update_bc_type(self):
        """Watches changes made to the bc_type object."""
        self.enabler.do_enabling(force_disable=False)
