"""BcComponent class. Data for Bc Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import filecmp
import os
from pathlib import Path
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem

# 4. Local modules
from xms.srh.data.bc_data import BcData

BC_INITIAL_ATT_ID_FILE = 'initial_bc.attids'
BC_INITIAL_COMP_ID_FILE = 'initial_bc.compids'


class Hy8Migrate:
    """Migrate from one hy8 file per bc coverage to one hy8 file for the project."""
    def __init__(self):
        """Initializes the base component class."""
        self.query = None
        self.comp_folder = ''
        self.hy8_backup_folder = ''
        self.project_hy8_file = ''
        self.hy8_files = []
        self.messages = []
        self._hy8_files_different = False
        self._hy8_merge_data = None
        self._base_crossings = {}
        self._renamed_crossings = {}

    def merge_file_into_project(self, proj_hy8, bc_data, bc_hy8):
        """Merge the bc_hy8 into the projection hy8 file.

        Args:
            proj_hy8 (:obj:`str`): the project hy8 file
            bc_data (:obj:`BcData`): the data class for the bc coverage
            bc_hy8 (:obj:`str`): the hy8 file associated with the bc coverage
        """
        self.project_hy8_file = proj_hy8
        self.hy8_files = [
            (
                proj_hy8, {
                    'hy8_file': proj_hy8,
                    'cover_name': 'Project_file',
                    'culvert_in_cover': True,
                    'bc_data': None,
                    'diff_from_base_file': False
                }
            ),
            (
                bc_hy8, {
                    'hy8_file': bc_hy8,
                    'cover_name': 'SMART_NAME',
                    'culvert_in_cover': True,
                    'bc_data': bc_data,
                    'diff_from_base_file': True
                }
            ),
        ]
        self._merge_hy8_files()

    def migrate(self, query):
        """Does the migration.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS

        Returns:
            Message and ActionRequest lists
        """
        tmp_dir = XmEnv.xms_environ_temp_directory()
        hy8_file = os.path.join(tmp_dir, 'hy8.txt')
        if not os.path.isfile(hy8_file):
            return [], []  # return because no txt file exists
        filesystem.removefile(hy8_file)
        self.query = query

        self.comp_folder = os.path.join(tmp_dir, 'Components')
        self._get_all_hy8_files()
        self._copy_newest_hy8()
        self._check_hy8_files_different()
        if self._hy8_files_different:
            self._create_hy8_backup_folder()
            self._copy_hy8_to_backup()
            self._merge_hy8_files()
        self._remove_old_hy8_files()
        return self.messages, []

    def _remove_old_hy8_files(self):
        """Removes old hy8 files from the components area."""
        for item in self.hy8_files:
            fname = item[1]['hy8_file']
            filesystem.removefile(fname)

    def _check_hy8_files_different(self):
        """Check if the hy8 files are different."""
        for item in self.hy8_files:
            mod_time = item[0]
            culvert_in_cover = item[1]['culvert_in_cover']
            fname = item[1]['hy8_file']
            if mod_time > 0.0 and culvert_in_cover and not filecmp.cmp(self.project_hy8_file, fname):
                item[1]['diff_from_base_file'] = True
                self._hy8_files_different = True

    def _copy_hy8_to_backup(self):
        """Copy hy8 files to the backup area."""
        if os.path.isdir(self.hy8_backup_folder):
            for item in self.hy8_files:
                if item[0] > 0.0:
                    new_hy8 = os.path.join(self.hy8_backup_folder, f'{item[1]["cover_name"]}.hy8')
                    filesystem.copyfile(item[1]['hy8_file'], new_hy8)

    def _copy_newest_hy8(self):
        """Copy the newest hy8 to the project shared_data area."""
        if self.hy8_files:
            hy8_file = os.path.join(self.comp_folder, 'shared_data', 'culvert.hy8')
            filesystem.copyfile(self.hy8_files[0][1]['hy8_file'], hy8_file)
            self.project_hy8_file = hy8_file

    def _create_hy8_backup_folder(self):
        """Creates a folder for a backup of the hy8 files."""
        proj_name = XmEnv.xms_environ_project_path()
        srh_folder = os.path.join(os.path.splitext(proj_name)[0], 'SRH-2D')
        hy8_backup_folder = os.path.join(os.path.dirname(proj_name), 'hy8_backup')
        if os.path.isdir(srh_folder):
            hy8_backup_folder = os.path.join(srh_folder, 'hy8_backup')
        if not os.path.isdir(hy8_backup_folder):
            os.mkdir(hy8_backup_folder)
        self.hy8_backup_folder = hy8_backup_folder

    def _get_all_hy8_files(self):
        """Get all the hy8 files in the project."""
        # get all hy8 files
        bc_files = [str(f.absolute()) for f in Path(self.comp_folder).rglob('bc_comp.nc')]
        hy8_files = []
        for bf in bc_files:
            data = BcData(bf)
            bc_types = data.bc_data.to_dataframe()['bc_type'].to_list()
            cov_uuid = data.info.attrs['cov_uuid']
            cov = tree_util.find_tree_node_by_uuid(self.query.project_tree, cov_uuid)
            if cov:
                hy8_file = os.path.join(os.path.dirname(bf), 'culvert.hy8')
                if os.path.isfile(hy8_file):
                    f_size = os.path.getsize(hy8_file)
                    mod_time = os.path.getmtime(hy8_file) if f_size > 0 else 0.0
                    hy8_files.append(
                        (
                            mod_time, hy8_file, {
                                'hy8_file': hy8_file,
                                'cover_name': cov.name,
                                'culvert_in_cover': 'Culvert HY-8' in bc_types,
                                'bc_data': data,
                                'diff_from_base_file': False
                            }
                        )
                    )
        hy8_files.sort(reverse=True)
        self.hy8_files = [(item[0], item[2]) for item in hy8_files]

    def _merge_hy8_files(self):
        """Merge info from various files to the project hy8 file."""
        self._read_hy8_data()
        self.hy8_files[0][1]['diff_from_base_file'] = True  # force to use the first hy8 file
        for item in self.hy8_files:
            if not item[1]['diff_from_base_file']:
                continue
            # need to merge the data from this hy8 file
            self._hy8_merge_data = item[1]
            self._merge_coverage_hy8()
        self._rewrite_project_hy8()

    def _rewrite_project_hy8(self):
        """Rewrites the project hy8 file with crossings that were added."""
        with open(self.project_hy8_file, 'r') as f:
            lines = f.readlines()

        idx = 0
        for i, line in enumerate(lines):
            if line.startswith('NUMCROSSINGS'):
                idx = i
                break
        new_lines = lines[:idx]
        cross_dict = self._base_crossings
        new_lines.append(f'NUMCROSSINGS         {len(cross_dict)}\n')
        for guid, name_data in cross_dict.items():
            new_lines.append(f'STARTCROSSING        {name_data[0]}\n')
            new_lines.extend(name_data[1])
            new_lines.append(f'CROSSGUID            {guid}\n')
            new_lines.append(f'ENDCROSSING          {name_data[0]}\n')
        new_lines.append('ENDPROJECTFILE')

        with open(self.project_hy8_file, 'w') as f:
            f.writelines(new_lines)

    def _merge_coverage_hy8(self):
        """Merge the hy8 data in _cov_hy8_merge into the project hy8."""
        base_crossings = self._base_crossings
        base_names = {data[0]: guid for guid, data in base_crossings.items()}
        crossings = self._hy8_merge_data['cross_dict']
        old_to_new_guid = {}
        for guid, name_data in crossings.items():
            cross_data = name_data[1]
            cross_name = name_data[0]

            cnt = 1
            orig_cross_name = cross_name
            done = False
            while not done:
                if cross_name in base_names:
                    base_guid = base_names[cross_name]
                    base_data = base_crossings[base_guid][1]
                    if cross_data == base_data:
                        # name and data match something in the base so change the guid to the base guid
                        if guid != base_guid:
                            old_to_new_guid[guid] = base_guid
                        done = True
                    else:  # crossing data does not match so change the name of the crossing
                        cross_name = f'{orig_cross_name}---{cnt}'
                        cnt += 1
                else:
                    self._create_user_message()
                    if guid in base_crossings:  # make sure guid does not collide
                        new_guid = self._get_uuid_str()
                        base_crossings[new_guid] = (cross_name, cross_data)
                        base_names[cross_name] = new_guid
                        old_to_new_guid[guid] = new_guid
                    else:
                        base_crossings[guid] = (cross_name, cross_data)
                    done = True

        if len(old_to_new_guid) > 0:
            data_updated = False
            # update the bc coverage
            bc_data = self._hy8_merge_data['bc_data']
            df = bc_data.bc_data.to_dataframe()
            df = df[df.bc_type == 'Culvert HY-8']
            ids = df['id'].astype(int).to_list()
            for bc_id in ids:
                p = bc_data.bc_data_param_from_id(bc_id)
                if p.hy8_culvert.hy8_crossing_guid in old_to_new_guid:
                    p.hy8_culvert.hy8_crossing_guid = old_to_new_guid[p.hy8_culvert.hy8_crossing_guid]
                    bc_data.set_bc_data_with_id(p, bc_id)
                    data_updated = True
            if data_updated:
                bc_data.commit()  # update the bc data for the component

    def _get_uuid_str(self):
        """Returns a UUID string.

        Returns:
            (:obj:`str`): uuid as string
        """
        return str(uuid.uuid4())

    def _read_hy8_data(self):
        """Read the data from the hy8 files."""
        for item in self.hy8_files:
            with open(item[1]['hy8_file']) as f:
                lines = f.readlines()
            # put the crossings from this file into data blocks
            name = guid = ''
            cross_data = []
            cross_dict = {}
            for line in lines:
                items = line.split()
                if len(items) < 1:
                    pass
                elif 'STARTCROSSING' == items[0]:
                    name = line
                    name = name.replace('STARTCROSSING', '')
                    name = name.rstrip().lstrip()
                    name = name.rstrip('"').lstrip('"')
                elif 'CROSSGUID' == items[0]:
                    guid = items[1]
                elif 'ENDCROSSING' == items[0]:
                    if guid not in cross_dict:
                        cross_dict[guid] = (name, cross_data)
                    name = guid = ''
                    cross_data = []
                elif name != '':
                    cross_data.append(line)
            item[1]['cross_dict'] = cross_dict

    def _create_user_message(self):
        """Fills the message that goes to the user."""
        if not self.messages:
            url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Structures#HY8_File'
            cov_str = ''.join([f'{d[1]["cover_name"]}\n' for d in self.hy8_files[1:] if d[1]['culvert_in_cover']])
            if self.hy8_files:
                msg = f'Different HY8 files were in the project.\n' \
                      f'Beginning in SMS 13.2, only one HY8 file will be in the project.\n' \
                      f'Crossing data from all coverages has been merged into the project hy8 file.\n' \
                      f'Some crossings were renamed to avoid duplicate crossing names.' \
                      f'All HY8 files from the project were saved to this location:\n\n' \
                      f'{self.hy8_backup_folder}.\n\n' \
                      f'More details can be found at\n {url}\n\n' \
                      f'Verify that any HY8 structures in the following coverages are using the correct ' \
                      f'crossing:\n{cov_str}'
                self.messages = [('INFO', msg)]
