"""Read an SRH-2D monitor point file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Point

# 4. Local modules


class MonitorPointReader:
    """Reads a SRH-2D monitor point file."""
    def __init__(self):
        """Constructor."""
        self._logger = logging.getLogger('xms.srh')
        self.monitor_pts = []

    def read(self, filename):
        """Reads an SRH monitor point file.

        Args:
            filename (:obj:`str`): Filepath of monitor point file to be read.
        """
        try:
            with open(filename, 'r') as file:
                lines = file.readlines()
            if len(lines) < 3:  # Invalid or empty file
                self._logger.warning(f'No monitor points found in file: {filename}')
                return

            for line in lines:
                line = line.split()
                if len(line) < 4:  # Invalid, empty, or header line
                    continue
                self.monitor_pts.append((float(line[2]), float(line[3])))
        except Exception:  # pragma no cover
            self._logger.exception(f'Error reading monitor point file: {filename}')

    def get_do_points(self):
        """Returns a list of the monitor points as data_objects with ids numbered 1-N."""
        return [Point(x=coord[0], y=coord[1], feature_id=i + 1) for i, coord in enumerate(self.monitor_pts)]
