"""Handles exporting simulations that use PEST."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import sys

# 2. Third party modules
import orjson

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.file_io.pest_pst_writer import PestPstWriter
from xms.srh.srh_run import get_sms_srh_exes


class PestExporter:
    """Handles exporting simulations that use PEST."""
    def __init__(self):
        """Initializes the class."""
        self._exporter = None  # FileWriter class
        self._logger = logging.getLogger('xms.srh')

    def export(self, exporter):
        """Export the PEST files.

        Args:
            exporter (:obj:`FileWriter`): The worker class containing the data to export
        """
        self._exporter = exporter

        # Export non-parameter stuff
        self._exporter.export_hydro(True)
        self._export_pest_run_files()

    def _export_pest_run_files(self):
        """Exports the files needed to run PEST with the exception of the SRHHYDRO template file."""
        self._logger.info('Writing PEST files.')
        monitor_pts = self._exporter.coverage_mapper.monitor_points
        mon_pts = []
        obs_pts = []
        if monitor_pts:
            for pt in monitor_pts:
                mon_pts.append(pt[3])
                if pt[3] is True:
                    obs_pts.append([pt[4], pt[5]])
        if len(obs_pts) == 0:
            self._logger.error(
                'Unable to save PEST files.  Please assign an SRH-2D monitor coverage containing points '
                'with observation values to this simulation.'
            )
            return

        json_dict = {
            'PROJECT_NAME': self._exporter.project_name,
            'CASE_NAME': self._exporter.sim_component.data.hydro.case_name,
            'OBSERVATIONS': obs_pts,
            'MONITOR_PTS': mon_pts,
            'PARAMETER_NAMES': [],  # Will get set by PestPstWriter
        }
        file_name = os.path.join(self._exporter.out_dir, f'{self._exporter.project_name}.pst')
        writer = PestPstWriter(file_name, json_dict, main_file=self._exporter.sim_component.main_file)
        writer.write()

        # Write the json file to pass data to the pest run script
        pest_json_file = os.path.join(self._exporter.out_dir, 'srh_pest.json')
        with open(pest_json_file, 'wb') as f:
            data = orjson.dumps(json_dict)
            f.write(data)
            f.close()

        # Write the .ins (instruction) file
        pest_ins_file = os.path.join(self._exporter.out_dir, f'{self._exporter.project_name}_obs.ins')
        with open(pest_ins_file, 'w') as f:
            f.write('pif #\n')
            num_obs = len(json_dict['OBSERVATIONS'])
            for i in range(1, num_obs + 1):
                f.write(f'l1 !wse{i}!\n')
            f.close()

        # Write batch file to run the SRH2D manager python script from PEST
        py_interpreter = sys.executable
        pest_bat_file = os.path.join(self._exporter.out_dir, 'pest_run_model.bat')
        with open(pest_bat_file, 'w') as f:
            f.write(f'"{py_interpreter}" -m xms.srh pest -j "{pest_json_file}"\n')
            f.write('exit\n')
            f.close()

        # Write batch files to run SRH2D executables from PEST
        srh2d_pre_path, srh2d_path = self._get_srh2d_executable_paths()
        run_pre_bat_file = os.path.join(self._exporter.out_dir, 'pest_run_pre.bat')
        with open(run_pre_bat_file, 'w') as f:
            f.write(f'"{srh2d_pre_path}" 3 "{self._exporter.project_name}.srhhydro"\n')
            f.write('exit\n')
            f.close()
        run_pest_bat_file = os.path.join(self._exporter.out_dir, 'pest_run_srh.bat')
        with open(run_pest_bat_file, 'w') as f:
            f.write(f'"{srh2d_path}" "{self._exporter.sim_component.data.hydro.case_name}.dat"\n')
            f.write('exit\n')
            f.close()
        self._logger.info('Success writing PEST files.')

    @staticmethod
    def _get_srh2d_executable_paths():
        """Gets SRH2D executable paths.

        Returns:
            (:obj:`tuple(str)`): SRH-Pre path, SRH-2D path
        """
        exes = get_sms_srh_exes()
        return exes['SRH-2D - PreSRH-2D'], exes['SRH-2D - SRH-2D']
