"""The Compare Simulation Data Output dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QApplication

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.srh.gui.compare_simulation_data_output_dialog_ui import Ui_CompareSimulationDataOutputDialog  # noqa: I202


class CompareSimulationDataOutputDialog(XmsDlg):
    """The Compare Simulation Data dialog."""

    # Columns in the output table
    SIMULATION_OUTPUT_ARC_ID = 0
    SIMULATION_OUTPUT_FP_WSE = 1
    SIMULATION_OUTPUT_FW_WSE = 2
    SIMULATION_OUTPUT_WSE_MIN_DIFF = 3
    SIMULATION_OUTPUT_WSE_MAX_DIFF = 4
    SIMULATION_OUTPUT_WSE_AVG_DIFF = 5
    SIMULATION_OUTPUT_FP_VMAG = 6
    SIMULATION_OUTPUT_FW_VMAG = 7
    SIMULATION_OUTPUT_VMAG_MIN_DIFF = 8
    SIMULATION_OUTPUT_VMAG_MAX_DIFF = 9
    SIMULATION_OUTPUT_VMAG_AVG_DIFF = 10

    def __init__(self, data, significant_figures, parent=None):
        """Initializes the dialog.

        Args:
            data (:obj:`dict`): Python dict containing a set of keys that represent the columns in the table with their
                values as lists of floats containing the data
            significant_figures (:obj:`int`): The number of significant figures (+1) to round the comparison data.
            parent (:obj:`QObject`): The Qt parent object
        """
        super().__init__(parent, 'xms.srh.gui.compare_simulation_data_output_dialog')

        self.significant_figures = significant_figures

        self.ui = Ui_CompareSimulationDataOutputDialog()
        self.ui.setupUi(self)
        self.params_model = None
        self.data = data
        self.setup_data_table()
        self.ui.btn_copy.clicked.connect(self.copy_data)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Floodway_Tools#Compare_Simulation_Data'
        self.ui.buttonBox.button(self.ui.buttonBox.Help).clicked.connect(self.help_requested)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def setup_data_table(self):
        """Sets up the output data table."""
        full_df = pd.DataFrame(self.data)
        df = full_df.round(self.significant_figures + 1)
        self.params_model = QxPandasTableModel(df)
        self.params_model.read_only_columns = {i for i in range(self.SIMULATION_OUTPUT_VMAG_AVG_DIFF + 1)}
        self.ui.tbl_compare.setModel(self.params_model)
        widget_builder.style_table_view(self.ui.tbl_compare)
        self.ui.tbl_compare.resizeColumnsToContents()
        self.ui.tbl_compare.verticalHeader().setVisible(False)
        self.ui.tbl_compare.horizontalHeader().setStretchLastSection(True)

    def copy_data(self):
        """Handle copy events for the flows table."""
        full_df = pd.DataFrame(self.data)
        df = full_df.round(self.significant_figures + 1)
        df_string = df.to_string()
        QApplication.clipboard().setText(df_string)
        # selection = self.ui.table_data.selectedIndexes()
        # if selection:
        #     rows = sorted(index.row() for index in selection)
        #     columns = sorted(index.column() for index in selection)
        #     row_count = rows[-1] - rows[0] + 1
        #     col_count = columns[-1] - columns[0] + 1
        #     table = [[''] * col_count for _ in range(row_count)]
        #     for index in selection:
        #         row = index.row() - rows[0]
        #         column = index.column() - columns[0]
        #         table[row][column] = index.data()
        #     stream = io.StringIO()
        #     csv.writer(stream, delimiter='\t').writerows(table)
        #     QApplication.clipboard().setText(stream.getvalue())
