"""Filter model for the material properties table."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.data.material_data import MaterialData


class MaterialFilterModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        row = index.row()
        col = index.column()

        if row == MaterialData.UNASSIGNED_MAT:
            if col == MaterialData.COL_NAME:  # Disable editing of the unassigned material name
                ret_flags = ret_flags & (~Qt.ItemIsEditable)
            elif col in [MaterialData.COL_N, MaterialData.COL_DEPTH_TOG, MaterialData.COL_DEPTH_CURVE]:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        elif col == MaterialData.COL_DEPTH_TOG:
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
            ret_flags |= Qt.ItemIsEnabled
        elif col == MaterialData.COL_N:
            checked = index.model().index(row, MaterialData.COL_DEPTH_TOG).data(Qt.CheckStateRole)
            if checked != Qt.Checked:
                ret_flags |= Qt.ItemIsEnabled
            else:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        elif index.column() == MaterialData.COL_DEPTH_CURVE:
            checked = index.model().index(row, MaterialData.COL_DEPTH_TOG).data(Qt.CheckStateRole)
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
            if checked == Qt.Checked:
                ret_flags |= Qt.ItemIsEnabled
            else:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        else:
            ret_flags |= Qt.ItemIsEnabled
        return ret_flags
