"""Classes for displaying a column with dataset selectors, file selectors, and edit fields in a Qt table."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QModelIndex
from PySide2.QtGui import QColor
from PySide2.QtWidgets import QLineEdit, QStyledItemDelegate

# 3. Aquaveo modules
from xms.guipy.delegates.dataset_button_delegate import DatasetButtonDelegate
from xms.guipy.delegates.file_selector_delegate import FileSelectorButtonDelegate
from xms.guipy.dialogs import treeitem_selector_datasets
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.resources import resources_util
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.srh.components.parameters import initial_condition_parameter
from xms.srh.file_io import io_util


class QxDelegateConstantFileDataset(QStyledItemDelegate):
    """Delegate for the file selector button column."""
    def __init__(
        self, proj_dir, caption, filter_func=None, pe_tree=None, parent=None, query=None, allow_all_time_steps=False
    ):
        """Initializes the class.

        Args:
            proj_dir (:obj:`str`): Path to the project save location. If one exists, will be used to convert relative
                paths.
            caption (:obj:`str`): Caption text for file selector dialog when button is pressed.
            filter_func (:obj:`callable`): If provided, will be used to set filters in file selector dialog.
                Should return a filter string appropriate for use in a Qt file dialog given a table model index.
            pe_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): The XMS project explorer tree
            parent (:obj:`QObject`): The parent object.
            query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object for retrieving dataset dumps
            allow_all_time_steps (:obj:`bool`): If True, dataset selector will have option that allows user to select
                all time steps of a dataset
        """
        super().__init__(parent)
        self._parent = parent

        self.file_selector_delegate = FileSelectorButtonDelegate(proj_dir, caption, filter_func, parent)
        self.dataset_button_delegate = DatasetButtonDelegate(
            pe_tree=pe_tree,
            filter_method=DatasetSelector.is_scalar_if_dset,
            icon_method=resources_util.get_tree_icon_from_xms_typename,
            parent=parent,
            query=query,
            allow_all_time_steps=allow_all_time_steps
        )

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (:obj:`QWidget`): The editor widget.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        init_cond_type = initial_condition_parameter.get_init_cond_type(index)
        if init_cond_type == 'Restart File':
            self.file_selector_delegate.updateEditorGeometry(editor, option, index)
        elif init_cond_type == 'Water Surface Elevation Dataset':
            self.dataset_button_delegate.updateEditorGeometry(editor, option, index)
        else:
            super().updateEditorGeometry(editor, option, index)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        init_cond_type = initial_condition_parameter.get_init_cond_type(index)
        if init_cond_type == 'Dry':
            painter.fillRect(option.rect, QColor(240, 240, 240))  # Gray out the cell
            super().paint(painter, option, QModelIndex())
        elif init_cond_type == 'Initial Water Surface Elevation':
            value = index.model().data(index)
            if not io_util.is_float(value):
                index.model().setData(index, 0.0)
            super().paint(painter, option, index)
        elif init_cond_type == 'Automatic':
            painter.fillRect(option.rect, QColor(240, 240, 240))  # Gray out the cell
            super().paint(painter, option, QModelIndex())
        elif init_cond_type == 'Restart File':
            value = index.model().data(index)
            if io_util.is_float(value) or io_util.is_valid_uuid(value):
                index.model().setData(index, '')
            self.file_selector_delegate.paint(painter, option, index)
        elif init_cond_type == 'Water Surface Elevation Dataset':
            value = index.model().data(index)
            dset_uuid, _ = treeitem_selector_datasets.uuid_and_time_step_index_from_string(value)
            if not io_util.is_valid_uuid(dset_uuid):
                index.model().setData(index, '')
            self.dataset_button_delegate.paint(painter, option, index)
        else:
            super().paint(painter, option, index)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        init_cond_type = initial_condition_parameter.get_init_cond_type(index)
        if init_cond_type == 'Restart File':
            return self.file_selector_delegate.editorEvent(event, model, option, index)
        elif init_cond_type == 'Water Surface Elevation Dataset':
            return self.dataset_button_delegate.editorEvent(event, model, option, index)
        else:
            return super().editorEvent(event, model, option, index)

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates a QLineEdit widget for editing if the row uses constant WSE.

        Args:
            parent (:obj:`QWidget`): Qt parent of the new editor widget
            option (:obj:`QStyleOptionViewItem`): The style options
            index (:obj:`QModelIndex`): Index of the item being edited

        Returns:
            (:obj:`QWidget`): See description
        """
        init_cond_type = initial_condition_parameter.get_init_cond_type(index)
        if init_cond_type == 'Initial Water Surface Elevation':
            editor = QLineEdit(parent)
            editor.setValidator(QxDoubleValidator(parent=self._parent))
            return editor
        else:
            return super().createEditor(parent, option, index)
