"""XMS DMI component for migrating old SRH-2D SMS projects."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import traceback
import uuid

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.components.bases.migrate_base import MigrateBase
from xms.components.runners import migrate_runner as mgrun
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component, Coverage, FilterLocation, Simulation
from xms.guipy.param import param_h5_io

# 4. Local modules
from xms.srh.components.sim_component import SimComponent
from xms.srh.file_io.bc_component_builder import build_bc_component
from xms.srh.file_io.material_component_builder import MaterialComponentBuilder
from xms.srh.file_io.monitor_component_builder import build_monitor_component
from xms.srh.file_io.obstructions_component_builder import build_obstructions_component
from xms.srh.migrate import bc_att_migrate as bam
from xms.srh.migrate import material_cov_migrate as mcm
from xms.srh.migrate import obstruction_att_migrate as obm

ATT_DEFAULT_ID = -2


class MigrateSrh(MigrateBase):
    """Convert old SMS projects to component-based interface."""
    def __init__(self):
        """Constructor."""
        super().__init__()
        self._errors = []
        self._xms_data = None  # Old data to migrate. Get from Query or pass in for testing.
        self._old_takes = None  # passed in to first pass of xms.components migrate process
        self._old_items = None  # passed in to first pass of xms.components migrate process
        self._widgets = None  # passed in to first pass of xms.components migrate process
        self._mat_atts = None  # passed in to first pass of xms.components migrate process
        self._mat_polys = None  # passed in to first pass of xms.components migrate process
        self._hy8_widgets = None
        self._bc_monitor_lines = {}  # Monitor lines from old BC coverages: {old bc cov UUID: [xms.data_objects Arcs]}
        self._bc_monitor_merges = {}  # {old BC UUID: {old Monitor Points UUID: merged Monitor UUID}}
        self._linked_bc_covs = set()  # Need to check BC coverages that aren't linked to any sim for monitor lines.

        # Stuff we want to send back to SMS
        self._delete_uuids = []  # Things we want to delete: [uuid]
        self._new_sims = []  # [(do_simulation, do_component)]
        # This comment is not entirely true. If the new coverage is a monitor coverage that either did
        # not previously exist (old BC monitoring lines) or the monitor coverage is the product of a
        # merge with old monitor point coverages, the dict key will be the new coverage's UUID.
        self._new_covs = {}  # Old cov UUID: (cov data_object, covtype, comp data_object, comp unique_name)
        self._take_uuids = []  # [(taken_uuid, taker_uuid)]

    @staticmethod
    def _append_bc_monitor_lines(monitor_cov, bc_monitor_arcs):
        """Move monitor arcs from an old BC coverage to an existing Monitor coverage taken by the same simulation.

        Args:
            monitor_cov (:obj:`xms.data_objects.parameters.Coverage`): The existing Monitor coverage to append to.
            bc_monitor_arcs (:obj:`list`): List of the monitoring xms.data_objects.parameters.Arc that were on an old
                BC coverage.
        """
        if not monitor_cov or not bc_monitor_arcs:
            return

        # Find the maximum point id in existing Monitor coverage.
        next_pt_id = -1
        old_monitor_pts = monitor_cov.get_points(FilterLocation.PT_LOC_ALL)
        for monitor_pt in old_monitor_pts:
            next_pt_id = max(next_pt_id, monitor_pt.id)
        next_pt_id += 1
        if not next_pt_id:  # Existing coverage is empty.
            next_pt_id = 1

        # Find the maximum point id in existing Monitor coverage.
        next_arc_id = -1
        old_monitor_arcs = monitor_cov.arcs
        for monitor_arc in old_monitor_arcs:
            next_arc_id = max(next_arc_id, monitor_arc.id)
        next_arc_id += 1
        if not next_arc_id:  # Existing coverage is empty or only contained points.
            next_arc_id = 1

        # Renumber the points and arcs we are appending.
        old_to_new_pt_ids = {}
        for bc_monitor in bc_monitor_arcs:
            bc_monitor.id = next_arc_id
            next_arc_id += 1
            start_node = bc_monitor.start_node
            old_id = start_node.id
            if old_id not in old_to_new_pt_ids:
                old_to_new_pt_ids[old_id] = next_pt_id
                next_pt_id += 1
            start_node.id = old_to_new_pt_ids[old_id]
            bc_monitor.start_node = start_node
            end_node = bc_monitor.end_node
            old_id = end_node.id
            if old_id not in old_to_new_pt_ids:
                old_to_new_pt_ids[old_id] = next_pt_id
                next_pt_id += 1
            end_node.id = old_to_new_pt_ids[old_id]
            bc_monitor.end_node = end_node

        # Add the renumbered arcs to the Monitor coverage.
        monitor_cov.arcs = bc_monitor_arcs

    def _get_xms_data(self, query):
        """Get the XMS temp component directory for creating new components.

        ::
            self._xms_data = {
                'comp_dir': ''  # XMS temp component directory
                'cov_dumps': []  # Old coverages to migrate [(dump object, coveragetype)]
                'projection': xms.data_objects.parameters.Projection  # Projection of exported items (should be same)
                'proj_dir': str  # Path to the project's folder
            }

        Args:
            query (:obj:`xms.dmi.Query`): Object for communicating with SMS.
        """
        if self._xms_data:  # Initial XMS data provided (probably testing)
            return

        self._xms_data = {
            'comp_dir': '',
            'cov_dumps': [],
            'projection': None,
            'proj_dir': '',
            'map_tree_root': None,
        }
        try:
            self._xms_data['proj_dir'] = os.path.dirname(mgrun.project_db_file)
            # Get the SMS temp directory
            self._xms_data['comp_dir'] = os.path.join(os.path.dirname(query.xms_temp_directory), 'Components')
            self._get_coverages_to_migrate(query)
            # Get the Map Data tree root
            self._xms_data['map_tree_root'] = tree_util.first_descendant_with_name(query.project_tree, 'Map Data')
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not retrieve SMS data required for migration:\n"{traceback_msg}'))

    def _get_next_sim_comp_uuid(self):
        """Get a randomly generated UUID for a new sim component or hard-coded one for testing."""
        if 'new_sim_comp_uuids' in self._xms_data and self._xms_data['new_sim_comp_uuids']:
            return self._xms_data['new_sim_comp_uuids'].pop(0)
        return str(uuid.uuid4())

    def _get_next_cov_comp_uuid(self):
        """Get a randomly generated UUID for a new BC component or hard-coded one for testing."""
        if 'new_cov_comp_uuids' in self._xms_data and self._xms_data['new_cov_comp_uuids']:
            return self._xms_data['new_cov_comp_uuids'].pop(0)
        return str(uuid.uuid4())

    def _get_coverages_to_migrate(self, query):
        """Get dumps of all the coverages we need to migrate.

        Args:
            query (:obj:`xms.dmi.Query`): Object for communicating with SMS.
        """
        for item_uuid, info in self._old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == 'SRH-2D' and info[2] == 'Coverage':
                cov_dump = query.item_with_uuid(item_uuid)
                if not cov_dump:
                    continue
                self._xms_data['cov_dumps'].append((cov_dump, info[1]))
                self._xms_data['projection'] = cov_dump.projection

    def _get_hy8_widget_map(self):
        """Get dict of the HY-8 executable widget data keyed by hidden simulation id."""
        hy8_widgets = {}
        for item_uuid, info in self._old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == 'HY-8' and info[2] == 'Simulation':
                hy8_widgets[info[3]] = self._widgets[item_uuid]['Simulation'][-1]
        return hy8_widgets

    def _should_build_monitor_cov(self, old_uuid, new_uuid):
        """Determine if a monitor coverage should be built.

        Args:
            old_uuid (:obj:`str`): UUID of the old

        Returns:
            (:obj:`bool`): True if the migrated Monitor Coverage should be built
        """
        for take_pair in self._take_uuids:
            if new_uuid == take_pair[0]:
                return True  # Migrated coverage is taken by a simulation, need to build it.
        for old_takes in self._old_takes.values():
            if old_uuid in old_takes:
                return False  # Source coverage was taken, but no longer is (got merged). Don't build.
        return True  # Item isn't taken and never was. No merge possible. Create points-only coverage.

    def _migrate_unlinked_monitor_coverages(self, ):
        """Migrate any monitor coverages that were created from old BC coverages that were not linked to any sim."""
        # If the old BC coverage was taken by a simulation, it has already been created and removed from the map.
        for old_bc_uuid, bc_monitor_arcs in self._bc_monitor_lines.items():
            if old_bc_uuid in self._linked_bc_covs:
                continue  # This BC coverage is linked to at least one sim. It's potential monitor lines are migrated.

            new_bc_coverage = self._new_covs.get(old_bc_uuid, None)
            if not new_bc_coverage or not bc_monitor_arcs:
                continue  # Original coverage had no monitor arcs

            new_monitor_cov = self._get_do_coverage(f'{new_bc_coverage[0].name} - Monitor', arcs=bc_monitor_arcs)
            new_monitor_uuid = new_monitor_cov.uuid
            new_comp = build_monitor_component(
                new_monitor_uuid, self._xms_data['comp_dir'], self._get_next_cov_comp_uuid()
            )
            self._new_covs[new_monitor_uuid] = (new_monitor_cov, 'Monitor', new_comp, 'Monitor_Component')

    def _migrate_model_control(self, old_sim_uuid):
        """Migrate the simulation Model Control widgets.

        Args:
            old_sim_uuid (:obj:`str`): UUID of the old simulation

        Returns:
            (:obj:`xms.data_objects.parameters.Component, ModelControl, str`): The new simulation's hidden component,
            its data, and path to the main file where data should be written.
        """
        # Create a folder and UUID for the new sim component.
        comp_uuid = self._get_next_sim_comp_uuid()
        sim_comp_dir = os.path.join(self._xms_data['comp_dir'], comp_uuid)
        os.makedirs(sim_comp_dir, exist_ok=True)

        # Create the component data_object to send back to SMS.
        sim_main_file = os.path.join(sim_comp_dir, 'sim_comp.nc')
        sim_comp = Component(
            main_file=sim_main_file, model_name='SRH-2D', unique_name='Sim_Manager', comp_uuid=comp_uuid
        )

        # Fill component data from widget values in the old database.
        sim_py_comp = SimComponent(sim_main_file)  # Initialize some default data
        data = sim_py_comp.data
        try:
            if old_sim_uuid in self._widgets:
                sim_widgets = self._widgets[old_sim_uuid]['Simulation'][-1]

                # General options
                data.hydro.simulation_description = sim_widgets['edtSimDescription'][0][2]
                data.hydro.case_name = sim_widgets['edtCaseName'][0][2]
                data.hydro.start_time = sim_widgets['edtStartTime'][0][2]
                data.hydro.time_step = sim_widgets['edtTimeStep'][0][2]
                data.hydro.end_time = sim_widgets['edtSimulationTime'][0][2]
                initial_condition = sim_widgets['cbxInitialCondition'][0][2]
                data.hydro.initial_condition = initial_condition
                if 'edtInitWSE' in sim_widgets:
                    data.hydro.initial_water_surface_elevation = sim_widgets['edtInitWSE'][0][2]
                if 'cbxInitWSEUnits' in sim_widgets:
                    data.hydro.initial_water_surface_elevation_units = sim_widgets['cbxInitWSEUnits'][0][2]
                restart_file = sim_widgets['fileRST'][0][2]
                if restart_file and restart_file != '(none selected)':
                    restart_file = filesystem.resolve_relative_path(self._xms_data['proj_dir'], restart_file)
                    if os.path.isfile(restart_file):
                        comp_restart = os.path.join(os.path.dirname(sim_main_file), os.path.basename(restart_file))
                        filesystem.copyfile(restart_file, comp_restart)
                        data.hydro.restart_file = os.path.basename(restart_file)
                    else:
                        self._errors.append(('ERROR', 'Could not find restart file.'))

                # Advanced options
                data.advanced.turbulence_model = sim_widgets['cbxTurbulenceModel'][0][2]
                data.advanced.parabolic_turbulence = sim_widgets['edtParabolicTurbulence'][0][2]

                # Output options
                # XMDFC was an option that we removed because we always request the cell-centered solution from SRH
                # and srh_post interpolates to nodes. Clean up here if we find one.
                output_format = sim_widgets['cbxOutputFormat'][0][2]
                data.output.output_format = output_format if output_format != 'XMDFC' else 'XMDF'
                output_units = sim_widgets['cbxOutputUnit'][0][2]
                output_units = output_units if output_units != 'SI' else 'Metric'
                if output_units.upper() in ['SI', 'METRIC']:
                    output_units = 'Metric'
                elif output_units.upper() in ['EN', 'ENGLISH']:
                    output_units = 'English'
                data.output.output_units = output_units
                if 'cbxOutputMethod' in sim_widgets:
                    data.output.output_method = sim_widgets['cbxOutputMethod'][0][2]
                data.output.output_frequency = sim_widgets['edtOutputFrequency'][0][2]
                data.output.output_frequency_units = 'Hours'  # Always hours in XML widget interface
                if 'tblColTime' in sim_widgets and sim_widgets['tblColTime']:
                    data.output.output_specified_times = pd.DataFrame(
                        data={'Times': [row[2] for row in sim_widgets['tblColTime']]}
                    )
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not migrate SRH-2D model control:\n"{traceback_msg}'))

        return sim_comp, data, sim_main_file

    def _migrate_bc_to_model_control(self, sim_data, old_bc_uuid):
        """Migrate the simulation Model Control widgets.

        Args:
            sim_data (:obj:`ModelControl`): The new simulation's component data
            old_bc_uuid (:obj:`str`): UUID of the old BC coverage
        """
        try:
            if old_bc_uuid in self._widgets:
                bc_widgets = self._widgets[old_bc_uuid]['Coverage'][-1]
                if 'cbxRunType' not in bc_widgets:
                    return  # No sediment parameters in the database for this coverage. Use defaults.

                sim_data.enable_sediment = bc_widgets['cbxRunType'][0][2] == 'Mobile'
                sim_data.advanced.unsteady_output = bc_widgets['cbxUnsteadyOutput'][0][2] == 'On'
                sim_data.sediment.sediment_specific_gravity = bc_widgets['edtSedimentDensity'][0][2]
                if 'tblColDiameterThreshold' in bc_widgets and bc_widgets['tblColDiameterThreshold']:
                    sim_data.sediment.particle_diameter_threshold = pd.DataFrame(
                        data={
                            'Particle Diameter Threshold (mm)':
                                [row[2] for row in bc_widgets['tblColDiameterThreshold']]
                        }
                    )

                # Sediment transport equation parameters
                eq_params = sim_data.sediment.transport_equation_parameters
                lower_eq_params = eq_params.lower_transport_parameters
                higher_eq_params = eq_params.higher_transport_parameters
                eq_params.sediment_transport_equation = bc_widgets['cbxSedTransEquation'][0][2]
                eq_params.mixed_sediment_size_class_cutoff = bc_widgets['edtMixedSizeCut'][0][2]
                lower_eq_params.sediment_transport_equation = bc_widgets['cbxSedTransEquation1'][0][2]
                higher_eq_params.sediment_transport_equation = bc_widgets['cbxSedTransEquation2'][0][2]
                eq_params.meyer_peter_muller_hiding_factor = bc_widgets['edtMPMHidingCoeff'][0][2]
                lower_eq_params.meyer_peter_muller_hiding_factor = bc_widgets['edtMPMHidingCoeff1'][0][2]
                higher_eq_params.meyer_peter_muller_hiding_factor = bc_widgets['edtMPMHidingCoeff2'][0][2]
                eq_params.parker_reference_shields_parameter = bc_widgets['edtRefShieldsParam'][0][2]
                lower_eq_params.parker_reference_shields_parameter = bc_widgets['edtRefShieldsParam1'][0][2]
                higher_eq_params.parker_reference_shields_parameter = bc_widgets['edtRefShieldsParam2'][0][2]
                eq_params.parker_hiding_coefficient = bc_widgets['edtHidingCoeff'][0][2]
                lower_eq_params.parker_hiding_coefficient = bc_widgets['edtHidingCoeff1'][0][2]
                higher_eq_params.parker_hiding_coefficient = bc_widgets['edtHidingCoeff2'][0][2]
                eq_params.wilcox_t1_coefficient = bc_widgets['edtWilcockT1'][0][2]
                lower_eq_params.wilcox_t1_coefficient = bc_widgets['edtWilcockT11'][0][2]
                higher_eq_params.wilcox_t1_coefficient = bc_widgets['edtWilcockT12'][0][2]
                eq_params.wilcox_t2_coefficient = bc_widgets['edtWilcockT2'][0][2]
                lower_eq_params.wilcox_t2_coefficient = bc_widgets['edtWilcockT21'][0][2]
                higher_eq_params.wilcox_t2_coefficient = bc_widgets['edtWilcockT22'][0][2]
                eq_params.wilcox_sand_diameter = bc_widgets['edtWilcockD_Sand'][0][2]
                lower_eq_params.wilcox_sand_diameter = bc_widgets['edtWilcockD_Sand1'][0][2]
                higher_eq_params.wilcox_sand_diameter = bc_widgets['edtWilcockD_Sand2'][0][2]
                eq_params.wu_critical_shields_parameter = bc_widgets['edtWuCriticalShieldsParam'][0][2]
                lower_eq_params.wu_critical_shields_parameter = bc_widgets['edtWuCriticalShieldsParam1'][0][2]
                higher_eq_params.wu_critical_shields_parameter = bc_widgets['edtWuCriticalShieldsParam2'][0][2]

                # Sediment transport parameters not dependent on equation
                non_eq_params = sim_data.sediment.transport_parameters
                non_eq_params.water_temperature = bc_widgets['edtWaterTemp'][0][2]
                non_eq_params.deposition_coefficient = bc_widgets['edtDepCoeff'][0][2]
                non_eq_params.erosion_coefficient = bc_widgets['edtEroCoeff'][0][2]
                non_eq_params.adaptation_length_bedload_mode = bc_widgets['cbxModAdapLng'][0][2]
                non_eq_params.adaptation_length_bedload_length = bc_widgets['edtAdapLength'][0][2]
                layer_mode = bc_widgets['cbxActiveLayerMode'][0][2]
                layer_mode = layer_mode if layer_mode != 'Thickness Based on D90' else 'Thickness based on D90'
                non_eq_params.active_layer_thickness_mode = layer_mode
                if layer_mode == 'Thickness based on D90':
                    non_eq_params.active_layer_d90_scale = bc_widgets['edtActiveLyrThickness'][0][2]
                else:
                    non_eq_params.active_layer_constant_thickness = bc_widgets['edtActiveLyrThickness'][0][2]

                # Sediment transport cohesion parameters
                cohesion_params = sim_data.sediment.cohesive
                sim_data.sediment.enable_cohesive_sediment_modeling = bc_widgets['cbxCohesiveOnOff'][0][2] == 'On'
                fall_velocity = bc_widgets['cbxCohesiveFallVelocity'][0][2]
                fall_velocity = fall_velocity if fall_velocity != 'Data File' else 'Velocity Data File (mm/sec)'
                cohesion_params.fall_velocity = fall_velocity
                cohesion_params.fall_velocity_data_file = bc_widgets['edtCohesiveFallVelocityFile'][0][2]
                erosion_rate = bc_widgets['cbxCohesiveErosionRate'][0][2]
                if erosion_rate == 'Data File':
                    erosion_rate = 'Erosion Data File'
                else:  # erosion_rate == 'Input Parameters'
                    erosion_rate = 'Parameters'
                cohesion_params.erosion_rate = erosion_rate
                cohesion_params.erosion_rate_data_file = bc_widgets['edtCohesiveErosionRateFile'][0][2]
                cohesion_params.surface_erosion = bc_widgets['edtCohesiveCriticalShearErosion'][0][2]
                cohesion_params.mass_erosion = bc_widgets['edtCohesiveCriticalShearMassErosion'][0][2]
                cohesion_params.surface_erosion_constant = bc_widgets['edtCohesiveSurfaceErosionConstant'][0][2]
                cohesion_params.mass_erosion_constant = bc_widgets['edtCohesiveMassErosionConstant'][0][2]
                cohesion_params.erosion_units = bc_widgets['cbxCohesiveShearErosionUnits'][0][2]
                cohesion_params.full_depostion = bc_widgets['edtCohesiveCriticalShearFullDeposition'][0][2]
                cohesion_params.partial_depostion = bc_widgets['edtCohesiveCriticalShearPartialDeposition'][0][2]
                cohesion_params.equilibrium_concentration = bc_widgets['edtCohesiveEquilibriumConcentration'][0][2]
                cohesion_params.deposition_units = bc_widgets['cbxCohesiveShearDepositionUnits'][0][2]
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(
                (
                    'ERROR', f'Could not migrate SRH-2D sediment parameters from Boundary Conditions coverage to Model '
                    f'Control:\n"{traceback_msg}'
                )
            )

    def _migrate_sims(self):
        """Migrate simulations to current component based version.

        Returns:
            (:obj:`dict`): Mapping of old simulation to list containing the UUID of the new simulation that replaces it.
            Needed for second pass of migration.
        """
        replace_map = {}
        for item_uuid, info in self._old_items.items():
            # Look for SRH-2D simulations. Filter out non-visible simulations. They are garbage. Tree item names
            # cannot contain '*' characters. They are old mesh-based simulations from a previous migration.
            # info = (modelname, simname, entitytype, simid)
            if info[0] == 'SRH-2D' and info[2] == 'Simulation' and not info[1].startswith('*'):
                try:
                    # Create a new simulation and its hidden component
                    new_sim_uuid = str(uuid.uuid4())
                    new_sim = Simulation(model='SRH-2D', sim_uuid=new_sim_uuid, name=info[1])
                    replace_map[item_uuid] = [new_sim_uuid]
                    sim_comp, sim_data, main_file = self._migrate_model_control(item_uuid)

                    # Add the new simulation and its component to the Query.
                    self._new_sims.append((new_sim, sim_comp))

                    # Search for a taken items under the old simulation.
                    if item_uuid in self._old_takes:
                        new_monitor_uuid = ''
                        old_monitor_uuid = ''
                        old_bc_uuid = ''
                        bc_monitor_arcs = None
                        for take_uuid in self._old_takes[item_uuid]:
                            if take_uuid not in self._old_items:
                                continue

                            type_name = self._old_items[take_uuid][1]
                            if type_name == 'MESH2D':
                                # No migration needed for the mesh, just relink.
                                sim_data.mesh_uuid = take_uuid
                                new_take_uuid = take_uuid
                            else:
                                if take_uuid not in self._new_covs:
                                    continue

                                if type_name == 'Monitor Points':  # 0 or 1 per simulation
                                    # If we have an old Monitor Points coverage linked to us, append any monitoring
                                    # arcs that were in the linked BC coverage to the migrated Monitor coverage.
                                    old_monitor_uuid = take_uuid
                                    continue  # Don't take it yet, might have already built the merged coverage.
                                elif type_name == 'Boundary Conditions':  # 0 or 1 per simulation
                                    old_bc_uuid = take_uuid
                                    self._linked_bc_covs.add(old_bc_uuid)
                                    # Migrate widget values to Model Control that used to be on the BC coverage.
                                    self._migrate_bc_to_model_control(sim_data, take_uuid)
                                    # Monitoring arcs used to be on the BC coverage. Move them to a Monitor coverage.
                                    bc_monitor_arcs = self._bc_monitor_lines.get(take_uuid, None)

                                # Replace old take coverage with migrated one.
                                new_cov = self._new_covs[take_uuid][0]
                                new_take_uuid = new_cov.uuid

                            self._take_uuids.append((new_take_uuid, new_sim_uuid))

                        if bc_monitor_arcs:  # Had monitoring arcs in the old linked BC coverage
                            if old_monitor_uuid in self._bc_monitor_merges[old_bc_uuid]:
                                # Already migrated this BC-Monitor Points combo, link to simulation.
                                new_monitor_uuid = self._bc_monitor_merges[old_bc_uuid][old_monitor_uuid]
                            elif old_monitor_uuid:  # Had an old linked Monitor Points coverage
                                monitor_pt_cov = self._new_covs[old_monitor_uuid][0]
                                # Copy original monitor points to a new coverage and append BC monitor arcs.
                                new_cov = self._get_do_coverage(
                                    monitor_pt_cov.name,
                                    pts=monitor_pt_cov.get_points(FilterLocation.PT_LOC_ALL),
                                )
                                self._append_bc_monitor_lines(new_cov, bc_monitor_arcs)
                                new_monitor_uuid = new_cov.uuid
                                self._bc_monitor_merges[old_bc_uuid][old_monitor_uuid] = new_monitor_uuid
                                new_comp = build_monitor_component(
                                    new_monitor_uuid, self._xms_data['comp_dir'], self._get_next_cov_comp_uuid()
                                )
                                self._new_covs[new_monitor_uuid] = (new_cov, 'Monitor', new_comp, 'Monitor_Component')
                            else:  # Create a new Monitor coverage with just monitor arcs from this BC coverage
                                new_cov = self._get_do_coverage('Monitor', arcs=bc_monitor_arcs)
                                new_monitor_uuid = new_cov.uuid
                                new_comp = build_monitor_component(
                                    new_monitor_uuid, self._xms_data['comp_dir'], self._get_next_cov_comp_uuid()
                                )
                                self._new_covs[new_monitor_uuid] = (new_cov, 'Monitor', new_comp, 'Monitor_Component')
                        elif old_monitor_uuid:  # Had a linked monitor point coverage but no BC monitor arcs
                            new_monitor_uuid = self._new_covs[old_monitor_uuid][0].uuid

                        if new_monitor_uuid:  # Link in migrated monitor coverage last
                            self._take_uuids.append((new_monitor_uuid, new_sim_uuid))

                    # Write sim component data to main file.
                    param_h5_io.write_to_h5_file(main_file, sim_data)
                except Exception as ex:
                    traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                    self._errors.append(('ERROR', f'Could not migrate SRH-2D simulation:\n"{traceback_msg}'))

        return replace_map

    def _migrate_coverages(self):
        """Migrate all SRH-2D coverages.

        Returns:
            (:obj:`dict`): Mapping of old simulation to list containing the UUID of the new simulation that replaces it.
            Needed for second pass of migration.
        """
        replace_map = {}
        for old_cov in self._xms_data['cov_dumps']:
            try:
                cov_dump = old_cov[0]
                type_name = old_cov[1]
                new_uuids = []
                if type_name == 'Boundary Conditions':
                    new_uuids.append(self._migrate_bc_coverage(cov_dump))
                elif type_name == 'Materials':
                    new_uuids.append(self._migrate_material_coverage(cov_dump, False))
                elif type_name == 'Sediment Materials':
                    new_uuids.append(self._migrate_material_coverage(cov_dump, True))
                elif type_name == 'Obstructions':
                    new_uuids.append(self._migrate_obstruction_coverage(cov_dump))
                elif type_name in ['Monitor', 'Monitor Points']:
                    new_uuids.append(self._migrate_monitor_coverage(cov_dump))
                old_uuid = cov_dump.uuid
                self._delete_uuids.append(old_uuid)
                replace_map[old_uuid] = new_uuids
            except Exception as ex:
                traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                self._errors.append(('ERROR', f'Could not migrate SRH-2D coverage:\n"{traceback_msg}'))
        return replace_map

    def _get_do_coverage(self, name, pts=None, arcs=None, polys=None):
        """Get a data_objects Coverage to send back to SMS.

        Args:
            name (:obj:`str`): Name of the new coverage
            pts (:obj:`list`): List of disjoint xms.data_objects.parameters.Point locations in the new coverage, if any
            arcs (:obj:`list`): List of disjoint xms.data_objects.parameters.Arc definitions in the new coverage, if any
            polys (:obj:`list`):  List of xms.data_objects.parameters.Polygon definitions in the new coverage, if any

        Returns:
            (:obj:`xms.data_objects.parameters.Coverage`): The new coverage
        """
        new_cov = Coverage(name=name, uuid=str(uuid.uuid4()), projection=self._xms_data['projection'])
        if pts:
            new_cov.set_points(pts)
        if arcs:
            new_cov.arcs = arcs
        if polys:
            new_cov.polygons = polys
        new_cov.complete()
        return new_cov

    def _migrate_bc_coverage(self, cov_geom):
        """Migrate old BC coverages to new component-based coverages.

        Args:
            cov_geom (:obj:`xms.data_objects.parameters.Coverage`): Dump of the coverage geometry

        Returns:
            (:obj:`str`): UUID of the new coverage geometry
        """
        old_uuid = cov_geom.uuid
        self._bc_monitor_lines[old_uuid] = []
        self._bc_monitor_merges[old_uuid] = {}
        widget_map = self._widgets[old_uuid].get('Arc', {})
        set_map = self._widgets[old_uuid].get('Set', {})
        bam.add_sets_to_arc_db_data(widget_map, set_map)

        # Filter out arcs if they are monitoring lines (used to be in the BC coverage).
        monitor_is_default = (
            ATT_DEFAULT_ID in widget_map and widget_map[ATT_DEFAULT_ID]['cbxLineType'][0][2] == 'Monitor-Line'
        )
        old_arcs = cov_geom.arcs
        new_arcs = []
        for arc in old_arcs:
            arc_id = arc.id
            has_default_atts = arc_id not in widget_map
            if (monitor_is_default and has_default_atts) or \
               (not has_default_atts and widget_map[arc_id]['cbxLineType'][0][2] == 'Monitor-Line'):
                self._bc_monitor_lines[old_uuid].append(arc)
            else:
                new_arcs.append(arc)

        new_cov = self._get_do_coverage(cov_geom.name, arcs=new_arcs)
        new_cov_uuid = new_cov.uuid

        # Migrate arc/set widget values to coverage component's data.
        att_migrator = bam.MigrateBcAtts(self._xms_data['proj_dir'], new_arcs, widget_map, self._hy8_widgets)
        att_migrator.migrate()
        att_ids = list(att_migrator.bcs.keys())

        # Copy over the referenced HY-8 file if there is one.
        hy8_file = ''
        cov_atts = self._widgets[old_uuid]['Coverage'][-1]
        if 'fileHY8' in cov_atts:
            hy8_file = cov_atts['fileHY8'][0][2]
            if hy8_file and hy8_file != '(none selected)':
                # Might be stored as relative from the project database file, depending on version.
                hy8_file = filesystem.resolve_relative_path(self._xms_data['proj_dir'], hy8_file)

        new_comp = build_bc_component(
            att_migrator.bcs, new_cov_uuid, self._xms_data['comp_dir'], att_ids, hy8_file,
            self._get_next_cov_comp_uuid()
        )

        self._new_covs[old_uuid] = (new_cov, 'Boundary Conditions', new_comp, 'Bc_Component')
        return new_cov_uuid

    def _migrate_material_coverage(self, cov_geom, is_sediment):
        """Migrate old material coverages to new component-based coverages.

        Args:
            cov_geom (:obj:`xms.data_objects.parameters.Coverage`): Dump of the coverage geometry
            is_sediment (:obj:`bool`): True if the coverage we are migrating is a Sediment Materials type.

        Returns:
            (:obj:`str`): UUID of the new coverage geometry
        """
        old_uuid = cov_geom.uuid
        if old_uuid in self._widgets:
            widget_map = self._widgets[old_uuid].get('Material', {})
        else:
            widget_map = {}
        att_map = self._mat_atts.get(old_uuid, {})
        poly_assign_map = self._mat_polys.get(old_uuid, {})

        new_cov = self._get_do_coverage(
            cov_geom.name,
            pts=cov_geom.get_points(FilterLocation.PT_LOC_DISJOINT),
            arcs=cov_geom.arcs,
            polys=cov_geom.polygons
        )
        new_cov_uuid = new_cov.uuid

        # Migrate arc/set widget values to coverage component's data.
        att_migrator = mcm.MigrateMaterialCoverage(self._xms_data['proj_dir'], widget_map, att_map)
        att_migrator.migrate(is_sediment)

        comp_builder = MaterialComponentBuilder(
            self._xms_data['comp_dir'], att_migrator.mat_names, poly_assign_map, self._get_next_cov_comp_uuid(),
            att_migrator.mat_display
        )
        if is_sediment:
            new_comp = comp_builder.build_sed_material_component(new_cov_uuid, att_migrator.sed_mat_data)
            self._new_covs[old_uuid] = (new_cov, 'Sediment Materials', new_comp, 'SedMaterial_Component')
        else:
            new_comp = comp_builder.build_material_component(new_cov_uuid, att_migrator.manningsn)
            self._new_covs[old_uuid] = (new_cov, 'Materials', new_comp, 'Material_Component')
        return new_cov_uuid

    def _migrate_obstruction_coverage(self, cov_geom):
        """Migrate old obstruction coverages to new component-based coverages.

        Args:
            cov_geom (:obj:`xms.data_objects.parameters.Coverage`): Dump of the coverage geometry

        Returns:
            (:obj:`str`): UUID of the new coverage geometry
        """
        old_uuid = cov_geom.uuid
        arc_widget_map = self._widgets[old_uuid].get('Arc', {})
        point_widget_map = self._widgets[old_uuid].get('Point', {})
        obs_pts = cov_geom.get_points(FilterLocation.PT_LOC_DISJOINT)
        obs_arcs = cov_geom.arcs

        new_cov = self._get_do_coverage(cov_geom.name, pts=obs_pts, arcs=obs_arcs)
        new_cov_uuid = new_cov.uuid

        # Migrate point and arc widget values to coverage component's data.
        att_migrator = obm.MigrateObstructionAtts(obs_pts, point_widget_map, obs_arcs, arc_widget_map)
        att_migrator.migrate()
        new_comp = build_obstructions_component(
            att_migrator.obs_data, new_cov_uuid, self._xms_data['comp_dir'], att_migrator.pier_ids,
            att_migrator.deck_ids, self._get_next_cov_comp_uuid()
        )

        self._new_covs[old_uuid] = (new_cov, 'Obstructions', new_comp, 'Obstruction_Component')
        return new_cov_uuid

    def _migrate_monitor_coverage(self, cov_geom):
        """Migrate old Monitor and Monitor Points coverages to new component-based coverages.

        Args:
            cov_geom (:obj:`xms.data_objects.parameters.Coverage`): Dump of the coverage geometry

        Returns:
            (:obj:`str`): UUID of the new coverage geometry
        """
        old_uuid = cov_geom.uuid
        monitor_pts = cov_geom.get_points(FilterLocation.PT_LOC_DISJOINT)
        monitor_arcs = cov_geom.arcs
        new_cov = self._get_do_coverage(cov_geom.name, pts=monitor_pts, arcs=monitor_arcs)
        new_cov_uuid = new_cov.uuid

        # No need to migrate any atts because monitor did not have any atts until after version 0.0.11

        new_comp = build_monitor_component(new_cov_uuid, self._xms_data['comp_dir'], self._get_next_cov_comp_uuid())
        self._new_covs[old_uuid] = (new_cov, 'Monitor', new_comp, 'Monitor_Component')
        return new_cov.uuid

    def _build_coverage_tree_path(self, old_uuid):
        """Build the folder path a coverage should be placed into, if the old coverage was inside a folder.

        Args:
            old_uuid (:obj:`str`): UUID of the old coverage

        Returns:
            (:obj:`str`): The folder path the new coverage should have under its parent, or empty string if the old
            coverage was not in a folder.
        """
        map_root = self._xms_data.get('map_tree_root')
        tree_path = tree_util.build_tree_path(map_root, old_uuid)
        split_path = tree_path.split('/', 1)  # Remove 'Map Data' from the folder tree path
        if len(split_path) > 1:
            return os.path.dirname(split_path[1])  # Remove the coverage name from the folder tree path
        return ''

    def _remove_temp_monitor_covs(self):
        """If testing, clear temporary monitor coverages from output map."""
        didnt_build = []
        for old_uuid, new_cov in self._new_covs.items():
            # new_cov = (data_objects.parameters.Coverage, covtype, dat_objects.parameters.Component, comp_unique_name)
            cov_dump = new_cov[0]
            cov_type = new_cov[1]
            # Check for Monitor coverages that were linked to simulations, but have no uses of the original geometry
            # (all got merged into new coverage with BC monitor arcs). Do not build these.
            if cov_type == 'Monitor' and not self._should_build_monitor_cov(old_uuid, cov_dump.uuid):
                didnt_build.append(old_uuid)

        # Delete the component data directory.
        import shutil
        for unused_uuid in didnt_build:
            comp_dir = os.path.dirname(self._new_covs[unused_uuid][2].main_file)
            shutil.rmtree(comp_dir)
        # Remove the unused monitor coverages from the output map.
        self._new_covs = {
            old_uuid: new_cov
            for old_uuid, new_cov in self._new_covs.items() if old_uuid not in didnt_build
        }

    def map_to_typename(
        self, uuid_map, widget_map, take_map, main_file_map, hidden_component_map, material_atts, material_polys, query
    ):
        """This is the first pass of a two-pass system.

        This function performs all the data object migrations necessary for use between versions of model interfaces.

        Args:
            uuid_map (:obj:`dict{uuid : obj}`): This is a map from uuid's to object attributes.
                This is a collection of things you want to find the conversions for.
            widget_map (:obj:`dict{uuid : obj}`): This is a larger dictionary mapping uuid's to
                their conversions.
            take_map (:obj:`dict{uuid : obj}`): This is a map from uuid's of taking
                objects to a list of uuids of objects taken by the taker.
            main_file_map (:obj:`dict{uuid : str}`): This is a map from uuid's to main
                files of components.
            hidden_component_map (:obj:`dict{uuid : obj}`): This is a map from uuid's of owning
                objects to a list of uuids of hidden components owned by the object.
            material_atts (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is tuple of display options (name, r, g, b, alpha, texture)
            material_polys (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is a set of polygon ids with that material assignment.
            query (:obj:`xms.data_objects.parameters.Query`): This is used to communicate with XMS.
                Do not call the 'send' method on this instance as 'send' will be called later.

        Returns:
            (:obj:`dict{uuid  : obj}`): This is the map of delete and replacement requests.
        """
        if not query:  # Testing. Initialize XMS data.
            self._xms_data = mgrun.xms_migration_data

        self._old_items = uuid_map
        self._old_takes = take_map
        self._widgets = widget_map
        self._mat_atts = material_atts
        self._mat_polys = material_polys
        if query:
            self._get_xms_data(query)

        try:
            self._hy8_widgets = self._get_hy8_widget_map()
            # Migrate all SRH-2D coverages
            replace_map = self._migrate_coverages()
            # Migrate all SRH-2D simulations
            replace_map.update(self._migrate_sims())
            # Migrate Monitor coverages that were created from BC coverages that were never linked to a simulation.
            self._migrate_unlinked_monitor_coverages()
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not migrate SRH-2D project:\n"{traceback_msg}'))
            return {}

        return replace_map

    def send_replace_map(self, replace_map, query):
        """This is the second pass of a two-pass system.

        Args:
            replace_map (:obj:`dict{uuid : obj}`): This is the map of delete and replace requests.
                This can be generated by map_to_typename().
            query (:obj:`xms.data_objects.parameters.Query`): This is used to communicate with SMS.

        """
        if not query:
            self._remove_temp_monitor_covs()
            self._xms_data['new_covs'] = self._new_covs
            return  # testing

        # Add delete requests to the Query
        for delete_uuid in self._delete_uuids:
            query.delete_item(delete_uuid)

        # Add new coverages to the Query
        for old_uuid, new_cov in self._new_covs.items():
            # new_cov = (data_objects.parameters.Coverage, covtype, dat_objects.parameters.Component, comp_unique_name)
            cov_type = new_cov[1]
            cov_dump = new_cov[0]
            # Check for Monitor coverages that were linked to simulations, but have no uses of the original geometry
            # (all got merged into new coverage with BC monitor arcs). Do not build these.
            if cov_type == 'Monitor' and not self._should_build_monitor_cov(old_uuid, cov_dump.uuid):
                continue
            # Add the coverage geometry and its hidden component if there is one.
            components = [new_cov[2]] if new_cov[2] else None
            query.add_coverage(
                cov_dump,
                model_name='SRH-2D',
                coverage_type=cov_type,
                components=components,
                folder_path=self._build_coverage_tree_path(old_uuid)
            )

        # Add new simulations to the Query
        for new_sim in self._new_sims:
            query.add_simulation(new_sim[0], components=[new_sim[1]])

        # Link up takes to the new simulations
        for take_pair in self._take_uuids:
            query.link_item(taker_uuid=take_pair[1], taken_uuid=take_pair[0])

    def get_messages_and_actions(self):
        """Called at the end of migration.

        This is for when a message needs to be given to the user about some change due to migration.
        Also, an ActionRequest can be given if there is ambiguity in the migration.

        Returns:
            (:obj:`tuple(list[str], list[xms.api.dmi.ActionRequest])`):
                messages, action_requests - Where messages is a list of tuples with the first element of the tuple being
                the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                action_requests is a list of actions for XMS to perform.
        """
        if self._errors:
            messages = [('ERROR', 'Error(s) encountered during migration of SRH-2D project.')]
            messages.extend(self._errors)
        else:
            messages = [('INFO', 'Successfully migrated SRH-2D project.')]

        return messages, []
