"""The SRH-W Model Control dialog."""
# 1. Standard python modules
from datetime import datetime
import os

# 2. Third party modules
import pandas
from PySide2.QtWidgets import QDialog, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.api.tree import tree_util as tr_util
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs import treeitem_selector_datasets
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets import TreeItemSelectorDatasetsDlg
from xms.guipy.param.param_layout import ParamLayout
from xms.srh.gui.model_control_dialog import ModelControlDialog

# 4. Local modules
from xms.srhw.gui.table_dialog import TableDialog

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"


def watershed_layout():
    """Param layout for watershed fields."""
    param_layout = dict()
    param_layout['select_landuse_raster'] = ParamLayout(
        horizontal_layout='select_landuse_raster'
    )
    param_layout['landuse_raster_string'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_landuse_raster'
    )
    param_layout['select_soil_shapefile'] = ParamLayout(
        horizontal_layout='select_soil_shapefile'
    )
    param_layout['soil_shapefile_string'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_soil_shapefile'
    )
    param_layout['select_stream_coverage'] = ParamLayout(
        horizontal_layout='select_stream_coverage'
    )
    param_layout['stream_coverage_string'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_stream_coverage'
    )
    param_layout['select_bedrock_dataset'] = ParamLayout(
        horizontal_layout='select_bedrock_dataset'
    )
    param_layout['bedrock_dataset'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_bedrock_dataset'
    )
    param_layout['select_unsat_dataset'] = ParamLayout(
        horizontal_layout='select_unsat_dataset'
    )
    param_layout['unsat_dataset'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_unsat_dataset'
    )
    param_layout['select_sat_dataset'] = ParamLayout(
        horizontal_layout='select_sat_dataset'
    )
    param_layout['sat_dataset'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_sat_dataset'
    )
    param_layout['select_init_soil_temp_dataset'] = ParamLayout(
        horizontal_layout='select_init_soil_temp_dataset'
    )
    param_layout['init_soil_temp_dataset'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_init_soil_temp_dataset'
    )
    return param_layout


def _find_node_by_uuid(item, mesh_uuid):
    # Recursive function to find a node by UUID
    if item.uuid == mesh_uuid:
        return item
    for child in item.children:
        result = _find_node_by_uuid(child, mesh_uuid)
        if result:
            return result
    return None


class ModelControlDialogW(ModelControlDialog):
    """The SRH-W Model Control dialog."""

    def __init__(self, win_cont, sim_comp, project_tree, query):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (:obj:`QWidget`): Parent window
            sim_comp (:obj:`SimComponent`): SRH simulation component control class
            project_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): XMS project explorer tree
            query (:obj:`xms.api.dmi.Query`): Object for communicating with GMS
        """
        super().__init__(win_cont, sim_comp, project_tree, query)
        if self.mesh_tree is None:
            self.mesh_tree = self.trim_tree_to_ugrid(project_tree)
        self.setWindowTitle('SRH-W Model Control')
        self.widgets['tab_widget'].removeTab(3)
        self.widgets['watershed_tab'] = QWidget()
        self.widgets['tab_widget'].addTab(self.widgets['watershed_tab'], 'Watershed')
        defaults = land_use_defaults()
        self.land_use_data = pandas.DataFrame(defaults)
        land_use_file = os.path.join(os.path.dirname(self._main_file), 'land_use_table.json')
        num_columns = len(defaults)
        if os.path.isfile(land_use_file):
            temp_data = pandas.read_json(land_use_file)
            if len(temp_data.columns) == num_columns:
                self.land_use_data = temp_data
            else:
                self._logger.warning("The number of fields in the land use table is not correct. The land use table "
                                     "was reinitialized to use the default table.")
        # Always start dataframe index at 1 so the dialog works correctly
        self.land_use_data.index = range(1, len(self.land_use_data) + 1)
        defaults = soil_type_defaults()
        self.soil_type_data = pandas.DataFrame(defaults)
        soil_type_file = os.path.join(os.path.dirname(self._main_file), 'soil_type_table.json')
        num_columns = len(defaults)
        if os.path.isfile(soil_type_file):
            temp_data = pandas.read_json(soil_type_file)
            if len(temp_data.columns) == num_columns:
                self.soil_type_data = temp_data
            else:
                self._logger.warning("The number of fields in the soil type table is not correct. The soil type table "
                                     "was reinitialized to use the default table.")
        # Always start dataframe index at 1 so the dialog works correctly
        self.soil_type_data.index = range(1, len(self.soil_type_data) + 1)
        self.data.watershed.param_layout = watershed_layout()
        self.data.watershed.select_landuse_raster = lambda: self.select_landuse_raster()
        self.data.watershed.select_soil_shapefile = lambda: self.select_soil_shapefile()
        self.data.watershed.select_stream_coverage = lambda: self.select_stream_coverage()
        self.data.watershed.land_use_table_button = lambda: self.define_land_use_table()
        self.data.watershed.soil_type_table_button = lambda: self.define_soil_type_table()
        self.data.watershed.select_bedrock_dataset = lambda: self.select_bedrock_dataset()
        self.data.watershed.select_unsat_dataset = lambda: self.select_unsat_dataset()
        self.data.watershed.select_sat_dataset = lambda: self.select_sat_dataset()
        self.data.watershed.select_init_soil_temp_dataset = lambda: self.select_init_soil_temp_dataset()
        self.landuse_raster_names = self.data.watershed.landuse_raster_string.split('|')
        landuse_raster_items = []
        for name in self.landuse_raster_names:
            item = tr_util.item_from_path(project_tree, name)
            if item:
                landuse_raster_items.append(item)
        self.landuse_raster_uuids = []
        for item in landuse_raster_items:
            self.landuse_raster_uuids.append(item.uuid)
        self.soil_shape_names = self.data.watershed.soil_shapefile_string.split('|')
        soil_shape_items = []
        for name in self.soil_shape_names:
            item = tr_util.item_from_path(project_tree, name)
            if item:
                soil_shape_items.append(item)
        self.soil_shape_uuids = []
        for item in soil_shape_items:
            self.soil_shape_uuids.append(item.uuid)
        stream_coverage_name = self.data.watershed.stream_coverage_string
        stream_coverage_item = tr_util.item_from_path(project_tree, stream_coverage_name)
        self.stream_coverage_uuid = ''
        if stream_coverage_item:
            self.stream_coverage_uuid = stream_coverage_item.uuid
        self._set_layout('watershed_tab', 'watershed_tab_layout', QVBoxLayout())
        self.param_helper.add_params_to_layout(self.widgets['watershed_tab_layout'], self.data.watershed)
        self.param_helper.do_param_widgets(None)
        self.widgets['watershed_tab_layout'].addStretch()
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.bedrock_dataset_uuid = self.data.watershed.bedrock_dataset
        self.bedrock_widget = None
        self._accepted_called = True
        if self.data.watershed.param.bedrock_dataset in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.watershed.param.bedrock_dataset]
            self.bedrock_widget = items['value_widget']
            self.update_bedrock_widget()
        self.unsat_dataset_uuid = self.data.watershed.unsat_dataset
        self.unsat_widget = None
        self._accepted_called = True
        if self.data.watershed.param.unsat_dataset in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.watershed.param.unsat_dataset]
            self.unsat_widget = items['value_widget']
            self.update_unsat_widget()
        self.sat_dataset_uuid = self.data.watershed.sat_dataset
        self.sat_widget = None
        self._accepted_called = True
        if self.data.watershed.param.sat_dataset in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.watershed.param.sat_dataset]
            self.sat_widget = items['value_widget']
            self.update_sat_widget()
        self.init_soil_temp_dataset_uuid = self.data.watershed.init_soil_temp_dataset
        self.init_soil_temp_widget = None
        self._accepted_called = True
        if self.data.watershed.param.init_soil_temp_dataset in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.watershed.param.init_soil_temp_dataset]
            self.init_soil_temp_widget = items['value_widget']
            self.update_init_soil_temp_widget()

    def trim_tree_to_ugrid(self, project_tree):
        """Trims the project_tree down to the ugrid that is part of this simulation.

        Args:
            project_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): XMS project explorer tree

        Returns:
            (:obj:`xms.guipy.tree.tree_node.TreeNode`): The trimmed project explorer tree
        """
        if project_tree is None:
            return None
        if self.data.mesh_uuid:
            for child in project_tree.children:
                if child.name == 'UGrid Data':
                    items = child.children
                    for item in items:
                        result_item = _find_node_by_uuid(item, self.data.mesh_uuid)
                        if result_item:
                            return result_item
        return None

    def select_landuse_raster(self):
        """Select the Land Use dataset."""
        # Set the target_type to be an 'int' because the TreeItemSelectorDlg checks if the tree item is either
        # the target_type or one of the selectable_xms_types.  We never have a tree type of type 'int' so we're
        # just checking if the type is a TI_IMAGE.
        dialog = TreeItemSelectorDlg(title='Select Land Use Raster', target_type=int,
                                     pe_tree=self.project_tree, previous_selection=self.landuse_raster_uuids,
                                     selectable_xms_types=['TI_IMAGE'],
                                     show_root=True, parent=self, allow_multi_select=True)

        if dialog.exec() == QDialog.Accepted:
            self.landuse_raster_uuids = dialog.get_selected_item_uuid()
            txt = ''
            if self.landuse_raster_uuids:
                for uuid in self.landuse_raster_uuids:
                    txt += tr_util.build_tree_path(self.project_tree, uuid)
                    if uuid != self.landuse_raster_uuids[-1]:
                        txt += '|'
            if not txt:
                self.data.watershed.landuse_raster_string = 'No land use raster selected'
            else:
                self.data.watershed.landuse_raster_string = txt
            items = self.param_helper.param_dict[self.data.watershed.param.landuse_raster_string]
            widget = items['value_widget']
            widget.setText(self.data.watershed.landuse_raster_string)

    def select_soil_shapefile(self):
        """Select the Land Use dataset."""
        # Set the target_type to be an 'int' because the TreeItemSelectorDlg checks if the tree item is either
        # the target_type or one of the selectable_xms_types.  We never have a tree type of type 'int' so we're
        # just checking if the type is a TI_SHAPEFILE.
        dialog = TreeItemSelectorDlg(title='Select Soil Type Shapefile', target_type=int,
                                     pe_tree=self.project_tree, previous_selection=self.soil_shape_uuids,
                                     selectable_xms_types=['TI_GENERIC_POLY'],
                                     show_root=True, parent=self, allow_multi_select=True)

        if dialog.exec() == QDialog.Accepted:
            self.soil_shape_uuids = dialog.get_selected_item_uuid()
            txt = ''
            if self.soil_shape_uuids:
                for uuid in self.soil_shape_uuids:
                    txt += tr_util.build_tree_path(self.project_tree, uuid)
                    if uuid != self.soil_shape_uuids[-1]:
                        txt += '|'
            if not txt:
                self.data.watershed.soil_shapefile_string = 'No soil shapefile selected'
            else:
                self.data.watershed.soil_shapefile_string = txt
            items = self.param_helper.param_dict[self.data.watershed.param.soil_shapefile_string]
            widget = items['value_widget']
            widget.setText(self.data.watershed.soil_shapefile_string)

    def select_stream_coverage(self):
        """Select the Stream coverage."""
        # Set the target_type to be an 'int' because the TreeItemSelectorDlg checks if the tree item is either
        # the target_type or one of the selectable_xms_types.  We never have a tree type of type 'int' so we're
        # just checking if the type is a TI_COVER.
        dialog = TreeItemSelectorDlg(title='Select Stream Coverage', target_type=int,
                                     pe_tree=self.project_tree, previous_selection=self.stream_coverage_uuid,
                                     selectable_xms_types=['TI_COVER'],
                                     show_root=True, parent=self)

        if dialog.exec() == QDialog.Accepted:
            self.stream_coverage_uuid = dialog.get_selected_item_uuid()
            txt = ''
            if self.stream_coverage_uuid:
                txt = tr_util.build_tree_path(self.project_tree, self.stream_coverage_uuid)
            if not txt:
                self.data.watershed.stream_coverage_string = 'No stream coverage selected'
            else:
                self.data.watershed.stream_coverage_string = txt
            items = self.param_helper.param_dict[self.data.watershed.param.stream_coverage_string]
            widget = items['value_widget']
            widget.setText(self.data.watershed.stream_coverage_string)

    def define_land_use_table(self):
        """Define the Land Use table."""
        dialog = TableDialog(dialog_title='Define Land Use Table', data=self.land_use_data, parent=self)
        if dialog.exec() == QDialog.Accepted:
            self.land_use_data = dialog.df

    def _check_for_mesh(self):
        if not self.mesh_tree:
            msg = 'Error: unable to get mesh from SMS.'
            app_name = os.environ.get('XMS_PYTHON_APP_NAME')
            message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
            return False
        return True

    def _tree_item_selector(self, uuid_and_ts):
        uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_and_ts)
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select Dataset',
            pe_tree=self.mesh_tree,
            selected_dataset=uuid,
            selected_time_step=ts_idx,
            query=self._query,
            show_root=True,
            parent=self
        )
        if dialog.exec() == QDialog.DialogCode.Accepted:
            dset_uuid = dialog.get_selected_item_uuid()
            if dset_uuid:
                ts = dialog.get_selected_time_step_string()
                return True, dset_uuid, ts
        return False, None, -1

    def update_bedrock_widget(self):
        """Set the text of the bedrock dataset selector widget."""
        if self.bedrock_widget:
            uuid_str = self.bedrock_dataset_uuid
            uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_str)
            txt = tr_util.build_tree_path(self.mesh_tree, uuid)
            if not txt:
                self.data.watershed.bedrock_dataset = 'No Dataset selected'  # No valid dataset selected.
            else:
                self.data.watershed.bedrock_dataset = f'{txt}:{ts_idx + 1}'
            self.bedrock_widget.setText(self.data.watershed.bedrock_dataset)

    def select_bedrock_dataset(self):
        """Select the bedrock dataset."""
        if self._check_for_mesh():
            uuid_and_ts = self.data.watershed.bedrock_dataset
            dlg_ok, dset_uuid, ts = self._tree_item_selector(uuid_and_ts)
            if dlg_ok:
                uuid_and_ts = f'{dset_uuid}{ts}'  # Store a string with both the uuid and the time step index
                self.bedrock_dataset_uuid = uuid_and_ts
                self.data.hydro.bedrock_dataset = tr_util.build_tree_path(self.mesh_tree, dset_uuid)
                self.update_bedrock_widget()

    def update_unsat_widget(self):
        """Set the text of the unsat dataset selector widget."""
        if self.unsat_widget:
            uuid_str = self.unsat_dataset_uuid
            uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_str)
            txt = tr_util.build_tree_path(self.mesh_tree, uuid)
            if not txt:
                self.data.watershed.unsat_dataset = 'No Dataset selected'  # No valid dataset selected.
            else:
                self.data.watershed.unsat_dataset = f'{txt}:{ts_idx + 1}'
            self.unsat_widget.setText(self.data.watershed.unsat_dataset)

    def select_unsat_dataset(self):
        """Select the unsat dataset."""
        if self._check_for_mesh():
            uuid_and_ts = self.data.watershed.unsat_dataset
            dlg_ok, dset_uuid, ts = self._tree_item_selector(uuid_and_ts)
            if dlg_ok:
                uuid_and_ts = f'{dset_uuid}{ts}'  # Store a string with both the uuid and the time step index
                self.unsat_dataset_uuid = uuid_and_ts
                self.data.hydro.unsat_dataset = tr_util.build_tree_path(self.mesh_tree, dset_uuid)
                self.update_unsat_widget()

    def update_sat_widget(self):
        """Set the text of the sat dataset selector widget."""
        if self.sat_widget:
            uuid_str = self.sat_dataset_uuid
            uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_str)
            txt = tr_util.build_tree_path(self.mesh_tree, uuid)
            if not txt:
                self.data.watershed.sat_dataset = 'No Dataset selected'  # No valid dataset selected.
            else:
                self.data.watershed.sat_dataset = f'{txt}:{ts_idx + 1}'
            self.sat_widget.setText(self.data.watershed.sat_dataset)

    def select_sat_dataset(self):
        """Select the sat dataset."""
        if self._check_for_mesh():
            uuid_and_ts = self.data.watershed.sat_dataset
            dlg_ok, dset_uuid, ts = self._tree_item_selector(uuid_and_ts)
            if dlg_ok:
                uuid_and_ts = f'{dset_uuid}{ts}'  # Store a string with both the uuid and the time step index
                self.sat_dataset_uuid = uuid_and_ts
                self.data.hydro.sat_dataset = tr_util.build_tree_path(self.mesh_tree, dset_uuid)
                self.update_sat_widget()

    def update_init_soil_temp_widget(self):
        """Set the text of the init_soil_temp dataset selector widget."""
        if self.init_soil_temp_widget:
            uuid_str = self.init_soil_temp_dataset_uuid
            uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_str)
            txt = tr_util.build_tree_path(self.mesh_tree, uuid)
            if not txt:
                self.data.watershed.init_soil_temp_dataset = 'No Dataset selected'  # No valid dataset selected.
            else:
                self.data.watershed.init_soil_temp_dataset = f'{txt}:{ts_idx + 1}'
            self.init_soil_temp_widget.setText(self.data.watershed.init_soil_temp_dataset)

    def select_init_soil_temp_dataset(self):
        """Select the init_soil_temp dataset."""
        if self._check_for_mesh():
            uuid_and_ts = self.data.watershed.init_soil_temp_dataset
            dlg_ok, dset_uuid, ts = self._tree_item_selector(uuid_and_ts)
            if dlg_ok:
                uuid_and_ts = f'{dset_uuid}{ts}'  # Store a string with both the uuid and the time step index
                self.init_soil_temp_dataset_uuid = uuid_and_ts
                self.data.hydro.init_soil_temp_dataset = tr_util.build_tree_path(self.mesh_tree, dset_uuid)
                self.update_init_soil_temp_widget()

    def define_soil_type_table(self):
        """Define the Land Use table."""
        dialog = TableDialog(dialog_title='Define Soil Type Table', data=self.soil_type_data, parent=self)
        if dialog.exec() == QDialog.Accepted:
            self.soil_type_data = dialog.df

    def accept(self):
        """Dialog accept."""
        # Check to make sure the date is in a valid format.\
        try:
            date_time_value = datetime.strptime(self.data.watershed.start_date_time, '%Y-%m-%d %H:%M')
            test_value = date_time_value.strftime('%Y-%m-%d %H:%M')
            if test_value == self.data.watershed.start_date_time:
                super().accept()
                land_use_file = os.path.join(os.path.dirname(self._main_file), 'land_use_table.json')
                self.land_use_data.to_json(land_use_file)
                soil_type_file = os.path.join(os.path.dirname(self._main_file), 'soil_type_table.json')
                self.soil_type_data.to_json(soil_type_file)
                self.data.watershed.bedrock_dataset = self.bedrock_dataset_uuid
                self.data.watershed.unsat_dataset = self.unsat_dataset_uuid
                self.data.watershed.sat_dataset = self.sat_dataset_uuid
                self.data.watershed.init_soil_temp_dataset = self.init_soil_temp_dataset_uuid
                self._accepted_called = True
                return
        except ValueError:
            pass
        msg = 'Error: Date/Time is not in the correct format.'
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())


def soil_type_defaults():
    """Returns the default soil type table.

    Returns:
        (:obj:`dict`): Dictionary of lists with the default soil data definitions. The soil parameter names are
         the keys.
    """
    return {'ID': [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23],
            'Name': ['Sand', 'Loamy sand', 'Sandy loam', 'Loam', 'Silt loam', 'Sandy clay loam',
                     'Clay loam', 'Silty clay loam', 'Sandy clay', 'Silty clay', 'Clay',
                     'Fine sandy loam', 'Gravelly sand', 'Stratified sandy loam to clay', 'Bedrock',
                     'Gravelly clay loam', 'Gravelly loam', 'Very cobbly clay loam',
                     'Very cobbly loam', 'Very gravelly loam', 'Extremely gravelly loamy fine sand',
                     'Cobbly loam', 'Cobbly silt loam'],
            'Vertical Hydraulic Conductivity': [1.00E-05, 1.40E-05, 2.00E-05, 2.00E-08, 1.00E-05,
                                                1.00E-07, 1.00E-05, 1.00E-05, 1.00E-05, 1.00E-05,
                                                1.00E-05, 1.00E-05, 1.00E-05, 1.00E-05, 2.00E-08,
                                                2.00E-08, 2.00E-08, 2.00E-08, 2.00E-08, 2.00E-08,
                                                2.00E-08, 2.00E-08, 2.00E-08],
            'Lateral Hydraulic Conductivity': [1.60E-05, 8.00E-06, 8.00E-06, 8.00E-06, 1.60E-05,
                                               1.20E-05, 1.60E-05, 1.60E-05, 1.60E-05, 1.60E-05,
                                               1.60E-05, 1.60E-05, 1.60E-05, 1.60E-05, 8.00E-06,
                                               8.00E-06, 8.00E-06, 8.00E-06, 8.00E-06, 8.00E-06,
                                               8.00E-06, 8.00E-06, 8.00E-06],
            'Vertical Depth': [0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025,
                               0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025, 0.025,
                               0.025, 0.025, 0.025],
            'Porosity': [0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41,
                         0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41],
            'Field Capacity': [0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41,
                               0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41, 0.41],
            'Residual Soil Moisture': [0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01,
                                       0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01, 0.01,
                                       0.01, 0.01, 0.01],
            'Alpha': [0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2,
                      0.2, 0.2, 0.2, 0.2, 0.2, 0.2, 0.2],
            'Beta': [6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6],
            'Sigmoid X': [0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7,
                          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7],
            'Sigmoid N': [16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
                          16, 16, 16, 16],
            'Psi Min': [-70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70,
                        -70, -70, -70, -70, -70, -70, -70, -70],
            'Macro Pore Depth': [0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25,
                                 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25,
                                 0.25],
            'Macro Pore Hydraulic Conductivity': [5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03,
                                                  5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03,
                                                  5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03,
                                                  5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03, 5.00E-03,
                                                  5.00E-03, 5.00E-03, 5.00E-03],
            'Macro Pore Volume Fraction': [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                                           0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            'Volume Specific Heat': [1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06,
                                     1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06,
                                     1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06,
                                     1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06, 1.00E+06],
            'Ice Latent Heat': [4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06,
                                4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06,
                                4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06, 4.11E+06,
                                4.11E+06, 4.11E+06],
            'Soil Thermal Conductivity': [0.797, 0.797, 0.797, 0.797, 0.797, 0.797, 0.797, 0.797,
                                          0.797, 0.797, 0.797, 0.797, 0.797, 0.797, 0.797, 0.797,
                                          0.797, 0.797, 0.797, 0.797, 0.797, 0.797, 0.797],
            'Damping Coefficient': [-2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1,
                                    -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1, -2.1,
                                    -2.1],
            'Ts Depth': [0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                         0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1]
            }


def land_use_defaults():
    """Returns the default land use table.

    Returns:
        (:obj:`dict`): Dictionary of lists with the default land use definitions. The land use parameter names are
         the keys.
    """
    return {'ID': [11, 12, 21, 22, 23, 24, 31, 41, 42, 43, 52, 71, 81, 82, 90, 95],
            'Name': ['Open Water', 'Perennial Ice/Snow', 'Developed, Open Space',
                     'Developed, Low Intensity', 'Developed, Medium Intensity',
                     'Developed High Intensity', 'Barren Land (Rock/Sand/Clay)', 'Deciduous Forest',
                     'Evergreen Forest', 'Mixed Forest', 'Shrub/Scrub', 'Grassland/Herbaceous Arid',
                     'Pasture/Hay Fair', 'Cultivated Crops', 'Woody Wetlands',
                     'Emergent Herbaceous Wetlands'],
            "Manning's Roughness": [0.03, 0.03, 0.04, 0.03, 0.01, 0.04, 0.035, 0.08, 0.07, 0.06, 0.04,
                                    0.04, 0.02, 0.04, 0.04, 0.04],
            'Leaf Area Index': [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                                0.0, 0.0],
            'Shade Factor': [0.6, 0.6, 0.3, 0.27, 0.11, 0.02, 0.01, 0.8, 0.95, 0.8, 0.7, 0.6, 0.8, 0.8,
                             0.6, 0.6],
            'Min Albedo': [0.12, 0.12, 0.2, 0.2, 0.2, 0.2, 0.38, 0.14, 0.12, 0.17, 0.22, 0.14, 0.17,
                           0.2, 0.14, 0.12],
            'Max Albedo': [0.12, 0.12, 0.2, 0.2, 0.2, 0.2, 0.38, 0.17, 0.12, 0.25, 0.3, 0.14, 0.23,
                           0.25, 0.14, 0.12],
            'Vegetation Height': [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                                  0.0, 0.0],
            'Root Depth': [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                           0.0],
            'Crop Coefficient': [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                                 0.0, 0.0],
            'Canopy Cover Coefficient': [0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.1, 0.001,
                                         0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001],
            'Surface Depression Storage Volume': [2.40E-02, 2.40E-02, 2.40E-02, 2.40E-02, 2.40E-02,
                                                  2.40E-02, 2.40E-02, 3.00E-02, 2.40E-02, 2.40E-02,
                                                  2.40E-02, 2.40E-02, 2.40E-02, 2.40E-02, 2.40E-02,
                                                  2.40E-02]}
