"""Module for exporting a CMS-Wave simulation with feedback."""

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.stwave.file_io.dataset_writer import export_all_datasets, retrieve_xms_data
from xms.stwave.file_io.sim_writer import SimWriter
from xms.stwave.file_io.spectral_coverage_writer import SpectralCoverageWriter


class ExportSim:
    """Class for exporting a CMS-Wave simulation in a worker thread."""

    def __init__(self, xms_data, template=False):
        """Constructor.

        Args:
            xms_data (dict): dict with data from xms
            template (bool): If True will export a template for CSTORM
        """
        self._logger = logging.getLogger('xms.stwave')
        self._xms_data = xms_data
        self._template = template

    def set_logger(self, logger):
        """Set the logger to something different than the default.

        Args:
            logger (logging.Logger): the logger
        """
        self._logger = logger

    def export(self):
        """Exports the simulation."""
        self._logger.info('Exporting STWAVE control file...')
        sim_writer = SimWriter(xms_data=self._xms_data, template=self._template)
        try:
            do_grid = sim_writer.write()
            if do_grid is None:
                return  # Error has already been logged
        except Exception:
            self._logger.exception('Error! Could not export STWAVE control file.')

        self._logger.info('Exporting STWAVE spatial datasets...')
        try:
            self._logger.info('Retrieving spatial datasets from SMS...')
            xms_data = retrieve_xms_data(sim_writer.query, True, self._xms_data)
            self._logger.info('Writing spatial datasets to specified location...')
            export_all_datasets(xms_data)
        except Exception:
            self._logger.exception('Error! Could not export STWAVE spatial datasets.')

        self._logger.info('Exporting STWAVE spectral coverage file...')
        try:
            spec_writer = SpectralCoverageWriter(query=sim_writer.query, sim_export=True, xms_data=self._xms_data)
            spec_writer.export_coverage()
        except Exception:
            self._logger.exception('Error! Could not export STWAVE spectral coverage file.')
