"""This is a dialog for specifying STWAVE model control values."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import webbrowser

# 2. Third party modules
import numpy as np
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QHeaderView, QMessageBox
import xarray as xr

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import FilterLocation, julian_to_datetime
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.resources.resources_util import get_tree_icon_from_xms_typename
from xms.guipy.time_format import datetime_to_string, ISO_DATETIME_FORMAT, string_to_datetime
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator
from xms.guipy.widgets import widget_builder as wbd
from xms.guipy.widgets.twod_rect_grid_preview_plot import TwoDRectGridPlot

# 4. Local modules
from xms.stwave.data import simulation_data
from xms.stwave.data import stwave_consts as const
from xms.stwave.gui import gui_util
from xms.stwave.gui.model_control_dialog_ui import Ui_ModelControlDialog
from xms.stwave.gui.ref_time_dialog import RefTimeDialog
from xms.stwave.gui.widgets.case_data_table import CaseDataTable


class ModelControlDialog(XmsDlg):
    """A dialog for viewing model control data for a simulation."""

    def __init__(self, data, pe_tree, spec_cov, time_format, grid_data, parent=None):
        """
        Initializes the class, sets up the ui.

        Args:
            data (:obj:`SimulationData`): The simulation data for the dialog.
            pe_tree (:obj:`TreeNode`): The project explorer tree from XMS.
            spec_cov (Optional[xms.coverage.spectral.SpectralCoverage]):  The spectral coverage.
            time_format (str): Format to use for display of datetimes. Needs to use Qt specifiers.
            grid_data (dict): Domain CGrid data:
                {
                    'cogrid': CoGrid,
                    'grid_name': str,
                }
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent, 'stwave.gui.model_control_dialog')
        self._data = data
        self._previous_friction = self._data.info.attrs['friction']
        self._time_format = time_format
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:STWAVE'

        self._location_coverage = tree_util.copy_tree(pe_tree)
        self._scalar_dataset = tree_util.copy_tree(pe_tree)
        self._vector_dataset = tree_util.copy_tree(pe_tree)
        self._output_stations_coverage = tree_util.copy_tree(pe_tree)
        self._monitor_coverage = tree_util.copy_tree(pe_tree)
        self._nesting_coverage = tree_util.copy_tree(pe_tree)
        self._spec_cov = spec_cov
        self._grid_data = grid_data
        self.simulation_grid = tree_util.trim_project_explorer(self._scalar_dataset, self._data.info.grid_uuid)

        self._double_validator = QxDoubleValidator(parent=self)
        self._int_validator = QxIntValidator(parent=self)

        self.ui = Ui_ModelControlDialog()  # "Public" because accessed by the case data table
        self.ui.setupUi(self)
        self._setup_dialog()

    def _setup_dialog(self):
        """Called when the dialog is loaded to set up the initial values in the dialog."""
        # Setup the tabs on the dialog, initializing values.
        self._setup_parameters_tab()
        self._setup_boundary_tab()
        self._setup_output_tab()
        self._setup_iterations_tab()

        # Setup the connections for the widgets on the tabs.
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self._parameters_tab_connections()
        self._boundary_tab_connections()
        self._output_tab_connections()

        # Initialize the state of the widget dependencies
        self._initialize_state()

    def _setup_parameters_tab(self):
        """Called to setup the parameters tab when the dialog is loaded."""
        self.ui.plane_combo.setCurrentText(self._data.info.attrs['plane'])
        self.ui.source_terms_combo.setCurrentText(self._data.info.attrs['source_terms'])
        self.ui.depth_combo.setCurrentText(self._data.info.attrs['depth'])
        self.ui.depth_button.clicked.connect(
            lambda: DatasetSelector.select_dataset(self, self.ui.depth_label,
                                                   'Select depth dataset', self.simulation_grid,
                                                   DatasetSelector.is_scalar_if_dset, self._data.info.attrs,
                                                   'depth_uuid', self.icon_connected))
        self.ui.depth_label.setText(tree_util.build_tree_path(self._scalar_dataset,
                                                              self._data.info.attrs['depth_uuid']))
        self.ui.current_interaction_combo.setCurrentText(self._data.info.attrs['current_interaction'])
        self.ui.current_interaction_button.clicked.connect(
            lambda: DatasetSelector.select_dataset(self, self.ui.current_interaction_label,
                                                   'Select current interaction dataset', self.simulation_grid,
                                                   DatasetSelector.is_vector_if_dset, self._data.info.attrs,
                                                   'current_uuid', self.icon_connected))
        self.ui.current_interaction_label.setText(
            tree_util.build_tree_path(self._vector_dataset, self._data.info.attrs['current_uuid']))
        self.ui.friction_combo.setCurrentText(self._data.info.attrs['friction'])
        self.ui.friction_button.clicked.connect(self._on_friction_button)
        if const.FRIC_OPT_JONSWAP_CONST == self._data.info.attrs['friction']:
            self.ui.friction_edit.setText(str(self._data.info.attrs['JONSWAP']))
        elif const.FRIC_OPT_MANNING_CONST == self._data.info.attrs['friction']:
            self.ui.friction_edit.setText(str(self._data.info.attrs['manning']))
        else:
            self.ui.friction_edit.setText('0.0')
        self.ui.friction_edit.setValidator(self._double_validator)
        if const.FRIC_OPT_JONSWAP_DSET == self._data.info.attrs['friction']:
            self.ui.friction_label.setText(tree_util.build_tree_path(self._scalar_dataset,
                                                                     self._data.info.attrs['JONSWAP_uuid']))
        elif const.FRIC_OPT_MANNING_DSET == self._data.info.attrs['friction']:
            self.ui.friction_label.setText(tree_util.build_tree_path(self._scalar_dataset,
                                                                     self._data.info.attrs['manning_uuid']))
        self.ui.surge_combo.setCurrentText(self._data.info.attrs['surge'])
        self.ui.surge_button.clicked.connect(
            lambda: DatasetSelector.select_dataset(self, self.ui.surge_label,
                                                   'Select surge fields dataset', self.simulation_grid,
                                                   DatasetSelector.is_scalar_if_dset, self._data.info.attrs,
                                                   'surge_uuid', self.icon_connected))
        self.ui.surge_label.setText(tree_util.build_tree_path(self._scalar_dataset,
                                                              self._data.info.attrs['surge_uuid']))
        self.ui.wind_combo.setCurrentText(self._data.info.attrs['wind'])
        self.ui.wind_button.clicked.connect(
            lambda: DatasetSelector.select_dataset(self, self.ui.wind_label,
                                                   'Select wind fields dataset', self.simulation_grid,
                                                   DatasetSelector.is_vector_if_dset, self._data.info.attrs,
                                                   'wind_uuid', self.icon_connected))
        self.ui.wind_label.setText(tree_util.build_tree_path(self._vector_dataset,
                                                             self._data.info.attrs['wind_uuid']))
        self.ui.ice_combo.setCurrentText(self._data.info.attrs['ice'])
        self.ui.ice_button.clicked.connect(
            lambda: DatasetSelector.select_dataset(self, self.ui.ice_label,
                                                   'Select ice fields dataset', self.simulation_grid,
                                                   DatasetSelector.is_scalar_if_dset, self._data.info.attrs,
                                                   'ice_uuid', self.icon_connected))
        self.ui.ice_threshold_edit.setText(str(self._data.info.attrs['ice_threshold']))
        self.ui.ice_threshold_edit.setValidator(self._double_validator)
        self.ui.ice_label.setText(tree_util.build_tree_path(self._scalar_dataset, self._data.info.attrs['ice_uuid']))
        self.ui.processors_i_edit.setText(str(self._data.info.attrs['processors_i']))
        self.ui.processors_i_edit.setValidator(self._int_validator)
        self.ui.processors_j_edit.setText(str(self._data.info.attrs['processors_j']))
        self.ui.processors_j_edit.setValidator(self._int_validator)

    def _setup_boundary_tab(self):
        """Called to setup the boundary control tab when the dialog is loaded."""
        self.ui.boundary_source_combo.setCurrentText(self._data.info.attrs['boundary_source'])
        self.ui.interpolation_combo.setCurrentText(self._data.info.attrs['interpolation'])
        self.ui.num_frequencies_edit.setText(str(self._data.info.attrs['num_frequencies']))
        self.ui.num_frequencies_edit.setValidator(self._int_validator)
        self.ui.delta_frequency_edit.setText(str(self._data.info.attrs['delta_frequency']))
        self.ui.delta_frequency_edit.setValidator(self._double_validator)
        self.ui.min_frequency_edit.setText(str(self._data.info.attrs['min_frequency']))
        self.ui.min_frequency_edit.setValidator(self._double_validator)
        self.ui.location_coverage_check.setChecked(self._data.info.attrs['location_coverage'] != 0)
        self.ui.angle_convention_combo.setCurrentText(self._data.info.attrs['angle_convention'])
        self._reftime_label_update()
        self.ui.btn_populate_from_spectra.clicked.connect(self._set_populate_from_spectra)
        self.ui.btn_reftime.clicked.connect(self._set_reftime)
        self.ui.location_coverage_label.setText(
            tree_util.build_tree_path(self._location_coverage, self._data.info.attrs['location_coverage_uuid']))
        self._setup_case_data_table()
        self._setup_source_sides()
        self._setup_grid_preview()
        wbd.style_splitter(self.ui.splitter_horiz)
        wbd.style_splitter(self.ui.splitter_vert)

    def _setup_case_data_table(self):
        """Create the case data table and add it to the layout."""
        reftime = datetime.datetime.strptime(self._data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
        self._case_data_table = CaseDataTable(reftime, self._data.case_times.to_dataframe(), self)
        self.ui.case_data_group.layout().addWidget(self._case_data_table)
        self._case_data_table.table_view.setColumnWidth(0, 150)
        self._case_data_table.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)

    def _setup_source_sides(self):
        """Load data into the source side widgets."""
        # Look for the first side that is specified. Can only be one in half plane. If full plane, the sides
        # adjacent to a specified side are always open lateral and the side opposite of it can either be specified
        # or zero spectrum.
        side_attrs = ['side1', 'side2', 'side3', 'side4']
        specified_side = 'side1'
        for side_attr in side_attrs:
            if self._data.info.attrs[side_attr] == const.I_BC_SPECIFIED:
                specified_side = side_attr
                break

        # Set the half plane combo, there should only be one (side 1 by default).
        self.ui.half_source_side_combo.setCurrentIndex(side_attrs.index(specified_side))

        # Set up the full plane combos, there may be up to two but there are restrictions on which two they can be.
        if specified_side in ['side1', 'side3']:  # If side 1 and/or 3 is specified, the others are open lateral.
            self.ui.full_source_side_combo.setCurrentText(const.OPT_SIDES1_AND_3)
            side_a_type = self._data.info.attrs['side1']
            side_b_type = self._data.info.attrs['side3']
        else:  # Otherwise, sides 1 and 3 are open lateral and 2 and/or 4 are specified.
            side_a_type = self._data.info.attrs['side2']
            side_b_type = self._data.info.attrs['side4']
            self.ui.full_source_side_combo.setCurrentText(const.OPT_SIDES2_AND_4)
        if side_a_type != const.I_BC_SPECIFIED and side_b_type != const.I_BC_SPECIFIED:
            side_a_type = const.I_BC_SPECIFIED  # Ensure that at least one side is specified
        self.ui.full_side1_combo.setCurrentText(
            side_a_type if side_a_type in [const.I_BC_ZERO, const.I_BC_SPECIFIED] else const.I_BC_ZERO
        )
        self.ui.full_side2_combo.setCurrentText(
            side_b_type if side_b_type in [const.I_BC_ZERO, const.I_BC_SPECIFIED] else const.I_BC_ZERO
        )

    def _setup_grid_preview(self):
        """Sets up the grid preview plot."""
        cogrid = self._grid_data.get('cogrid')
        grid_name = self._grid_data.get('grid_name', '')
        plotter = TwoDRectGridPlot(cogrid, grid_name)
        plotter.generate_grid_preview_plot(self.ui.hlay_grid_preview)
        if cogrid is None:  # If there is no linked CGrid, dim the grid preview plot groupbox.
            self.ui.grid_preview_group.setEnabled(False)

    def _setup_output_tab(self):
        """Called to setup the output control tab when the dialog is loaded."""
        self.ui.rad_stress_check.setChecked(self._data.info.attrs['rad_stress'] != 0)
        self.ui.c2shore_check.setChecked(self._data.info.attrs['c2shore'] != 0)
        self.ui.output_stations_check.setChecked(self._data.info.attrs['output_stations'] != 0)
        self.ui.monitor_check.setChecked(self._data.info.attrs['monitoring'] != 0)
        self.ui.nesting_check.setChecked(self._data.info.attrs['nesting'] != 0)
        self.ui.breaking_type_combo.setCurrentText(self._data.info.attrs['breaking_type'])
        self.ui.output_stations_selected_label.setText(
            tree_util.build_tree_path(self._output_stations_coverage, self._data.info.attrs['output_stations_uuid']))
        self.ui.monitor_selected_label.setText(
            tree_util.build_tree_path(self._monitor_coverage, self._data.info.attrs['monitoring_uuid']))
        self.ui.nesting_selected_label.setText(
            tree_util.build_tree_path(self._nesting_coverage, self._data.info.attrs['nesting_uuid']))

    def _setup_iterations_tab(self):
        """Called to setup the output control tab when the dialog is loaded."""
        self.ui.max_init_iters_edit.setText(str(self._data.info.attrs['max_init_iters']))
        self.ui.max_init_iters_edit.setValidator(self._int_validator)
        self.ui.init_iters_stop_value_edit.setText(str(self._data.info.attrs['init_iters_stop_value']))
        self.ui.init_iters_stop_value_edit.setValidator(self._double_validator)
        self.ui.init_iters_stop_percent_edit.setText(str(self._data.info.attrs['init_iters_stop_percent']))
        self.ui.init_iters_stop_percent_edit.setValidator(self._double_validator)
        self.ui.max_final_iters_edit.setText(str(self._data.info.attrs['max_final_iters']))
        self.ui.max_final_iters_edit.setValidator(self._int_validator)
        self.ui.final_iters_stop_value_edit.setText(str(self._data.info.attrs['final_iters_stop_value']))
        self.ui.final_iters_stop_value_edit.setValidator(self._double_validator)
        self.ui.final_iters_stop_percent_edit.setText(str(self._data.info.attrs['final_iters_stop_percent']))
        self.ui.final_iters_stop_percent_edit.setValidator(self._double_validator)

    def _parameters_tab_connections(self):
        """Handle the connections on the parameters tab based on which options are selected."""
        self.ui.plane_combo.currentTextChanged[str].connect(self._on_plane_type_changed)
        self.ui.depth_combo.currentTextChanged[str].connect(self._depth_dataset_update)
        self.ui.current_interaction_combo.currentTextChanged[str].connect(self._current_dataset_update)
        self.ui.friction_combo.currentTextChanged[str].connect(self._friction_dataset_update)
        self.ui.surge_combo.currentTextChanged[str].connect(self._surge_dataset_update)
        self.ui.wind_combo.currentTextChanged[str].connect(self._wind_dataset_update)
        self.ui.source_terms_combo.currentTextChanged[str].connect(self._wind_group_update)
        self.ui.ice_combo.currentTextChanged[str].connect(self._ice_dataset_update)
        # Update on the boundary tab
        self.ui.source_terms_combo.currentTextChanged.connect(self._case_data_table.table_view.filter_model.invalidate)
        self.ui.surge_combo.currentTextChanged.connect(self._case_data_table.table_view.filter_model.invalidate)
        self.ui.wind_combo.currentTextChanged.connect(self._case_data_table.table_view.filter_model.invalidate)

        # Copied over from xmscmswave
        self._current_dataset_update(self.ui.current_interaction_combo.currentText())
        self._friction_dataset_update(self.ui.friction_combo.currentText())
        self._surge_dataset_update(self.ui.surge_combo.currentText())
        self._wind_dataset_update(self.ui.wind_combo.currentText())
        self._wind_group_update(self.ui.source_terms_combo.currentText())
        self._case_data_table.table_view.filter_model.invalidate()

    def _boundary_tab_connections(self):
        """Handle the connections on the parameters tab based on which options are selected."""
        self.ui.boundary_source_combo.currentTextChanged[str].connect(self._source_dependencies_update)
        self.ui.location_coverage_check.stateChanged[int].connect(self._use_location_update)
        self.ui.location_coverage_button.clicked.connect(self._select_location_coverage)
        self.ui.full_source_side_combo.currentTextChanged[str].connect(self._full_plane_source_sides_update)

    def _output_tab_connections(self):
        """Handle the connections on the output tab based on which options are selected."""
        self.ui.output_stations_check.stateChanged[int].connect(self._use_output_stations_update)
        self.ui.output_stations_button.clicked.connect(self._select_output_stations_coverage)
        self.ui.monitor_check.stateChanged[int].connect(self._use_monitor_update)
        self.ui.monitor_button.clicked.connect(self._select_monitoring_coverage)
        self.ui.nesting_check.stateChanged[int].connect(self._use_nesting_update)
        self.ui.nesting_button.clicked.connect(self._select_nesting_coverage)

    def _initialize_state(self):
        """Initialize the state of widget dependencies after the UI has been set up and data loaded."""
        # Parameters tab
        self._on_plane_type_changed(self.ui.plane_combo.currentText())
        self._depth_dataset_update(self.ui.depth_combo.currentText())
        self._current_dataset_update(self.ui.current_interaction_combo.currentText())
        self._friction_dataset_update(self.ui.friction_combo.currentText())
        self._surge_dataset_update(self.ui.surge_combo.currentText())
        self._wind_dataset_update(self.ui.wind_combo.currentText())
        self._wind_group_update(self.ui.source_terms_combo.currentText())
        self._ice_dataset_update(self.ui.ice_combo.currentText())
        self._case_data_table.table_view.filter_model.invalidate()
        # Boundary tab
        self._source_dependencies_update(self.ui.boundary_source_combo.currentText())
        self._use_location_update(self.ui.location_coverage_check.checkState())
        self._full_plane_source_sides_update(self.ui.full_source_side_combo.currentText())
        # Output tab
        self._use_output_stations_update(self.ui.output_stations_check.checkState())
        self._use_monitor_update(self.ui.monitor_check.checkState())
        self._use_nesting_update(self.ui.nesting_check.checkState())

    def _on_plane_type_changed(self, current_plane_type):
        """Handle dependency states when the plane type combobox changes.

        Args:
            current_plane_type (str): The current text on the STWAVE plane mode combo box on the Parameters tab.
        """
        self._plane_type_update(current_plane_type)
        self._angle_distribution_update(current_plane_type)
        self._source_sides_update(current_plane_type)
        self._iterations_update(current_plane_type)

    def _plane_type_update(self, current_plane_type):
        """Connections for plane combo box changing.

        Args:
            current_plane_type (str): The current text on the STWAVE plane mode combo box on the Parameters tab.
        """
        half_plane = current_plane_type == const.PLANE_TYPE_HALF
        self.ui.depth_group.setVisible(half_plane)
        self.ui.current_group.setVisible(half_plane)
        self.ui.processors_i_edit.setEnabled(not half_plane)
        if half_plane:
            grp_text = 'Grid partitions (1 * J = Number of computational processors)'
        else:
            grp_text = 'Grid partitions (I * J = Number of computational processors)'
        self.ui.partitions_group.setTitle(grp_text)

    def _source_sides_update(self, current_plane_type):
        """Hide/show source side widgets when plane type changes.

        Args:
            current_plane_type (str): The current text on the STWAVE plane mode combo box on the Parameters tab.
        """
        if current_plane_type == const.PLANE_TYPE_HALF:
            self.ui.layout_full_source.setVisible(False)
            self.ui.layout_half_source.setVisible(True)
        else:  # const.PLANE_TYPE_FULL
            self.ui.layout_full_source.setVisible(True)
            self.ui.layout_half_source.setVisible(False)

    def _angle_distribution_update(self, current_plane_type):
        """Connections for angle distribution changing.

        Args:
            current_plane_type (str): The current text on the STWAVE plane mode combo box on the Parameters tab.
        """
        if current_plane_type == const.PLANE_TYPE_HALF:
            self.ui.num_frequencies_angle_label.setText('35')
            self.ui.min_frequency_angle_label.setText('-85.0°')
        elif current_plane_type == const.PLANE_TYPE_FULL:
            self.ui.num_frequencies_angle_label.setText('72')
            self.ui.min_frequency_angle_label.setText('0.0°')

    def _iterations_update(self, current_plane_type):
        """Connections for the plane combo box and iterations tab.

        Args:
            current_plane_type (str): The current text from the STWAVE plane mode combo box.
        """
        val = current_plane_type == const.PLANE_TYPE_FULL
        self.ui.max_init_iters_edit.setVisible(val)
        self.ui.init_iters_stop_value_edit.setVisible(val)
        self.ui.init_iters_stop_percent_edit.setVisible(val)
        self.ui.max_final_iters_edit.setVisible(val)
        self.ui.final_iters_stop_value_edit.setVisible(val)
        self.ui.final_iters_stop_percent_edit.setVisible(val)
        self.ui.max_init_iters_label.setVisible(val)
        self.ui.init_iters_stop_value_label.setVisible(val)
        self.ui.init_iters_stop_percent_label.setVisible(val)
        self.ui.max_final_iters_label.setVisible(val)
        self.ui.final_iters_stop_value_label.setVisible(val)
        self.ui.final_iters_stop_percent_label.setVisible(val)
        # self.ui.iterations_edit_spacer.setVisible(val)
        self.ui.full_plane_only_label.setVisible(not val)

    def _depth_dataset_update(self, current_depth_type):
        """Connections for depth combo box changing.

        Args:
            current_depth_type (str): The current text on the Depth combo box on the Parameters tab.
        """
        self.ui.depth_label.setVisible(current_depth_type == const.DEP_OPT_TRANSIENT)
        self.ui.depth_button.setVisible(current_depth_type == const.DEP_OPT_TRANSIENT)

    def _current_dataset_update(self, current_current_type):
        """Connections for current interaction combo box changing.

        Args:
            current_current_type (str): The current text on the Current interaction combo box on the Parameters tab.
        """
        self.ui.current_interaction_label.setVisible(current_current_type == const.OPT_DSET)
        self.ui.current_interaction_button.setVisible(current_current_type == const.OPT_DSET)

    def _friction_dataset_update(self, current_friction_type):
        """Connections for bottom friction combo box changing.

        Args:
            current_friction_type (str): The current text on the Bottom friction combo box on the Parameters tab.
        """
        show_dataset = False
        show_constant = False

        if 'constant' in self._previous_friction:
            friction_val = self.ui.friction_edit.text()
            if self._previous_friction == const.FRIC_OPT_JONSWAP_CONST:
                self._data.info.attrs['JONSWAP'] = float(friction_val)
            elif self._previous_friction == const.FRIC_OPT_MANNING_CONST:
                self._data.info.attrs['manning'] = float(friction_val)

        if current_friction_type == const.FRIC_OPT_JONSWAP_DSET:
            show_dataset = True
            self.ui.friction_label.setText(tree_util.build_tree_path(self._scalar_dataset,
                                                                     self._data.info.attrs['JONSWAP_uuid']))
        elif current_friction_type == const.FRIC_OPT_MANNING_DSET:
            show_dataset = True
            self.ui.friction_label.setText(tree_util.build_tree_path(self._scalar_dataset,
                                                                     self._data.info.attrs['manning_uuid']))
        elif current_friction_type == const.FRIC_OPT_JONSWAP_CONST:
            show_constant = True
            self.ui.friction_edit.setText(str(self._data.info.attrs['JONSWAP']))
        elif current_friction_type == const.FRIC_OPT_MANNING_CONST:
            show_constant = True
            self.ui.friction_edit.setText(str(self._data.info.attrs['manning']))

        self.ui.friction_label.setVisible(show_dataset)
        self.ui.friction_button.setVisible(show_dataset)
        self.ui.friction_edit.setVisible(show_constant)
        self._previous_friction = current_friction_type

    def _surge_dataset_update(self, current_surge_type):
        """Connections for surge combo box changing.

        Args:
            current_surge_type (str): The current text on the Surge fields combo box on the Parameters tab.
        """
        self.ui.surge_label.setVisible(current_surge_type == const.OPT_DSET)
        self.ui.surge_button.setVisible(current_surge_type == const.OPT_DSET)

    def _wind_dataset_update(self, current_wind_type):
        """Connections for surge combo box changing.

        Args:
            current_wind_type (str): The current text on the Wind fields combo box on the Parameters tab.
        """
        self.ui.wind_label.setVisible(current_wind_type == const.OPT_DSET)
        self.ui.wind_button.setVisible(current_wind_type == const.OPT_DSET)

    def _wind_group_update(self, current_source_terms):
        """Connections for wind group based on source terms.

        Args:
            current_source_terms (str): The current text on the Source terms combo box on the Parameters tab.
        """
        visible = current_source_terms == const.SOURCE_PROP_AND_TERMS
        self.ui.wind_group.setVisible(visible)
        self.ui.wind_convention_widget.setVisible(visible)

    def _ice_dataset_update(self, current_ice_type):
        """Connections for surge combo box changing.

        Args:
            current_ice_type (str): The current text on the Ice combo box on the Parameters tab.
        """
        self.ui.ice_label.setVisible(current_ice_type == const.OPT_DSET)
        self.ui.ice_button.setVisible(current_ice_type == const.OPT_DSET)
        self.ui.ice_threshold_label.setVisible(current_ice_type == const.OPT_DSET)
        self.ui.ice_threshold_edit.setVisible(current_ice_type == const.OPT_DSET)
        # self.ui.ice_threshold_spacer.setVisible(current_ice_type == const.OPT_DSET)

    def _reftime_label_update(self):
        """Sets the reftime label text using datetime format from SMS preferences.

        Returns:
            QDateTime: The current reference datetime
        """
        reftime = datetime.datetime.strptime(self._data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
        qreftime = gui_util.datetime_to_qdatetime(reftime)
        abs_date = qreftime.toString(self._time_format) if self._time_format else qreftime.toString()
        msg = f'{abs_date}    Units: {self._data.info.attrs["reftime_units"]}'
        self.ui.lbl_reftime.setText(msg)
        return qreftime

    def _source_dependencies_update(self, current_source_type):
        """Slot to update widgets when the boundary source combo box changes.

        Args:
            current_source_type (str): The current text on the Source combo box on the Boundary control tab.
        """
        self.ui.sides_group.setVisible(current_source_type == const.SPEC_OPT_COV)
        self.ui.subset_group.setVisible(current_source_type == const.SPEC_OPT_COV)

    def _full_plane_source_sides_update(self, source_sides):
        """Slot to update widgets when the full plane source sides combobox option changes.

        Args:
            source_sides (str): The currently selected source sides (only applicable if full plane)
        """
        self.ui.full_side1_label.setText('Side 1:' if source_sides == const.OPT_SIDES1_AND_3 else 'Side 2:')
        self.ui.full_side2_label.setText('Side 3:' if source_sides == const.OPT_SIDES1_AND_3 else 'Side 4:')

    def _use_location_update(self, current_use_location):
        """Connections for the use location checkbox.

        Args:
            current_use_location (Qt.CheckState): The check state of the Use location coverage check box.
        """
        self.ui.location_coverage_button.setVisible(current_use_location == Qt.Checked)
        self.ui.location_coverage_label.setVisible(current_use_location == Qt.Checked)

    def _use_output_stations_update(self, current_use_output):
        """Handles connection for output stations checkbox."""
        self.ui.output_stations_button.setVisible(current_use_output == Qt.Checked)
        self.ui.output_stations_selected_label.setVisible(current_use_output == Qt.Checked)

    def _use_monitor_update(self, current_use_output):
        """Handles connection for monitoring cells checkbox.

        Args:
            current_use_output (Qt.CheckState): The check state of the Use monitoring cells check box.
        """
        self.ui.monitor_button.setVisible(current_use_output == Qt.Checked)
        self.ui.monitor_selected_label.setVisible(current_use_output == Qt.Checked)

    def _use_nesting_update(self, current_use_output):
        """Handles connection for nesting checkbox.

        Args:
            current_use_output (Qt.CheckState): The check state of the Use nesting points check box.
        """
        self.ui.nesting_button.setVisible(current_use_output == Qt.Checked)
        self.ui.nesting_selected_label.setVisible(current_use_output == Qt.Checked)

    def _on_friction_button(self):
        """Handles connecting the bottom friction button, depending on whether JONSWAP or manning is chosen."""
        if const.FRIC_OPT_JONSWAP_DSET == self.ui.friction_combo.currentText():
            DatasetSelector.select_dataset(self, self.ui.friction_label,
                                           'Select bottom friction dataset', self.simulation_grid,
                                           DatasetSelector.is_scalar_if_dset, self._data.info.attrs,
                                           'JONSWAP_uuid', self.icon_connected)
        elif const.FRIC_OPT_MANNING_DSET == self.ui.friction_combo.currentText():
            DatasetSelector.select_dataset(self, self.ui.friction_label,
                                           'Select bottom friction dataset', self.simulation_grid,
                                           DatasetSelector.is_scalar_if_dset, self._data.info.attrs,
                                           'manning_uuid', self.icon_connected)

    def _select_output_stations_coverage(self):
        """Slot to display the coverages tree item selector, for the output stations coverage."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select Station Points Coverage',
            target_type=xmd.CoverageItem,
            pe_tree=self._output_stations_coverage,
            previous_selection=self._data.info.attrs['output_stations_uuid'],
            parent=self,
            allow_multi_select=False
        )

        if selector_dlg.exec():
            selected_uuid = selector_dlg.get_selected_item_uuid()
            coverage_path = tree_util.build_tree_path(self._output_stations_coverage, selected_uuid)
            self.ui.output_stations_selected_label.setText(coverage_path)
            self._data.info.attrs['output_stations_uuid'] = selected_uuid

    def _select_monitoring_coverage(self):
        """Slot to display the coverages tree item selector for the monitoring coverage."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select Monitoring Cell Coverage',
            target_type=xmd.CoverageItem,
            pe_tree=self._monitor_coverage,
            previous_selection=self._data.info.attrs['monitoring_uuid'],
            parent=self,
            allow_multi_select=False
        )

        if selector_dlg.exec():
            selected_uuid = selector_dlg.get_selected_item_uuid()
            coverage_path = tree_util.build_tree_path(self._monitor_coverage, selected_uuid)
            self.ui.monitor_selected_label.setText(coverage_path)
            self._data.info.attrs['monitoring_uuid'] = selected_uuid

    def _select_nesting_coverage(self):
        """Slot to display the coverages tree item selector for the nesting coverage."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select Nesting Points Coverage',
            target_type=xmd.CoverageItem,
            pe_tree=self._nesting_coverage,
            previous_selection=self._data.info.attrs['nesting_uuid'],
            parent=self,
            allow_multi_select=False
        )

        if selector_dlg.exec():
            selected_uuid = selector_dlg.get_selected_item_uuid()
            coverage_path = tree_util.build_tree_path(self._nesting_coverage, selected_uuid)
            self.ui.nesting_selected_label.setText(coverage_path)
            self._data.info.attrs['nesting_uuid'] = selected_uuid

    def _select_location_coverage(self):
        """Slot to display the coverages tree item selector, for the location coverage."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select Location Coverages',
            target_type=xmd.CoverageItem,
            pe_tree=self._location_coverage,
            previous_selection=self._data.info.attrs['location_coverage_uuid'],
            parent=self,
            allow_multi_select=False
        )

        if selector_dlg.exec():
            selected_uuid = selector_dlg.get_selected_item_uuid()
            coverage_path = tree_util.build_tree_path(self._location_coverage, selected_uuid)
            self.ui.location_coverage_label.setText(coverage_path)
            self._data.info.attrs['location_coverage_uuid'] = selected_uuid

    def _set_populate_from_spectra(self):
        """Populate the case times table with the spectra."""
        if not self._spec_cov:
            msg = QMessageBox(QMessageBox.Warning, 'SMS', 'No spectral coverage in simulation.', QMessageBox.Ok, self)
            msg.exec()
            return

        # We want to use the earliest reference date from the spectral grids.
        reftime = None

        # Loop through the points in spectral coverage for side 1
        spectra_times = []
        spec_pts = self._spec_cov.m_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        for spec_pt in spec_pts:
            spec_pt_id = spec_pt.id
            spec_grids = self._spec_cov.GetSpectralGrids(spec_pt_id)
            for spec_grid in spec_grids:
                grid_reftime = julian_to_datetime(spec_grid.m_refTime)
                if reftime is None or grid_reftime < reftime:
                    reftime = grid_reftime
                # only do this once so we only have one file per dataset
                spec_dset = spec_grid.get_dataset(io_util.temp_filename())
                for i in range(spec_dset.num_times):
                    spec_dset.ts_idx = i
                    ts_time = julian_to_datetime(spec_dset.ts_time)
                    spectra_times.append(ts_time)

        spectra_times = list(set(spectra_times))
        if reftime is None:  # No spectral grids?
            reftime = string_to_datetime(self._data.info.attrs['reftime'])
        self._data.info.attrs['reftime'] = datetime_to_string(reftime)
        self._reftime_label_update()

        # Convert the timesteps to a delta time vs. reftime in seconds
        one_unit = 60.0  # self._data.info.attrs['reftime_units'] == 'minutes'
        if self._data.info.attrs['reftime_units'] == 'hours':
            one_unit = 3600.0
        elif self._data.info.attrs['reftime_units'] == 'days':
            one_unit = 3600.0 * 24.0
        spectra_delta_seconds = [(spec_time - reftime).total_seconds() for spec_time in spectra_times]
        spectra_delta_seconds.sort()

        # Set up case times data
        wind_dir = [0.0] * len(spectra_delta_seconds)
        wind_mag = [0.0] * len(spectra_delta_seconds)
        water_level = [0.0] * len(spectra_delta_seconds)
        times = np.array(spectra_delta_seconds, dtype=np.float64) / one_unit  # Convert to currently selected units
        times = np.around(times, 3)  # Discard excess precision
        case_time_data = simulation_data.case_data_table(times, wind_dir, wind_mag, water_level)
        case_time_dataset = xr.Dataset(data_vars=case_time_data)

        # Update the model which will update the table
        self._case_data_table.model.removeRows(0, self._case_data_table.model.rowCount(0))
        self._case_data_table.model.insertRows(0, len(spectra_delta_seconds))
        self._case_data_table.model.data_frame = case_time_dataset.to_dataframe()

    def _set_reftime(self):
        """Sets the reftime information for the case times table."""
        orig_reftime = datetime.datetime.strptime(self._data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
        orig_units = self._data.info.attrs['reftime_units']
        reftime_dlg = RefTimeDialog(data=self._data, time_format=self._time_format, parent=self)
        if reftime_dlg.exec():
            qreftime = self._reftime_label_update()  # Update the label text

            if self._case_data_table.model.rowCount() < 1:
                return  # Don't bother with the next bit if the table is empty

            # Update the values if the user desires
            reply = QMessageBox.question(self, 'SMS', 'Would you like the relative spectral times to be recalculated '
                                         'according to the new reference time?', QMessageBox.Yes, QMessageBox.No)
            if reply == QMessageBox.Yes:
                # Change the time values by the new reftime
                new_reftime = gui_util.qdatetime_to_datetime(qreftime)
                new_units = self._data.info.attrs['reftime_units']

                for i in range(self._case_data_table.model.rowCount(0)):
                    idx = self._case_data_table.model.index(i, 0)
                    time_data = float(self._case_data_table.model.data(idx))
                    # change value
                    time_data = self._convert_time(orig_reftime, orig_units, new_reftime, new_units, time_data)
                    self._case_data_table.model.setData(idx, time_data)

    def _convert_time(self, orig_timeref, orig_units, new_timeref, new_units, old_value):
        """Convert the old time value to a new time value (based on old and new reftimes).

        Args:
            orig_timeref (datetime.datetime):  The original time reference.
            orig_units (str):  The original time units (days, hours, minutes).
            new_timeref (datetime.datetime):  The new time reference.
            new_units (str):  The original time units (days, hours, minutes).
            old_value (float):  The old time difference value.

        Return:
            float:  The converted time difference value.
        """
        # Convert the original time offset to seconds
        if orig_units == const.TIME_UNITS_MINUTES:
            old_value = old_value * 60.0
        elif orig_units == const.TIME_UNITS_HOURS:
            old_value = old_value * 3600.0
        elif orig_units == const.TIME_UNITS_DAYS:
            old_value = old_value * 3600.0 * 24.0
        else:
            raise ValueError(f'Invalid time units:  {orig_units}')
        # Calculate the orinal time as a datetime, based on the original time offset
        orig_dt = orig_timeref + datetime.timedelta(seconds=float(old_value))

        # Get a difference between the original time and new time reference
        time_diff_seconds = (orig_dt - new_timeref).total_seconds()
        if new_units == const.TIME_UNITS_MINUTES:
            return time_diff_seconds / 60.0
        elif new_units == const.TIME_UNITS_HOURS:
            return time_diff_seconds / 3600.0
        elif new_units == const.TIME_UNITS_DAYS:
            return time_diff_seconds / (3600.0 * 24.0)
        else:
            raise ValueError(f'Invalid time units:  {new_units}')

    def _accept_parameters_tab(self):
        """Save the data on the Parameters tab on accept."""
        self._data.info.attrs['plane'] = self.ui.plane_combo.currentText()
        self._data.info.attrs['source_terms'] = self.ui.source_terms_combo.currentText()
        self._data.info.attrs['depth'] = self.ui.depth_combo.currentText()
        self._data.info.attrs['current_interaction'] = self.ui.current_interaction_combo.currentText()
        self._data.info.attrs['friction'] = self.ui.friction_combo.currentText()
        if const.FRIC_OPT_JONSWAP_CONST == self._data.info.attrs['friction']:
            self._data.info.attrs['JONSWAP'] = float(self.ui.friction_edit.text())
        elif const.FRIC_OPT_MANNING_CONST == self._data.info.attrs['friction']:
            self._data.info.attrs['manning'] = float(self.ui.friction_edit.text())
        self._data.info.attrs['surge'] = self.ui.surge_combo.currentText()
        self._data.info.attrs['wind'] = self.ui.wind_combo.currentText()
        self._data.info.attrs['ice'] = self.ui.ice_combo.currentText()
        self._data.info.attrs['ice_threshold'] = float(self.ui.ice_threshold_edit.text())
        self._data.info.attrs['processors_i'] = int(self.ui.processors_i_edit.text())
        self._data.info.attrs['processors_j'] = int(self.ui.processors_j_edit.text())

    def _accept_boundary_tab(self):
        """Save the data on the Boundary control tab on accept."""
        self._data.info.attrs['boundary_source'] = self.ui.boundary_source_combo.currentText()
        self._data.info.attrs['interpolation'] = self.ui.interpolation_combo.currentText()
        self._data.info.attrs['num_frequencies'] = int(self.ui.num_frequencies_edit.text())
        self._data.info.attrs['delta_frequency'] = float(self.ui.delta_frequency_edit.text())
        self._data.info.attrs['min_frequency'] = float(self.ui.min_frequency_edit.text())
        self._data.info.attrs['location_coverage'] = 1 if self.ui.location_coverage_check.isChecked() else 0
        self._data.info.attrs['angle_convention'] = self.ui.angle_convention_combo.currentText()
        self._accept_source_sides()
        self._data.case_times = self._case_data_table.model.data_frame.to_xarray()

    def _accept_source_sides(self):
        """Save the source side options based on the rules of STWAVE."""
        if self.ui.plane_combo.currentText() == const.PLANE_TYPE_HALF:  # Half plane rules
            # There is only one specified side, the adjacent sides are open lateral and the opposite is zero spectrum.
            specified_idx = self.ui.half_source_side_combo.currentIndex()
            for side_idx in range(self.ui.half_source_side_combo.count()):
                attr = f'side{side_idx + 1}'
                if side_idx == specified_idx:  # The one and only one specified side
                    self._data.info.attrs[attr] = const.I_BC_SPECIFIED
                elif side_idx + 2 == specified_idx or side_idx - 2 == specified_idx:  # Opposite side, must be zero
                    self._data.info.attrs[attr] = const.I_BC_ZERO
                else:  # One of the adjacent sides, must be open lateral
                    self._data.info.attrs[attr] = const.I_BC_LATERAL
        else:  # Full plane rules
            if self.ui.full_source_side_combo.currentText() == const.OPT_SIDES1_AND_3:
                self._data.info.attrs['side1'] = self.ui.full_side1_combo.currentText()  # Specified or zero
                self._data.info.attrs['side3'] = self.ui.full_side2_combo.currentText()  # Specified or zero
                self._data.info.attrs['side2'] = const.I_BC_LATERAL
                self._data.info.attrs['side4'] = const.I_BC_LATERAL
            else:
                self._data.info.attrs['side2'] = self.ui.full_side1_combo.currentText()  # Specified or zero
                self._data.info.attrs['side4'] = self.ui.full_side2_combo.currentText()  # Specified or zero
                self._data.info.attrs['side1'] = const.I_BC_LATERAL
                self._data.info.attrs['side3'] = const.I_BC_LATERAL

    def _accept_output_tab(self):
        """Save the data on the Output control tab."""
        self._data.info.attrs['rad_stress'] = 1 if self.ui.rad_stress_check.isChecked() else 0
        self._data.info.attrs['c2shore'] = 1 if self.ui.c2shore_check.isChecked() else 0
        self._data.info.attrs['output_stations'] = 1 if self.ui.output_stations_check.isChecked() else 0
        self._data.info.attrs['monitoring'] = 1 if self.ui.monitor_check.isChecked() else 0
        self._data.info.attrs['nesting'] = 1 if self.ui.nesting_check.isChecked() else 0
        self._data.info.attrs['breaking_type'] = self.ui.breaking_type_combo.currentText()

    def _accept_iterations_tab(self):
        """Save the data on the Iterations control tab."""
        self._data.info.attrs['max_init_iters'] = int(self.ui.max_init_iters_edit.text())
        self._data.info.attrs['init_iters_stop_value'] = float(self.ui.init_iters_stop_value_edit.text())
        self._data.info.attrs['init_iters_stop_percent'] = float(self.ui.init_iters_stop_percent_edit.text())
        self._data.info.attrs['max_final_iters'] = int(self.ui.max_final_iters_edit.text())
        self._data.info.attrs['final_iters_stop_value'] = float(self.ui.final_iters_stop_value_edit.text())
        self._data.info.attrs['final_iters_stop_percent'] = float(self.ui.final_iters_stop_percent_edit.text())

    @staticmethod
    def icon_connected(tree_node):
        """Gets CGRID2D icon.

        Args:
            tree_node: The tree node.

        Returns:
            str: Path to the icon.
        """
        return get_tree_icon_from_xms_typename(tree_node)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def showEvent(self, event):  # noqa: N802
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        wbd.restore_splitter_geometry(splitter=self.ui.splitter_horiz, package_name='xms.stwave',
                                      dialog_name=f'{self._dlg_name}_horiz')
        wbd.restore_splitter_geometry(splitter=self.ui.splitter_vert, package_name='xms.stwave',
                                      dialog_name=f'{self._dlg_name}_vert')

    def accept(self):
        """Handles the accept action."""
        self._accept_parameters_tab()
        self._accept_boundary_tab()
        self._accept_output_tab()
        self._accept_iterations_tab()
        wbd.save_splitter_geometry(splitter=self.ui.splitter_horiz, package_name='xms.stwave',
                                   dialog_name=f'{self._dlg_name}_horiz')
        wbd.save_splitter_geometry(splitter=self.ui.splitter_vert, package_name='xms.stwave',
                                   dialog_name=f'{self._dlg_name}_vert')
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        # We always save the splitter position, even if user rejects.
        wbd.save_splitter_geometry(splitter=self.ui.splitter_horiz, package_name='xms.stwave',
                                   dialog_name=f'{self._dlg_name}_horiz')
        wbd.save_splitter_geometry(splitter=self.ui.splitter_vert, package_name='xms.stwave',
                                   dialog_name=f'{self._dlg_name}_vert')
        super().reject()
