"""Dialog to export individual files of an STWAVT simulation."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtWidgets import QDialogButtonBox, QGridLayout, QGroupBox, QLabel, QPushButton, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy import settings
from xms.guipy.dialogs.file_selector_dialogs import get_open_foldername, get_save_filename
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.stwave.feedback.sim_export_worker_thread import SimExportWorkerThread
from xms.stwave.feedback.spatial_export_worker_thread import SpatialExportWorkerThread
from xms.stwave.feedback.spectral_export_worker_thread import SpectralExportWorkerThread


class PartialExportDialog(XmsDlg):
    """A dialog that appears for selecting individual files to export."""

    def __init__(self, parent, query):
        """Initializes the class, sets up the ui.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS

        """
        super().__init__(parent, 'xms.stwave.gui.partial_export_dialog')
        self.setWindowTitle('Export STWAVE Files to Location')
        self.query = query
        self.widgets = {}
        self.sim_name = ''
        self.setup_ui()
        self.filenames = []  # For testing
        self._get_simulation_name()

    def _get_simulation_name(self):
        """Get the simulation name for building default filenames."""
        sim_item = self.query.parent_item()
        self.sim_name = sim_item.name

    def _get_save_filename(self, selected_filter, file_filter, extension):
        """Get a filename to use for writing a file.

        Args:
            selected_filter (str): The file filter that should be selected in the file browser dialog
            file_filter (str): The file filter list that should be in the file browser dialog. Filters separated by ';;'

        Returns:
            str: The selected filename (or next hard-coded testing filename). Empty string if user cancels.
        """
        if self.filenames:  # If testing, don't bring up a dialog
            return self.filenames.pop()
        if extension:  # Select a file if filter passed in
            filename = os.path.join(settings.get_file_browser_directory(), f'{self.sim_name}.{extension}')
            return get_save_filename(self, selected_filter, file_filter, start_dir=filename)
        return get_open_foldername(self, 'Select an export location')  # Otherwise select a directory

    def setup_ui(self):
        """Add widgets to the dialog."""
        self.widgets['btn_sim'] = QPushButton('Control File...')
        self.widgets['btn_sim'].clicked.connect(self.on_btn_sim)
        self.widgets['btn_spatial'] = QPushButton('Spatial Datasets...')
        self.widgets['btn_spatial'].clicked.connect(self.on_btn_spatial)
        self.widgets['btn_spectral'] = QPushButton('Spectral File...')
        self.widgets['btn_spectral'].clicked.connect(self.on_btn_spectral)

        self.widgets['lbl_sim'] = QLabel('Export STWAVE simulation Model Control')
        self.widgets['lbl_spatial'] = QLabel('Export STWAVE spatial datasets')
        self.widgets['lbl_spectral'] = QLabel('Export STWAVE spectral coverage')

        self.widgets['grid_layout'] = QGridLayout()
        self.widgets['grid_layout'].addWidget(self.widgets['btn_sim'], 0, 0)
        self.widgets['grid_layout'].addWidget(self.widgets['lbl_sim'], 0, 1)
        self.widgets['grid_layout'].addWidget(self.widgets['btn_spatial'], 1, 0)
        self.widgets['grid_layout'].addWidget(self.widgets['lbl_spatial'], 1, 1)
        self.widgets['grid_layout'].addWidget(self.widgets['btn_spectral'], 2, 0)
        self.widgets['grid_layout'].addWidget(self.widgets['lbl_spectral'], 2, 1)

        self.widgets['grp_files'] = QGroupBox('STWAVE files')
        self.widgets['grp_files'].setLayout(self.widgets['grid_layout'])

        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setStandardButtons(QDialogButtonBox.Ok)
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['layout_vert_main'] = QVBoxLayout()
        self.widgets['layout_vert_main'].addWidget(self.widgets['grp_files'])
        self.widgets['layout_vert_main'].addWidget(self.widgets['btn_box'])
        self.setLayout(self.widgets['layout_vert_main'])

    def on_btn_sim(self):
        """Slot for when the user clicks the export control file button."""
        # Prompt the user for a save location
        selected_filter = 'STWAVE control file (*.sim)'
        file_filter = f'{selected_filter};;All types (*.*)'
        testing = 'testing' if self.filenames else ''
        filename = self._get_save_filename(selected_filter, file_filter, 'sim')
        if not filename:
            return  # User cancelled

        note = 'All current Model Control inputs will be exported to the .sim file, including relative file references.'
        worker = SimExportWorkerThread(filename, self.query, self)
        display_text = {
            'title': 'Exporting Control File',
            'working_prompt': f'Exporting control file to "{filename}" Please wait...',
            'error_prompt': 'Error(s) encountered exporting control file. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting control file. Review log output for more details.',
            'success_prompt': f'Successfully exported control file to "{filename}"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.stwave', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()

    def on_btn_spatial(self):
        """Slot for when the user clicks the export spatial datasets button."""
        # Prompt the user for a save location
        testing = 'testing' if self.filenames else ''
        folder = self._get_save_filename('', '', '')
        if not folder:
            return  # User cancelled
        os.makedirs(folder, exist_ok=True)

        old_cwd = os.getcwd()
        os.chdir(folder)
        note = 'All spatial datasets currently specified in the simulation model control will be exported to the ' \
               'specified location with the specified file basename.'
        worker = SpatialExportWorkerThread(self.query, self)
        display_text = {
            'title': 'Exporting Spatial Datasets',
            'working_prompt': f'Exporting spatial datasets to "{folder}" Please wait...',
            'error_prompt': 'Error(s) encountered exporting spatial datasets. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting spatial datasets. Review log output for more details.',
            'success_prompt': f'Successfully exported spatial datasets to "{folder}"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.stwave', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()
        os.chdir(old_cwd)

    def on_btn_spectral(self):
        """Slot for when the user clicks the export spectral datasets button."""
        # Prompt the user for a save location
        selected_filter = 'STWAVE spectral datasets (*.eng)'
        file_filter = f'{selected_filter};;All types (*.*)'
        testing = 'testing' if self.filenames else ''
        filename = self._get_save_filename(selected_filter, file_filter, 'eng')
        if not filename:
            return  # User cancelled

        note = 'The spectral coverage currently linked to the simulation will be exported.'
        worker = SpectralExportWorkerThread(filename, self.query, self)
        display_text = {
            'title': 'Exporting Spectral Coverage',
            'working_prompt': f'Exporting spectral coverage to "{filename}" Please wait...',
            'error_prompt': 'Error(s) encountered exporting spectral coverage. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting spectral coverage. Review log output for more details.',
            'success_prompt': f'Successfully exported spectral coverage to "{filename}"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.stwave', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()
