"""Functions that return the XMS data."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property
from pathlib import Path
from typing import Optional, Type

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.components.bases.visible_coverage_component_base import VisibleCoverageComponentBase
from xms.components.component_builders.coverage_component_builder import ComponentType
from xms.data_objects.parameters import Coverage
from xms.gmi.data.sim_data import SimData

# 4. Local modules
from xms.swmm.components.coverage_component import CoverageComponent, StormDrainLinkComponent, StormDrainNodeComponent


class SwmmData:
    """A class containing functions that return XMS data."""

    model_name = 'SWMM'
    sim_component_unique_name = 'SimComponent'
    coverage_component_type = CoverageComponent

    def __init__(self, query: Query):
        """Init function for SwmmData class."""
        self.query = query or Query()
        self.display_wkt = self.query.display_projection.well_known_text
        self.vertical_units = self.query.display_projection.vertical_units
        self._sim_uuid: Optional[str] = None

    @cached_property
    def project_tree(self):
        """The project tree."""
        return self.query.copy_project_tree()

    @property
    def sim_uuid(self) -> str:
        """The UUID of the simulation, not its component."""
        if not self._sim_uuid:
            tree = self.project_tree

            # We should be at either the simulation or its hidden component. The tree should contain the simulation, but
            # not its hidden component, so we can figure out where we are by checking if the current item's UUID is in
            # the tree. If it is, we must be at the simulation. If not, we're at its component, and its parent item is
            # the simulation.
            current_item_uuid = self.query.current_item_uuid()
            node = tree_util.find_tree_node_by_uuid(tree, current_item_uuid)
            if node is not None:
                return node.uuid

            parent_item_uuid = self.query.parent_item_uuid()
            node = tree_util.find_tree_node_by_uuid(tree, parent_item_uuid)
            self._sim_uuid = node.uuid
        return self._sim_uuid

    @sim_uuid.setter
    def sim_uuid(self, sim_uuid: str):
        """Setter for the sim_uuid."""
        if sim_uuid:
            self._sim_uuid = sim_uuid

    @cached_property
    def sim_tree_node(self) -> tree_util.TreeNode:
        """The simulation's tree node."""
        node = tree_util.find_tree_node_by_uuid(self.project_tree, self.sim_uuid)
        return node

    @cached_property
    def sim_data(self) -> SimData:
        """The data manager for the sim component."""
        # If/when we do snap previews, we might end up with a circular dependency:
        # SimComponent -> Feedback runner -> this -> SimComponent.
        # SimData is likely to be enough for any runner, and it breaks the cycle.
        do_sim = self.query.item_with_uuid(
            self.sim_uuid, model_name=self.model_name, unique_name=self.sim_component_unique_name
        )
        main_file = do_sim.main_file
        data = SimData(main_file)
        return data

    def _get_coverage(
            self, coverage_type: str, component_type: Type[ComponentType]
    ) -> tuple[Coverage, VisibleCoverageComponentBase]:
        sim_node = self.sim_tree_node
        coverage_node = tree_util.descendants_of_type(
            sim_node, model_name=self.model_name, coverage_type=coverage_type, allow_pointers=True,
            only_first=True
        )
        coverage_uuid = coverage_node.uuid
        coverage = self.query.item_with_uuid(coverage_uuid)
        component = self.query.item_with_uuid(
            coverage_uuid, model_name=self.model_name, unique_name=component_type.unique_name()
        )
        main_file = Path(component.main_file)
        component = component_type(main_file)
        self.query.load_component_ids(component, points=True, arcs=True, polygons=True, arc_groups=True)
        return coverage, component

    @cached_property
    def link_coverage(self) -> tuple[Coverage, VisibleCoverageComponentBase]:
        """Returns the XMS link coverage information."""
        return self._get_coverage('Storm Drain Links', StormDrainLinkComponent)

    @cached_property
    def node_coverage(self) -> tuple[Coverage, VisibleCoverageComponentBase]:
        """Returns the XMS link coverage information."""
        return self._get_coverage('Storm Drain Nodes', StormDrainNodeComponent)
