"""DMI component for the tidal constituents simulation."""
# 1. Standard python modules
from distutils.dir_util import copy_tree
import os

# 2. Third party modules
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.api.tree import tree_util as tr_util
from xms.components.bases.component_base import ComponentBase
from xms.guipy.dialogs.dataset_selector import DatasetSelector

# 4. Local modules
from xms.tides.data.tidal_data import TidalData
from xms.tides.gui.tidal_dlg import TidalDlg

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class TidalComponent(ComponentBase):
    """A Dynamic Model Interface (DMI) component for a MODFLOW 6 simulation."""

    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = TidalData(self.main_file)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE - happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE - happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE - happens when re-saving this project.

                SAVE_AS - happens when saving a project in a new location. This happens the first time we save a
                project.

                UNLOCK - happens when the component is about to be changed and it does not have a matching uuid folder
                in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list[tuple(str)]): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list[xmsapi.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        action_requests = []

        os.makedirs(new_path, exist_ok=True)
        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))

        if not os.path.isfile(self.main_file):
            messages.append(('ERROR', 'Cannot save component because component does not exist.'))
            return self.main_file, messages, action_requests

        # Check if we are already in the new location
        if os.path.isfile(new_main_file) and os.path.samefile(new_main_file, self.main_file):
            return self.main_file, messages, action_requests

        copy_tree(os.path.dirname(self.main_file), os.path.dirname(new_main_file))

        return new_main_file, messages, action_requests

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list[str]): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list[xmsapi.dmi.MenuItem]):
                A list of menus and menu items to be shown. Note that this list can have objects of type xmsapi.dmi.Menu
                as well as xmsapi.dmi.MenuItem. "None" may be added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select or nothing selected

        menu_list = [None]  # None == spacer
        # Add the Open menu command that will display the well dialog when clicked.
        tidal_const_item = MenuItem()
        tidal_const_item.text = "Edit Constituents..."
        tidal_const_action = ActionRequest()
        tidal_const_action.dialog_modality = 'MODAL'
        tidal_const_action.method_name = "edit_tidal_dialog"
        tidal_const_action.class_name = "TidalComponent"
        tidal_const_action.module_name = "xms.tides.components.tidal_component"
        tidal_const_action.main_file = main_file_list[0][0]
        tidal_const_item.action_request = tidal_const_action
        menu_list.append(tidal_const_item)

        menu_list.append(None)  # Adds a separator
        return menu_list

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list[tuple(str)]): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list[xmsapi.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        open_action = ActionRequest()
        open_action.dialog_modality = 'MODAL'
        open_action.method_name = "edit_tidal_dialog"
        open_action.class_name = 'TidalComponent'
        open_action.module_name = 'xms.tides.components.tidal_component'
        open_action.main_file = self.main_file
        actions.append(open_action)

        return messages, actions

    def edit_tidal_dialog(self, query, params, win_cont):
        """Opens the tidal constituent dialog.

        Args:
            query (xmsapi.dmi.Query):
            params (list[str]):
            win_cont (PySide2.QtWidgets.QWidget): The window container.

        Returns:
            messages (list[tuple(str)]): A list of tuples with the first element of the tuple
            being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
            action_requests (list[xmsapi.dmi.ActionRequest]): A list of actions for XMS to perform.
        """
        messages = []

        geom_children = []
        if query.project_tree:
            for child in query.project_tree.children:  # Only allow datasets on 2D mesh, Quadtree, and UGrid.
                if child.name == 'Mesh Data':
                    geom_children.append(child)
                elif child.name == 'UGrid Data':
                    geom_children.append(child)
                elif child.name == 'Quadtree Data':
                    geom_children.append(child)
            query.project_tree.children = geom_children
        tr_util.filter_project_explorer(query.project_tree, DatasetSelector.is_scalar_if_dset)
        dlg = TidalDlg(self.data, query.project_tree, win_cont)
        dlg.setModal(True)
        dlg_result = dlg.exec()
        if dlg_result == QDialog.Accepted:
            self.data.commit()

        actions = []
        return messages, actions
