"""Qt table view for the user defined constituents table."""

# 1. Standard python modules

# 2. Third party modules
from harmonica.tidal_constituents import Constituents
from PySide2.QtCore import QItemSelectionModel, QSortFilterProxyModel, Qt
from PySide2.QtGui import QDoubleValidator
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules
from xms.api.tree import tree_util as tr_util
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.validators.number_corrector import NumberCorrector
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules
from xms.tides.data import tidal_data as td
from xms.tides.gui import gui_util
from xms.tides.gui.constituent_button_delegate import ConstituentButtonDelegate
from xms.tides.gui.dset_selector_button_delegate import DsetSelectorButtonDelegate
from xms.tides.gui.tides_pandas_table_model import QxTidesPandasTableModel


COL_CONSTITUENT = 0  # Type of constituent, e.g. 'M2' or 'User defined'.
COL_NAME = 1  # Custom name of the constituent. Only used when COL_CONSTITUENT is 'User defined'.
COL_POTENTIAL_AMPLITUDE = 2
COL_FREQUENCY = 3
COL_NODAL_FACTOR = 4
COL_EQUILIBRIUM_ARGUMENT = 5
COL_ETRF = 6
COL_AMP = 7
COL_PHASE = 8
COL_FORCE_VELOCITY = 9
COL_VEL_AMP_X = 10
COL_VEL_PHASE_X = 11
COL_VEL_AMP_Y = 12
COL_VEL_PHASE_Y = 13

ALWAYS_ON_DATASET_COLUMNS = {COL_AMP, COL_PHASE}
CONDITIONAL_DATASET_COLUMNS = {COL_VEL_AMP_X, COL_VEL_PHASE_X, COL_VEL_AMP_Y, COL_VEL_PHASE_Y}
DATASET_COLUMNS = ALWAYS_ON_DATASET_COLUMNS | CONDITIONAL_DATASET_COLUMNS

DEFAULTS = {
    COL_CONSTITUENT: 0,
    COL_NAME: 'M2',
    COL_POTENTIAL_AMPLITUDE: 0.0,
    COL_FREQUENCY: 0.0,
    COL_NODAL_FACTOR: 1.0,
    COL_EQUILIBRIUM_ARGUMENT: 0.0,
    COL_ETRF: 0.0,
    COL_AMP: '(none selected)',
    COL_PHASE: '(none selected)',
    COL_FORCE_VELOCITY: 0,
    COL_VEL_AMP_X: '',
    COL_VEL_PHASE_X: '',
    COL_VEL_AMP_Y: '',
    COL_VEL_PHASE_Y: '',
}


class ConTableDisableEditModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, columns, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from QObject): The parent object.
            columns (list of int): Types for each column. See COL_* constants above.
        """
        super().__init__(parent)
        self.columns = columns

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index

        Returns:
            Qt.ItemFlags: Updated flags for the item
        """
        ret_flags = super().flags(index) | Qt.ItemIsEditable | Qt.ItemIsEnabled

        column_type = self.columns[index.column()]

        if column_type in ALWAYS_ON_DATASET_COLUMNS:
            return ret_flags & (~Qt.ItemIsEditable)  # Datasets are never editable.

        if column_type in CONDITIONAL_DATASET_COLUMNS:
            ret_flags = ret_flags & (~Qt.ItemIsEditable)  # Datasets are never editable
            if COL_FORCE_VELOCITY in self.columns:
                forcing_column = self.columns.index(COL_FORCE_VELOCITY)
                check_state = index.model().index(index.row(), forcing_column).data(Qt.CheckStateRole)
                if check_state != Qt.Checked:
                    return ret_flags & (~Qt.ItemIsEnabled)
            return ret_flags

        if column_type in [COL_CONSTITUENT, COL_FORCE_VELOCITY]:
            return ret_flags

        if COL_CONSTITUENT in self.columns:
            constituent_column = self.columns.index(COL_CONSTITUENT)
            if index.model().index(index.row(), constituent_column).data() != 'User specified':
                return ret_flags & (~Qt.ItemIsEnabled)  # Not user specified means everything is standard, so disable.

        return ret_flags  # No constituent column implies user specified, so everything else is editable.


class UserDefinedWidget(BasicTableWidget):
    """The User Defined tidal consituents selector table widget."""
    def __init__(self, data_frame, pe_tree, reftime, columns, parent=None):
        """Construct the widget.

        Args:
            data_frame (pandas.DataFrame): The model data.
            pe_tree (TreeNode): Project explorer tree
            reftime (datetime): Reference time to use for constituent property extraction
            parent (Something derived from QObject): The parent object.
            columns (list of int): List of types for each column. COL_* constants defined above.
        """
        super().__init__(parent)
        self.columns = columns
        self.reftime = reftime
        self.model = QxTidesPandasTableModel(data_frame, self)
        self.tidal_model = Constituents('leprovost')  # Specific model doesn't matter. LeProvost always available.
        self.enable_model = None
        self.enable_model = ConTableDisableEditModel(columns, self)
        self.enable_model.setSourceModel(self.model)

        self.con_delegate = ConstituentButtonDelegate(td.STANDARD_CONSTITUENTS, self)
        if COL_CONSTITUENT in columns:
            constituent_column = columns.index(COL_CONSTITUENT)
            self.model.set_combobox_column(constituent_column, td.STANDARD_CONSTITUENTS)
        if COL_FORCE_VELOCITY in columns:
            forcing_column = columns.index(COL_FORCE_VELOCITY)
            self.model.set_checkbox_columns({forcing_column})
            self.model.dataChanged.connect(self.on_force_velocity_changed)
        self.model.dataChanged.connect(self.on_constituent_changed)

        self._build_tree_paths(pe_tree)
        self.dset_selector_delegate = DsetSelectorButtonDelegate(pe_tree, COL_PHASE, self)
        self.checkbox_delegate = CheckBoxNoTextDelegate(self)

        dbl_validator = QDoubleValidator(self)
        dbl_validator.setRange(0.0, 10.0)
        dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.dbl_delegate = EditFieldValidator(dbl_validator, self)

        column_names, _defaults = self.model.get_column_info()
        defaults = {column_name: DEFAULTS[columns[i]] for i, column_name in enumerate(column_names)}
        self.model.set_default_values(defaults)

        self.setup_ui()

    def _build_tree_paths(self, pe_tree):
        """Build the tree paths from selected UUIDs based on the current state of the tree in XMS.

        Args:
            pe_tree (xms.api.tree.tree_node.TreeNode): Root of the project explorer tree
        """
        for row in range(self.model.rowCount()):
            for col in range(self.model.columnCount()):
                if self.columns[col] in DATASET_COLUMNS:
                    self._build_tree_path(pe_tree, row, col)

    def _build_tree_path(self, pe_tree, row, col):
        """Build a tree path from selected UUID based on the current state of the tree in XMS.

        Args:
            pe_tree (xms.api.tree.tree_node.TreeNode): Root of the project explorer tree
            row: Row of the dataset cell in the table.
            col: Column of the dataset cell in the table.
        """
        # Using the entire tree path makes the table look ugly. Truncate to tree item name. User
        # can see full path by pushing the button again.
        index = self.model.index(row, col)
        uuid = index.data()
        if uuid and uuid != gui_util.NULL_SELECTION:
            selected_node = tr_util.find_tree_node_by_uuid(pe_tree, uuid)
            if selected_node:
                self.model.tree_paths[uuid] = f'.../{selected_node.name}'

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        delegates = {
            COL_CONSTITUENT: self.con_delegate,
            COL_NAME: None,
            COL_POTENTIAL_AMPLITUDE: self.dbl_delegate,
            COL_FREQUENCY: self.dbl_delegate,
            COL_NODAL_FACTOR: self.dbl_delegate,
            COL_EQUILIBRIUM_ARGUMENT: self.dbl_delegate,
            COL_ETRF: self.dbl_delegate,
            COL_AMP: self.dset_selector_delegate,
            COL_PHASE: self.dset_selector_delegate,
            COL_FORCE_VELOCITY: self.checkbox_delegate,
            COL_VEL_AMP_X: self.dset_selector_delegate,
            COL_VEL_PHASE_X: self.dset_selector_delegate,
            COL_VEL_AMP_Y: self.dset_selector_delegate,
            COL_VEL_PHASE_Y: self.dset_selector_delegate,
        }

        delegates = {i: delegates[col_type] for i, col_type in enumerate(self.columns)}
        delegates.pop(COL_NAME, None)

        super()._setup_ui(delegates, False, False, self.enable_model)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)

    def on_btn_add_row(self):
        """Called when a new constituent row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.model.index(row_idx, 0)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
        self.table_view.resize_height_to_contents()

        if COL_CONSTITUENT in self.columns:
            constituent_column = self.columns.index(COL_CONSTITUENT)
            new_index = self.model.index(row_idx, constituent_column)
            self.on_constituent_changed(new_index, new_index)

    def on_btn_delete_row(self):
        """Called when a file row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, 0)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )
        self.table_view.resize_height_to_contents()

    def on_constituent_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index. Same as top_left_index. QxPandasTableModel only
                supports single cell editing.
        """
        if not top_left_index.isValid():
            return

        if COL_CONSTITUENT not in self.columns:
            return

        col = top_left_index.column()
        if not (0 <= col < len(self.columns)):
            # Not sure why this happens, but I've only seen it when we're deleting the row, and this method's only job
            # is to update the values in the (about-to-be-deleted) row, so we'll just do nothing.
            return
        if self.columns[col] != COL_CONSTITUENT:
            return  # False alarm, something other than the constituent changed.

        constituent_name = self.model.data(top_left_index)
        if constituent_name == td.USER_SPECIFIED_OPTION:
            return  # Nothing to do, the user can specify whatever they want.

        # This is a standard one, so we need to fill in the standard values.
        df = self.tidal_model.get_nodal_factor([constituent_name], self.reftime)
        con_df = df.loc[constituent_name]
        row = top_left_index.row()

        if 'frequency' in con_df and COL_FREQUENCY in self.columns:
            index = self.columns.index(COL_FREQUENCY)
            new_idx = self.model.index(row, index)
            freq = con_df['frequency']
            self.model.setData(new_idx, f'{freq:.6f}')

        if 'nodal_factor' in con_df and COL_NODAL_FACTOR in self.columns:
            index = self.columns.index(COL_NODAL_FACTOR)
            new_idx = self.model.index(row, index)
            nf = con_df['nodal_factor']
            self.model.setData(new_idx, f'{nf:.6f}')

        if 'equilibrium_argument' in con_df and COL_EQUILIBRIUM_ARGUMENT in self.columns:
            index = self.columns.index(COL_EQUILIBRIUM_ARGUMENT)
            new_idx = self.model.index(row, index)
            eq_arg = con_df['equilibrium_argument']
            self.model.setData(new_idx, f'{eq_arg:.6f}')

        if 'earth_tide_reduction_factor' in con_df and COL_ETRF in self.columns:
            index = self.columns.index(COL_ETRF)
            new_idx = self.model.index(row, index)
            etrf = con_df['earth_tide_reduction_factor']
            self.model.setData(new_idx, f'{etrf:.6f}')

        if COL_NAME in self.columns:
            index = self.columns.index(COL_NAME)
            new_idx = self.model.index(row, index)
            self.model.setData(new_idx, constituent_name)

        if 'amplitude' in con_df and COL_POTENTIAL_AMPLITUDE in self.columns:
            index = self.columns.index(COL_POTENTIAL_AMPLITUDE)
            new_idx = self.model.index(row, index)
            amp = con_df['amplitude']
            self.model.setData(new_idx, f'{amp:.6f}')

    def on_force_velocity_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index. Unused.
            bottom_right_index (QModelIndex): Bottom right index. Same as top_left_index. QxPandasTableModel only
                supports single cell editing. Unused.
        """
        # If you don't do this, the conditional dataset columns don't update their display when you click their
        # checkboxes. I have no idea why it works or why it's necessary and I only figured it out through dumb luck.
        # Opening the Constituent combobox also works if it's there, but that column isn't always present. Selecting
        # the Constituent column also works, but opens the combobox for some reason, and, again, it isn't always
        # present. Repainting the table seemed to work once, but never again.
        columns_to_update = [i for i, column in enumerate(self.columns) if column in CONDITIONAL_DATASET_COLUMNS]
        for column in columns_to_update:
            self.table_view.selectColumn(column)
        self.table_view.clearSelection()

    def on_ref_date_changed(self, new_ref_date):
        """Update the constituent properties when the ADCIRC interpolation reference date changes.

        Args:
            new_ref_date (QDateTime): The new interpolation reference date.
        """
        self.reftime = new_ref_date.toPython()
        for row in range(self.model.rowCount()):
            index = self.model.index(row, COL_CONSTITUENT)
            self.on_constituent_changed(index, index)
