"""Code for creating polygons from arcs."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math

# 2. Third party modules
from geopandas import GeoDataFrame
from matplotlib.patches import Ellipse
from pyproj.enums import WktVersion
from shapely import reverse
from shapely.geometry import LineString, Point, Polygon
from shapely.geometry.base import JOIN_STYLE
from shapely.ops import nearest_points


# 3. Aquaveo modules
from xms.gdal.utilities import gdal_utils as gu

# 4. Local modules
from xms.tool.utilities.coverage_conversion import convert_lines_to_coverage


class ArcData:
    """Class to hold arc data."""

    def __init__(self, arc_data):
        """Initializes the class.

        Args:
            arc_data (dict): arc points and attributes
            ugrid (UGrid): unstructured grid/mesh
        """
        self.arc_pts = arc_data.get('arc_pts', None)
        self.id = arc_data.get('id', -1)
        self.r_chan = []
        self.l_chan = []
        self.element_width = arc_data.get('element_width', 1.0)
        self.number_elements = arc_data.get('number_elements', 2)
        self.start_node = arc_data.get('start_node', -1)
        self.end_node = arc_data.get('end_node', -1)
        self.bias = arc_data.get('bias', 1.0)
        self.min_seg_length = None
        self.min_allowed_seg_length = None

        self._up_left_trim_pt = None
        self._up_left_trim_leftovers = []

        self._up_right_trim_pt = None
        self._up_right_trim_leftovers = []

        self._down_left_trim_pt = None
        self._down_left_trim_leftovers = []

        self._down_right_trim_pt = None
        self._down_right_trim_leftovers = []

        self._down_end_created = False
        self._up_end_created = False

    def set_trim_offset_pt(self, is_left, is_up, pt):
        """Sets where to trim the offset.

        Args:
            is_left (bool): is left offset
            is_up (bool): is upstream end
            pt (Point): shapely point of where to trim the offset
        """
        if is_left is True:
            if is_up is True:
                self._up_left_trim_pt = pt
            else:
                self._down_left_trim_pt = pt
        else:
            if is_up is True:
                self._up_right_trim_pt = pt
            else:
                self._down_right_trim_pt = pt

    def set_trim_leftovers(self, is_left, is_up, trim_pts):
        """Sets where to trim the offset.

        Args:
            is_left (bool): is left offset
            is_up (bool): is upstream end
            trim_pts (list): list of trimmed points
        """
        if is_left is True:
            if is_up is True:
                self._up_left_trim_leftovers = trim_pts
            else:
                self._down_left_trim_leftovers = trim_pts
        else:
            if is_up is True:
                self._up_right_trim_leftovers = trim_pts
            else:
                self._down_right_trim_leftovers = trim_pts

    def get_trim_offset_pt(self, is_left, is_up):
        """Sets where to trim the offset.

        Args:
            is_left (bool): is left offset
            is_up (bool): is upstream end

        Returns:
            pt (Point): shapely point of where to trim the offset
        """
        if is_left is True:
            return self._up_left_trim_pt if is_up else self._down_left_trim_pt
        else:
            return self._up_right_trim_pt if is_up else self._down_right_trim_pt

    def get_trim_leftovers(self, is_left, is_up):
        """Sets where to trim the offset.

        Args:
            is_left (bool): is left offset
            is_up (bool): is upstream end

        Returns:
            trim_pts (list): list of trimmed points
        """
        if is_left is True:
            return self._up_left_trim_leftovers if is_up else self._down_left_trim_leftovers
        else:
            return self._up_right_trim_leftovers if is_up else self._down_right_trim_leftovers

    def get_offset_pts(self, is_left):
        """Sets where to trim the offset.

        Args:
            is_left (bool): is left offset

        Returns:
            pts (list): list of xyz points for offset
        """
        return self.l_chan if is_left is True else self.r_chan

    def has_valid_offsets(self):
        """Sets intersection flags to False.

        Returns:
            (bool): has valid offsets
        """
        return len(self.l_chan) > 1 and len(self.r_chan) > 1

    def has_any_offsets(self):
        """Sets intersection flags to False.

        Returns:
            (bool): has valid offsets
        """
        return len(self.l_chan) > 0 and len(self.r_chan) > 0

    def is_loop(self):
        """Returns True if the arc is a loop.

        Returns:
            (bool): is a loop
        """
        return self.start_node == self.end_node

    def intersects_both_ends(self, is_left):
        """Returns True if the offset has been intersected at both upstream and downstream ends.

        Args:
            is_left (bool): is left offset
        Returns:
            (bool): is a loop
        """
        if is_left:
            return self._down_left_trim_pt is not None and self._up_left_trim_pt is not None
        else:
            return self._down_right_trim_pt is not None and self._up_right_trim_pt is not None

    def clear_offsets(self):
        """Sets intersection flags to False.

        Returns:
            (bool): has valid offsets
        """
        self.l_chan.clear()
        self.r_chan.clear()

    def get_arc_end_loc(self, is_up):
        """Gets original location of arc end.

        Args:
            is_up (bool): is upstream end
        Returns:
            loc (list): xyz location of end point
        """
        return self.arc_pts[0] if is_up is True else self.arc_pts[-1]

    def get_arc_end_node(self, is_up):
        """Gets original location of arc end.

        Args:
            is_up (bool): is upstream end
        Returns:
            loc (list): xyz location of end point
        """
        return self.start_node if is_up else self.end_node

    def set_offset_end_loc(self, is_left, is_up, loc):
        """Gets original location of arc end.

        Args:
            is_left (bool): is left offset
            is_up (bool): is upstream end
            loc (list): location of endpoint
        """
        end = 0 if is_up else -1
        if is_left is True:
            self.l_chan[end] = loc
        else:
            self.r_chan[end] = loc

    def is_node_upstream(self, node_id):
        """Sets intersection flags to False.

        Args:
            node_id (int): node id
        Returns:
            (bool): is it the start node
        """
        return self.start_node == node_id


class PolygonsFromArcs:
    """Class to do polygon arcs from arcs in a coverage."""

    def __init__(self, arc_data, new_cov_name, logger, pinch_ends, min_seg_length, wkt):
        """Initializes the class.

        Args:
            arc_data (list(dict)): arc points and attributes
            new_cov_name (string): the new coverage name
            logger (logging.Logger): optional logger
            pinch_ends (bool): should we pinch the free ends of polygons?
            min_seg_length (float): if not None, value to override calculated minimum segment length
            wkt (string): projection info
        """
        self._logger = None
        self._pinch_ends = pinch_ends
        self._override_segment_length = min_seg_length
        self._max_node_id = 0
        self._arc_data_list = arc_data
        self._orig_arcs = set()
        self._arc_data = None
        self._arc_to_data = dict()
        self._poly_arcs = []
        self._cov_geom = None if len(arc_data) < 1 else arc_data[0].get('cov_geom', None)
        self._node_to_arc = dict()
        self._node_to_shapely_pt = dict()
        self._checked_nodes = set()
        self._node_intersection_groups = []
        self._node_to_merge_arc = dict()
        self.new_cov = None
        self._logger = logging.getLogger('xms.ewn') if logger is None else logger
        self._calc_node_to_arc_lookup()
        self._new_cov_name = new_cov_name
        self._processed = dict()
        self._checked_for_intersections = set()
        self._extra_arcs_for_node_merge = dict()
        self._confluence_nodes = set()

        self._shared_ends = set()

        self._size_warning_threshold = 0.1

        # global parameters
        self._mitre_limit = 2.5

        # info for spacing warning
        self._do_size_warning = self._override_segment_length is None
        if self._do_size_warning and gu.valid_wkt(wkt):
            # look in GdalUtility.cpp - LaUnit imUnitFromProjectionWKT(const std::string wkt)
            sr = gu.wkt_to_sr(wkt)
            if sr.IsGeographic():
                self._size_warning_threshold = 0.0000001
            elif not sr.IsLocal():
                self._size_warning_threshold = 1.0
        self._wkt = wkt
        self._too_many_points = set()
        self._too_small_spacing = set()

        # lists for error report at end
        self._concave_arcs = set()
        self._overlapping_arcs = set()
        self._single_segments = set()
        self._short_arcs = set()

    def _get_num_segments_warning_threshold(self):
        """Creates polygons by offsetting arcs."""
        return 10000

    def generate_coverage(self):
        """Creates polygons by offsetting arcs.

        Returns:
            (GeoDataFrame): The new coverage
        """
        # get a list of the original arcs
        for arc in self._arc_data_list:
            self._orig_arcs.add(arc['id'])

        # merge arcs when there are exactly two that share a node
        self._logger.info('Merging adjacent arcs...')
        delete_nodes = set()
        for node_id in self._node_to_arc.keys():
            arc_list = self._node_to_arc[node_id]
            if len(arc_list) == 2 and self._merge_two_way(arc_list, node_id) is True:
                delete_nodes.add(node_id)
        for node_id in delete_nodes:
            del self._node_to_arc[node_id]

        # remove small segments
        self._logger.info('Removing small segments...')
        for arc_data in self._arc_data_list:
            self._check_min_segments_on_data(self._get_arc_data_from_arc(arc_data))

        if len(self._too_many_points) > 0:
            self._report_warning(self._too_many_points, 'The following arcs have more than 10,000 segments. Consider '
                                                        'aborting and redistributing vertices to a larger spacing for '
                                                        'faster computation.')
        if len(self._too_small_spacing) > 0:
            self._report_warning(self._too_small_spacing, 'The following arcs have segments smaller than '
                                                          f'{self._size_warning_threshold}. Consider aborting and '
                                                          'redistributing vertices to a larger spacing for faster '
                                                          'computation.')

        # split arcs where they will have offsets that overlap with other arcs' offsets
        self._logger.info('Splitting overlapping arcs...')
        for arc_data in self._arc_data_list:
            self._arc_data = self._get_arc_data_from_arc(arc_data)
            self._detect_potential_offset_intersections()

        # create all offsets
        self._logger.info('Creating offsets...')
        for arc_data in self._arc_data_list:
            self._arc_data = self._get_arc_data_from_arc(arc_data)
            self._create_offsets()

        # detect node intersections
        self._logger.info('Detecting node intersections...')
        arc_data = self._get_arc_data_from_arc(self._arc_data_list[0])
        distance = arc_data.element_width * arc_data.number_elements * 1.5
        for node_id in self._node_to_arc.keys():
            node_group = set()
            self._check_node_for_intersections(node_id, node_group, distance)
            if len(node_group) > 0:
                self._node_intersection_groups.append(node_group)

        # merge the node intersections
        self._logger.info('Merging node intersections...')
        for node_group in self._node_intersection_groups:
            if len(node_group) > 0:
                self.merge_node_intersections(node_group, [])

        # add extra arcs needed from the node intersections
        for cur_arc_id in self._extra_arcs_for_node_merge.keys():
            if cur_arc_id in self._arc_to_data.keys():
                cur_data = self._arc_to_data[cur_arc_id]
                if cur_data.has_any_offsets():
                    for connecting_arc in self._extra_arcs_for_node_merge[cur_arc_id]:
                        if connecting_arc[0] in self._arc_to_data.keys():
                            connecting_data = self._arc_to_data[connecting_arc[0]]
                            if connecting_data.has_any_offsets():
                                self._poly_arcs.append(connecting_arc[1])

        self._logger.info('Creating polygon arcs...')
        for arc_data in self._arc_data_list:
            self._arc_data = self._get_arc_data_from_arc(arc_data)
            if self._arc_data.has_valid_offsets():
                self._create_polygon_arcs()

        self._create_coverage()

        # report any errors
        if len(self._concave_arcs) > 0:
            self._report_warning(self._concave_arcs, 'The following arcs were too concave for the specified offset '
                                 'and were ignored. Try decreasing the width.')

        if len(self._overlapping_arcs) > 0:
            self._report_warning(self._overlapping_arcs, 'The following arcs overlapped and need to be reviewed. '
                                 'Try decreasing the width to avoid overlapping arcs.')

        if len(self._single_segments) > 0:
            self._report_warning(self._single_segments, 'The following arcs were standalone with only one segment and '
                                 'were ignored.')

        if len(self._short_arcs) > 0:
            self._report_warning(self._short_arcs, 'The following arcs were too short and were ignored.')

        return self.new_cov

    def _report_warning(self, error_ids, error_msg):
        """Creates polygons by offsetting arcs.

        Arguments:
            error_ids (set): list of arc ids
            error_msg (str): error message
        """
        error_ids = sorted(error_ids)
        warning_ids = []
        for id in error_ids:
            # only want to report on arcs that were originally there, not ones we created by splitting
            if id in self._orig_arcs:
                warning_ids.append(id)

        if len(warning_ids) > 0:
            self._logger.warning(error_msg)
            arc_ids = f'{warning_ids[0]}'
            for i in range(1, len(warning_ids)):
                arc_ids = f'{arc_ids}, {warning_ids[i]}'
            self._logger.warning(arc_ids)

    def _calc_node_to_arc_lookup(self):
        """Build a dict with node id as key and list of arcs as data."""
        for arc in self._arc_data_list:
            arc_list = self._node_to_arc.get(arc['start_node'], [])
            arc_list.append(arc)
            self._node_to_arc[arc['start_node']] = arc_list
            arc_list = self._node_to_arc.get(arc['end_node'], [])
            arc_list.append(arc)
            self._node_to_arc[arc['end_node']] = arc_list

            self._node_to_shapely_pt[arc['start_node']] = loc_to_shapely_pt(arc['arc_pts'][0])
            self._node_to_shapely_pt[arc['end_node']] = loc_to_shapely_pt(arc['arc_pts'][-1])

        self._max_node_id = max(list(self._node_to_arc.keys())) + 1

    def _create_coverage(self):
        """Creates a coverage from the generated polygon arcs."""
        self._logger.info('Creating coverage...')
        wkt = self._wkt
        if self._cov_geom is not None and self._cov_geom.crs is not None:
            wkt = self._cov_geom.crs.to_wkt(WktVersion.WKT1_GDAL)
        self.new_cov = convert_lines_to_coverage(self._poly_arcs, self._new_cov_name, wkt)

    def _check_min_segments_on_data(self, arc_data):
        """Offset the input arc to the left and right based on the width.

        Args:
            arc_data (ArcData): data for the arc

        """
        pts = arc_data.arc_pts
        arc_ls = LineString(pts)
        tot_length = arc_ls.length
        if self._override_segment_length is not None:
            arc_data.min_allowed_seg_length = self._override_segment_length
        else:
            arc_data.min_allowed_seg_length = 0.05 * (tot_length / (len(pts) - 1))
        self._check_min_segments_on_pts(pts, arc_data.min_allowed_seg_length, arc_data.id)

    def _check_min_segments_on_pts(self, arc_pts, min_allowed_seg_length, arc_id):
        """Offset the input arc to the left and right based on the width.

        Args:
            arc_pts (list): pts for the arc
            min_allowed_seg_length (float): minimum allowed segment length
            arc_id (int): arc id for warning

        """
        pt_prev = loc_to_shapely_pt(arc_pts[0])
        i = 1
        add_warning = self._do_size_warning
        while i < len(arc_pts) and len(arc_pts) > 2:
            pt_cur = loc_to_shapely_pt(arc_pts[i])
            dist = pt_prev.distance(pt_cur)
            advance = True
            # add warning if this is too small
            if add_warning and dist < self._size_warning_threshold:
                self._too_small_spacing.add(arc_id)
                add_warning = False
            if dist < min_allowed_seg_length:
                # remove this segment
                if i == len(arc_pts) - 1:
                    del arc_pts[i - 1]
                    pt_prev = arc_pts[i - 2]
                elif i == 1:
                    del arc_pts[i]
                else:
                    # if the next segment is also too small, delete this point
                    pt_next = loc_to_shapely_pt(arc_pts[i + 1])
                    dist = pt_cur.distance(pt_next)
                    if dist < min_allowed_seg_length:
                        del arc_pts[i]
                    else:
                        # get the midpoint of the small segment
                        arc_pts[i] = [(arc_pts[i][0] + arc_pts[i - 1][0]) / 2.0,
                                      (arc_pts[i][1] + arc_pts[i - 1][1]) / 2.0, 0.0]
                        del arc_pts[i - 1]
                        pt_prev = loc_to_shapely_pt(arc_pts[i - 1])
                # check it again
                i -= 1
                advance = False

            if advance is True:
                pt_prev = pt_cur

            i += 1

        if len(arc_pts) > self._get_num_segments_warning_threshold():
            self._too_many_points.add(arc_id)

    def _create_offsets(self):
        """Offset the input arc to the left and right based on the width.

        Returns:
            (bool): success
        """
        offset_dist = (self._arc_data.element_width * self._arc_data.number_elements) / 2.0
        self._arc_data.r_chan, self._arc_data.l_chan = \
            self._get_offset_points(self._arc_data.arc_pts, offset_dist,
                                    min_allowed_seg_length=self._arc_data.min_allowed_seg_length)
        return len(self._arc_data.r_chan) > 1 and len(self._arc_data.l_chan) > 1

    def _get_offset_points(self, arc_pts, offset_dist, min_allowed_seg_length=None, do_redist=True):
        """Offset the input arc to the left and right based on the offset_dist.

        Args:
            arc_pts (list): the points of the arc to offset
            offset_dist (float): offset max_dist
            min_allowed_seg_length (float): minimum allowed length of segment

        Returns:
            (list): the points of the right offset arc
            (list): the points of the left offset arc
        """
        is_loop = arc_pts[0] == arc_pts[-1]
        if is_loop:
            right_offset, left_offset = self._get_buffer_points_loop(arc_pts, offset_dist)
        else:
            right_offset, left_offset = self._get_buffer_points_arc(arc_pts, offset_dist)

        if len(left_offset) == 0 or len(right_offset) == 0:
            # in this case the arc is concave and degenerated to nothing
            self._concave_arcs.add(self._arc_data.id)
            return [], []

        intersect_dist = (self._mitre_limit + 0.1) * offset_dist
        if do_redist:
            right_offset = self._redist_offset_to_match_orig_arc(right_offset, arc_pts, -intersect_dist,
                                                                 min_allowed_seg_length)
            left_offset = self._redist_offset_to_match_orig_arc(left_offset, arc_pts, intersect_dist,
                                                                min_allowed_seg_length)

        return right_offset, left_offset

    def _get_buffer_points_arc(self, arc_pts, offset_dist):
        """Offset the input arc to the left and right based on the offset_dist.

        Args:
            arc_pts (list): the points of the arc to offset
            offset_dist (float): offset max_dist

        Returns:
            (list): the points of the right offset arc
            (list): the points of the left offset arc
        """
        arc_line = LineString(arc_pts)

        buffer = arc_line.buffer(offset_dist, join_style=JOIN_STYLE.mitre, mitre_limit=self._mitre_limit)
        all_pts = list(buffer.exterior.coords)

        # some z values turn out nan
        offset_pts = [(p[0], p[1], 0.0) for p in all_pts]
        offset_line = LineString(offset_pts)

        intersect_dist = (self._mitre_limit + 0.1) * offset_dist

        # move the node to where it won't interfere (the rounded part at one of the ends)
        parallel = self._get_parallel_arc(arc_pts[1], arc_pts[0], arc_pts[0], intersect_dist)
        new_node_loc = offset_line.intersection(parallel)
        offset_1, offset_2, _, _ = self._split_lines_ls(offset_line, parallel, new_node_loc)
        new_offset_order = offset_2 + offset_1
        offset_line = LineString(new_offset_order)

        # start with the last left point
        left_last, int_line = self._find_offset_end_point(arc_pts[-2], arc_pts[-1], arc_pts[-1], intersect_dist, 5.0,
                                                          offset_line)
        if left_last is not None:
            tmp_pts, _, _, _ = self._split_lines_ls(offset_line, int_line, left_last)
            tmp_pts.append(shapely_pt_to_loc(left_last))
            tmp_line = LineString(tmp_pts)

        # now cut off at the first left offset point
        left_first, int_line = self._find_offset_end_point(arc_pts[0], arc_pts[1], arc_pts[0], intersect_dist, 5.0,
                                                           offset_line)
        left_offset = []
        if left_first is not None:
            _, left_offset, _, _ = self._split_lines_ls(tmp_line, int_line, left_first)

        # right offset - start with first point
        right_first, int_line = self._find_offset_end_point(arc_pts[0], arc_pts[1], arc_pts[0], -intersect_dist, 5.0,
                                                            offset_line)

        tmp_pts, _, _, _ = self._split_lines_ls(offset_line, int_line, right_first)
        tmp_line = LineString(tmp_pts)

        right_last, int_line = self._find_offset_end_point(arc_pts[-2], arc_pts[-1], arc_pts[-1], -intersect_dist, -5.0,
                                                           offset_line)

        right_offset = []
        _, right_offset, _, _ = self._split_lines_ls(tmp_line, int_line, right_last)
        right_offset.reverse()

        return right_offset, left_offset

    def _get_buffer_points_loop(self, arc_pts, offset_dist):
        """Offset the input arc to the left and right based on the offset_dist.

        Args:
            arc_pts (list): the points of the arc to offset
            offset_dist (float): offset max_dist

        Returns:
            (list): the points of the right offset arc
            (list): the points of the left offset arc
        """
        loop = Polygon(arc_pts)
        right_offset_dist = offset_dist
        left_offset_dist = offset_dist
        if loop.exterior.is_ccw is True:
            left_offset_dist = -offset_dist
        else:
            right_offset_dist = -offset_dist

        r_buffer = loop.buffer(right_offset_dist, join_style=JOIN_STYLE.mitre, mitre_limit=self._mitre_limit)
        l_buffer = loop.buffer(left_offset_dist, join_style=JOIN_STYLE.mitre, mitre_limit=self._mitre_limit)

        do_split = False
        if r_buffer.geom_type == 'MultiPolygon' or l_buffer.geom_type == 'MultiPolygon':
            do_split = True
        elif len(r_buffer.exterior.coords) == 0 or len(l_buffer.exterior.coords) == 0:
            do_split = True

        if do_split:
            self._split_loop()
            return self._get_buffer_points_arc(self._arc_data.arc_pts, offset_dist)

        right_pts = [(p[0], p[1], 0.0) for p in list(r_buffer.exterior.coords)]
        left_pts = [(p[0], p[1], 0.0) for p in list(l_buffer.exterior.coords)]

        if loop.exterior.is_ccw is True:
            right_pts.reverse()
            left_pts.reverse()

        # make sure that the offset first points match with the arc first point - bug_14050_int_with_loop_2
        offsets = [right_pts, left_pts]
        s_pt1 = arc_pts[-2]
        s_pt2 = arc_pts[0]
        s_pt3 = arc_pts[1]
        for i_offset, offset in enumerate(offsets):
            p = None
            if i_offset == 0:
                p = self._get_perp_arc_3_pts(s_pt1, s_pt2, s_pt3, s_pt2, -offset_dist * (self._mitre_limit + 0.1))
            else:
                p = self._get_perp_arc_3_pts(s_pt1, s_pt2, s_pt3, s_pt2, offset_dist * (self._mitre_limit + 0.1))
            offset_ls = LineString(offset)
            intersect = offset_ls.intersection(p)

            if intersect is not None and intersect.geom_type == 'MultiPoint':
                min_dist = offset_dist
                closest = None
                start_pt = loc_to_shapely_pt(s_pt2)
                for int_pt in intersect.geoms:
                    dist = start_pt.distance(int_pt)
                    if closest is None or dist < min_dist:
                        closest = int_pt
                        min_dist = dist
                intersect = closest

            if intersect is not None and intersect.geom_type == 'Point':
                # insert this point on the offset and make it the first one
                new_offset = []
                new_offset.append(shapely_pt_to_loc(intersect))
                offset_length = offset_ls.length
                tol = 0.0001 * offset_length
                proj_dist = offset_ls.project(intersect)
                diff = min(offset_length - proj_dist, proj_dist)
                if diff > tol:
                    for i in range(len(offset)):
                        cur_pt = loc_to_shapely_pt(offset[i])
                        cur_dist = offset_ls.project(cur_pt)
                        if cur_dist >= proj_dist:
                            # found where to continue the line
                            new_offset = new_offset + offset[i:-1] + offset[:i]
                            new_offset.append(new_offset[0])
                            break
                    offsets[i_offset] = new_offset

        return offsets[0], offsets[1]

    def _get_closest_intersection(self, ls_1, ls_2, closest_pt):
        """Offset the input arc to the left and right based on the offset_dist.

        Args:
            ls_1 (LineString): line to intersect
            ls_2 (LineString): other line to intersect
            closest_pt (list): return intersection point closest to this point

        Returns:
            (Point): intersection point
        """
        intersect = ls_1.intersection(ls_2)

        if intersect is not None and intersect.geom_type == 'MultiPoint':
            min_dist = None
            closest = None
            start_pt = loc_to_shapely_pt(closest_pt)
            for int_pt in intersect.geoms:
                dist = start_pt.distance(int_pt)
                if closest is None or dist < min_dist:
                    closest = int_pt
                    min_dist = dist
            intersect = closest

        if intersect is not None and intersect.geom_type != 'Point':
            intersect = None

        return intersect

    def _redist_offset_to_match_orig_arc(self, t, s, offset, min_allowed_seg_length=None):
        """Redistribute the target arc.

        Args:
            t (list): the points of the target arc to redistribute
            s (list): points of the source arc
            offset (float): offset max_dist
            min_allowed_seg_length (float): minimum allowed segment length

        Returns:
            (list): the points of the redistributed arc
        """
        if min_allowed_seg_length is None:
            # get the min segment length and the intersect max_dist for redistribution
            arc_line = LineString(t)
            min_allowed_seg_length = arc_line.length
            valid_length = 0.05 * (arc_line.length / (len(t) - 1))
            for i in range(1, len(t)):
                dist = loc_to_shapely_pt(t[i]).distance(loc_to_shapely_pt(t[i - 1]))
                if dist < min_allowed_seg_length and dist > valid_length:
                    min_allowed_seg_length = dist

            new_spacing = min_allowed_seg_length / 20.0
            t = add_vertices_to_arc(t, new_spacing)

            if len(t) <= len(s):
                valid_length = 0.05 * (arc_line.length / (len(s) - 1))
                for i in range(1, len(s)):
                    dist = loc_to_shapely_pt(s[i]).distance(loc_to_shapely_pt(s[i - 1]))
                    if dist < min_allowed_seg_length and dist > valid_length:
                        min_allowed_seg_length = dist
                new_spacing = min_allowed_seg_length / 20.0
                t = add_vertices_to_arc(t, new_spacing)
        else:
            new_spacing = min_allowed_seg_length / 20.0
            t = add_vertices_to_arc(t, new_spacing)

        tgt_ls = LineString(t)
        src_ls = LineString(s)

        new_pt_ranges = []
        # not worrying about the first and last points
        for i in range(1, len(s) - 1):
            i_1, i_2 = self._get_pt_range(t, tgt_ls, s, src_ls, i, offset, new_spacing)
            new_pt_ranges.append([i_1, i_2])

        # put the new points in the middle of their ranges
        mid_indices = [0]
        mid_indices.extend([round((pt_r[0] + pt_r[1]) / 2) if pt_r[0] != - 1 else pt_r[1] for pt_r in new_pt_ranges])

        # check that the list is valid going forward
        prev_pt = -1
        new_pt_indices = []
        for i in range(len(mid_indices)):
            pt = mid_indices[i]
            if pt <= prev_pt:
                pt = prev_pt + 1

            new_pt_indices.append(pt)
            prev_pt = pt

        # add the last point
        new_pt_indices.append(len(t) - 1)

        # make sure we will be valid backwards
        if new_pt_indices[-2] > new_pt_indices[-1]:
            index = len(new_pt_indices) - 2
            next_pt = new_pt_indices[-1]
            while new_pt_indices[index] >= next_pt and index >= 0:
                new_pt_indices[index] = next_pt - 1
                next_pt = new_pt_indices[index]
                index = index - 1

        # get the point locations in a list
        return [t[index] for index in new_pt_indices]

    def _get_pt_range(self, t, t_ls, s, s_ls, s_idx, off_dist, spacing):
        """Redistribute the target arc.

        Args:
            t (list): the points of the target arc to redistribute
            t_ls (LineString): target linestring
            s (list): points of the source arc
            s_ls (LineString): source LineString
            s_idx (int): index of vertex on source arc
            off_dist (float): offset max_dist
            spacing (float): spacing between vertices on target arc

        Returns:
            (int): min valid index for new point on target arc
            (int): max valid index for new point on target arc
        """
        frm_s = -1
        frm_t = -1
        frm_t_f = frm_t_b = -1

        p = self._get_perp_arc_3_pts(s[s_idx - 1], s[s_idx], s[s_idx + 1], s[s_idx], off_dist)
        intersect = t_ls.intersection(p)

        # if it intersected more than once, take the closest point
        if intersect is not None and intersect.geom_type == 'MultiPoint':
            min_dist = off_dist
            closest = None
            start_pt = loc_to_shapely_pt(s[s_idx])
            for int_pt in intersect.geoms:
                dist = start_pt.distance(int_pt)
                if closest is None or dist < min_dist:
                    closest = int_pt
                    min_dist = dist
            intersect = closest

        if intersect is None or intersect.geom_type != 'Point':
            return -1, -1

        proj_dist = t_ls.project(intersect)
        frm_s = round(proj_dist / spacing)

        # make sure that we aren't using the beginning or end point
        frm_s = max(frm_s, 1)
        frm_s = min(frm_s, len(t) - 2)

        # where does a perpendicular line from this point on the target arc land on the source arc?
        src_pt_prj = s_ls.project(loc_to_shapely_pt(s[s_idx]))
        p = self._get_perp_arc_3_pts(t[frm_s - 1], t[frm_s], t[frm_s + 1], t[frm_s], -off_dist)
        intersect = s_ls.intersection(p)
        min_diff_forward = min_diff_backward = s_ls.length

        # chk_f is best idx going forward, chk_b is best idx going back
        chk_f = chk_b = frm_s
        if intersect is not None and intersect.geom_type == 'Point':
            diff = abs(s_ls.project(intersect) - src_pt_prj)
            # check if the points going forward get better
            while (diff < min_diff_forward):
                min_diff_forward = diff
                frm_t_f = chk_f

                chk_f += 1
                if chk_f < len(t) - 1:
                    p = self._get_perp_arc_3_pts(t[chk_f - 1], t[chk_f], t[chk_f + 1], t[chk_f], -off_dist)
                    intersect = s_ls.intersection(p)
                    if intersect is not None and intersect.geom_type == 'Point':
                        diff = abs(s_ls.project(intersect) - src_pt_prj)

        chk_b = frm_s - 1
        if chk_b > 0:
            p = self._get_perp_arc_3_pts(t[chk_b - 1], t[chk_b], t[chk_b + 1], t[chk_b], -off_dist)
            intersect = s_ls.intersection(p)
            if intersect is not None and intersect.geom_type == 'Point':
                diff = abs(s_ls.project(intersect) - src_pt_prj)
                while (diff < min_diff_backward):
                    min_diff_backward = diff
                    frm_t_b = chk_b

                    chk_b -= 1
                    if chk_b > 0:
                        p = self._get_perp_arc_3_pts(t[chk_b - 1], t[chk_b], t[chk_b + 1], t[chk_b], -off_dist)
                        intersect = s_ls.intersection(p)
                        if intersect is not None and intersect.geom_type == 'Point':
                            diff = abs(s_ls.project(intersect) - src_pt_prj)

        if frm_t_b != -1 and frm_t_f != -1:
            if min_diff_backward < min_diff_forward:
                frm_t = frm_t_b
            else:
                frm_t = frm_t_f
        elif frm_t_b != -1:
            frm_t = frm_t_b
        else:
            frm_t = frm_t_f

        return min(frm_s, frm_t), max(frm_s, frm_t)

    def _get_perpendicular_arc(self, pt1, pt2, start_pt, length):
        """Redistribute the target arc.

        Args:
            pt1 (list): xyz first point
            pt2 (list): xyz second point
            start_pt (list): starting location for arc
            length (float): length of arc in each direction

        Returns:
            (LineString): perpendicular line
        """
        dx = pt2[0] - pt1[0]
        dy = pt2[1] - pt1[1]
        int_angle = math.atan2(dx, -dy)

        intersect_begin = [start_pt[0], start_pt[1]]

        intersect_end = [start_pt[0] + length * math.cos(int_angle), start_pt[1] + length * math.sin(int_angle)]

        return LineString([intersect_begin, intersect_end])

    def _find_offset_end_point(self, pt1, pt2, start_pt, length, inc, the_ls):
        """Redistribute the target arc.

        Args:
            pt1 (list): xyz first point
            pt2 (list): xyz second point
            start_pt (list): starting location for arc
            length (float): length of arc in each direction
            inc (float): increment for changing the angle until we intersect
            the_ls (LineString): the linestring to intersect with

        Returns:
            (Point): point of intersection
            (LineString): line that intersects the offset
        """
        dx = pt2[0] - pt1[0]
        dy = pt2[1] - pt1[1]

        int_pt = None
        int_ls = None

        # start out perpendicular
        int_angle = math.atan2(dx, -dy)
        itr = 0
        inc_rad = math.pi * inc / 180.0
        intersect_begin = [start_pt[0], start_pt[1]]
        while itr < 10:
            intersect_end = [start_pt[0] + length * math.cos(int_angle), start_pt[1] + length * math.sin(int_angle)]
            int_ls = LineString([intersect_begin, intersect_end])

            int_pt = self._get_closest_intersection(the_ls, int_ls, start_pt)
            if int_pt is not None:
                break

            int_angle -= inc_rad
            itr += 1

        return int_pt, int_ls

    def _get_perp_arc_3_pts(self, pt1, pt2, pt3, start_pt, length):
        """Redistribute the target arc.

        Args:
            pt1 (list): xyz first point
            pt2 (list): xyz second point
            pt3 (list): xyz third point
            start_pt (list): starting location for arc
            length (float): length of arc

        Returns:
            (LineString): perpendicular line
        """
        ls_1 = LineString([pt1, pt2])
        ls_2 = LineString([pt2, pt3])
        diff = ls_1.length - ls_2.length
        if diff > 0:
            pt1 = shapely_pt_to_loc(ls_1.interpolate(diff))
        elif diff < 0:
            pt3 = shapely_pt_to_loc(ls_2.interpolate(ls_1.length))

        return self._get_perpendicular_arc(pt1, pt3, start_pt, length)

    def _get_angle_of_segment(self, pt1, pt2):
        """Redistribute the target arc.

        Args:
            pt1 (list): xyz first point
            pt2 (list): xyz second point
            pt3 (list): xyz third point
            start_pt (list): starting location for arc
            length (float): length of arc

        Returns:
            (LineString): perpendicular line
        """
        dx = pt2[0] - pt1[0]
        dy = pt2[1] - pt1[1]
        return math.atan2(dy, dx)

    def _get_parallel_arc(self, pt1, pt2, start_pt, length):
        """Redistribute the target arc.

        Args:
            pt1 (list): xyz first point
            pt2 (list): xyz second point
            start_pt (list): starting location for arc
            length (float): length of arc in each direction
            both_dirs (bool): extend the arc in both directions

        Returns:
            (LineString): parallel line
        """
        angle = self._get_angle_of_segment(pt1, pt2)

        arc_begin = [start_pt[0], start_pt[1], 0.0]

        arc_end = [start_pt[0] + length * math.cos(angle), start_pt[1] + length * math.sin(angle), 0.0]

        return LineString([arc_begin, arc_end])

    def _extend_arc(self, pts, is_up, extend_length):
        """Extend the arc.

        Args:
            pts (list): pts of arc to extend
            is_up (bool): extend at the upstream end
            extend_length (float): length to extend
        Returns:
            (list): list of points
        """
        end_pt = []
        if is_up is True:
            dx = pts[0][0] - pts[1][0]
            dy = pts[0][1] - pts[1][1]
            end_pt = pts[0]
        else:
            dx = pts[-1][0] - pts[-2][0]
            dy = pts[-1][1] - pts[-2][1]
            end_pt = pts[-1]

        int_angle = math.atan2(dy, dx)

        end_pt = [end_pt[0] + extend_length * math.cos(int_angle), end_pt[1] + extend_length * math.sin(int_angle), 0.0]

        if is_up is True:
            pts[0] = end_pt
        else:
            pts[-1] = end_pt

        return pts

    def _create_polygon_arcs(self):
        """Create the points that make up the arc polygon."""
        if self._do_offsets_intersect() is False:
            self._polygon_from_left_right_num_elem()

    def _polygon_from_left_right_num_elem(self):
        """Create a polygon from left and right side points with num_elem to define end channel segments.

        """
        left = self._arc_data.l_chan
        right = self._arc_data.r_chan
        num_elem = self._arc_data.number_elements

        end_is_confluence = self._arc_data.end_node in self._confluence_nodes
        start_is_confluence = self._arc_data.start_node in self._confluence_nodes

        # is this a standalone?
        if len(left) == 2 and not end_is_confluence and not start_is_confluence:
            self._single_segments.add(self._arc_data.id)
            return

        # taper the free ends
        if self._pinch_ends and not end_is_confluence and self._arc_data.is_loop() is False:
            left.remove(left[-1])
            right.remove(right[-1])
            # add the taper arc
            taper_arc = self._calc_taper_pts(self._arc_data, left, right, False)
            if len(taper_arc) > 1:
                self._poly_arcs.append(taper_arc)

        if self._pinch_ends and not start_is_confluence and self._arc_data.is_loop() is False:
            left.remove(left[0])
            right.remove(right[0])
            # add the taper arc
            taper_arc = self._calc_taper_pts(self._arc_data, left, right, True)
            if len(taper_arc) > 1:
                self._poly_arcs.append(taper_arc)

        if len(left) > 1:
            self._poly_arcs.append(left)

            create_end = True
            if end_is_confluence:
                create_end = (self._arc_data.id, False) not in self._shared_ends
            if create_end:
                end_channel = self._calc_end_channel_pts(left[-1], right[-1], num_elem)
                self._poly_arcs.append(end_channel)

            # reverse order on right channel because polygon needs to be ccw
            self._poly_arcs.append(right[::-1])

            num_elem = self._arc_data.number_elements
            create_end = True
            if start_is_confluence:
                create_end = (self._arc_data.id, True) not in self._shared_ends

            if create_end:
                self._poly_arcs.append(self._calc_end_channel_pts(right[0], left[0], num_elem))

        elif len(left) == 1:
            self._poly_arcs.append(self._calc_end_channel_pts(left[0], right[0], num_elem))

    def _calc_taper_pts(self, data, left, right, is_start):
        """Calculate new points on the end of a channel.

        Args:
            data (ArcData): Data for the arc
            left (list): left offset points
            right (list): right offset points
            is_start (bool): True if start node, False if end node

        Returns:
            (list): list of x,y,z locations
        """
        new_pts = []
        left_side = []
        right_side = []
        pt_l = left[0] if is_start else left[-1]
        pt_r = right[0] if is_start else right[-1]
        pt_top = data.arc_pts[0] if is_start else data.arc_pts[-1]

        if data.number_elements < 5:
            new_pts = [pt_l, pt_top, pt_r]
            return new_pts

        pt_l_sh = loc_to_shapely_pt(pt_l)
        pt_r_sh = loc_to_shapely_pt(pt_r)
        pt_top_sh = loc_to_shapely_pt(pt_top)
        center = data.arc_pts[1] if is_start else data.arc_pts[-2]
        height = (loc_to_shapely_pt(center).distance(loc_to_shapely_pt(pt_top))) * 2.0
        if is_start is False:
            height = -height

        # the center of the channel may not be the center for the ellipse if the angles are changing between
        # the next to last and the last segment of the arc - so calculate a new "center" that creates a right angle
        # with the ends of the ellipse
        perp_arc = self._get_perpendicular_arc(pt_l, pt_r, pt_top, height)
        new_center_sh = self._get_ls_intersect(perp_arc, LineString([pt_l, pt_r]), True)
        if new_center_sh is None:
            # the channel is too wide and/or the arc too concave and/or too narrow with not enough vertices to make a
            # valid ellipse, so just do the triangle
            # see test case "covered" and "bug_14050_a"
            left_side = LineString([pt_l, pt_top])
            right_side = LineString([pt_top, pt_r])
        else:
            new_center_loc = shapely_pt_to_loc(new_center_sh)
            angle = self._get_angle_of_segment(new_center_loc, pt_top) * 180.0 / math.pi + 90.0

            left_width = loc_to_shapely_pt(pt_l).distance(new_center_sh) * 2.0
            right_width = loc_to_shapely_pt(pt_r).distance(new_center_sh) * 2.0

            new_height = (new_center_sh.distance(loc_to_shapely_pt(pt_top))) * 2.0

            if is_start is False:
                left_ellipse = Ellipse(new_center_loc, left_width, -new_height, angle=angle)
                right_ellipse = Ellipse(new_center_loc, right_width, -new_height, angle=angle)
            else:
                left_ellipse = Ellipse(new_center_loc, left_width, new_height, angle=angle)
                right_ellipse = Ellipse(new_center_loc, right_width, -new_height, angle=angle)

            l_ellipse_ls = LineString(left_ellipse.get_verts())
            r_ellipse_ls = LineString(right_ellipse.get_verts())

            # cut off this ellipse at pt_l
            splits = self._split_ls_at_pt(l_ellipse_ls, pt_l_sh)
            # get the first half by cutting it off at the "top" pt
            splits = self._split_ls_at_pt(splits[1], pt_top_sh)
            left_side = splits[0]

            # now cut it off at pt_r
            if is_start:
                splits = self._split_ls_at_pt(r_ellipse_ls, pt_r_sh)
                splits = self._split_ls_at_pt(splits[1], pt_top_sh)
                right_side = reverse(splits[0])
            else:
                splits = self._split_ls_at_pt(r_ellipse_ls, pt_top_sh)
                splits = self._split_ls_at_pt(splits[1], pt_r_sh)
                right_side = splits[0]

        # how many points to distribute?
        num_segs_per_side = math.ceil(data.number_elements / 4)
        delta = 1.0 / (num_segs_per_side)

        new_pts.append(pt_l)
        total = 0.0
        for _ in range(1, num_segs_per_side):
            total += delta
            new_pts.append(shapely_pt_to_loc(left_side.interpolate(total, True)))
        new_pts.append(pt_top)

        total = 0.0
        for _ in range(1, num_segs_per_side):
            total += delta
            new_pts.append(shapely_pt_to_loc(right_side.interpolate(total, True)))
        new_pts.append(pt_r)
        return new_pts

    def _calc_end_channel_pts(self, pt0, pt1, num_segments):
        """Calculate new points on the end of a channel.

        Args:
            pt0 (list): x,y,z
            pt1 (list): x,y,z
            num_segments (int): number of segments between pt0 and pt1

        Returns:
            (list): list of x,y,z locations between pt0 and pt1
        """
        new_pts = []
        if num_segments > 1:
            num_seg = int(num_segments / 2)
            if num_seg == 1:
                new_pts.append([(pt0[0] + pt1[0]) / 2, (pt0[1] + pt1[1]) / 2, (pt0[2] + pt1[2]) / 2])
            else:
                bias = self._arc_data.bias if self._arc_data.bias <= 1.0 else 1 / self._arc_data.bias
                # s is parametric max_dist (0-1) from beginning to end of segment
                s = math.pow(10.0, math.log10(bias) / (num_seg - 1))
                x = 1.0
                d = 1.0
                for _ in range(1, num_seg):
                    x *= s
                    d += x
                spacing = [0.0] * (num_seg + 1)
                spacing[-1] = 1.0
                tprime = spacing[1] = 1 / d
                for i in range(2, num_seg):
                    tprime *= s
                    spacing[i] = spacing[i - 1] + tprime
                if self._arc_data.bias < 1.0:
                    old_spacing = spacing.copy()
                    for i in range(len(spacing)):
                        spacing[i] = 1.0 - old_spacing[i]

                mid_pt = [(pt0[0] + pt1[0]) / 2, (pt0[1] + pt1[1]) / 2, (pt0[2] + pt1[2]) / 2]
                d = [pt0[0] - mid_pt[0], pt0[1] - mid_pt[1], pt0[2] - mid_pt[2]]
                for i in range(1, num_seg):
                    t = spacing[i]
                    new_pt = [mid_pt[0] + d[0] * t, mid_pt[1] + d[1] * t, mid_pt[2] + d[2] * t]
                    new_pts.append(new_pt)
                if self._arc_data.bias >= 1.0:
                    new_pts.reverse()
                new_pts.append(mid_pt)

                new_pts2 = []
                d = [pt1[0] - mid_pt[0], pt1[1] - mid_pt[1], pt1[2] - mid_pt[2]]
                for i in range(1, num_seg):
                    t = spacing[i]
                    new_pt = [mid_pt[0] + d[0] * t, mid_pt[1] + d[1] * t, mid_pt[2] + d[2] * t]
                    new_pts2.append(new_pt)
                if self._arc_data.bias < 1.0:
                    new_pts2.reverse()
                new_pts.extend(new_pts2)

        new_pts.insert(0, pt0)
        new_pts.append(pt1)

        return new_pts

    def _check_this_arc(self, to_check_id):
        """Calculate new points on the end of a channel.

        Args:
            to_check_id (int): id of arc in question

        Returns:
            (bool): True if the arc needs to be checked
        """
        if to_check_id == self._arc_data.id:
            return False

        if to_check_id not in self._processed.keys():
            return True

        int_arcs = self._processed[to_check_id]
        return self._arc_data.id in int_arcs

    def _do_offsets_intersect(self):
        """See if the new polygon will intersect with other polygons."""
        cur_offsets = [[LineString(self._arc_data.l_chan)], [LineString(self._arc_data.r_chan)]]
        cur_arc_ls = LineString(self._arc_data.arc_pts)
        has_intersection = False
        set_intersections = set()
        cur_intersecting_data = []
        offset_dist = (self._arc_data.element_width * self._arc_data.number_elements) / 2.0

        # find out which arcs have the potential to intersect
        for arc in self._arc_data_list:
            if not self._check_this_arc(arc['id']):
                continue
            chk_data = self._get_arc_data_from_arc(arc)
            chk_ls = LineString(chk_data.arc_pts)

            if cur_arc_ls.distance(chk_ls) < (offset_dist * 2.0):
                cur_intersecting_data.append(chk_data)

        for i_cur_offset, cur_offset in enumerate(cur_offsets):
            for chk_data in cur_intersecting_data:
                if chk_data.has_valid_offsets() is False:
                    continue
                chk_orig_arc = LineString(chk_data.arc_pts)
                chk_offsets = [LineString(chk_data.l_chan), LineString(chk_data.r_chan)]
                for chk_offset in chk_offsets:
                    for i_piece in range(len(cur_offset)):
                        cur_piece = cur_offset[i_piece]
                        if cur_piece is None or cur_piece.geom_type != 'LineString':
                            continue

                        if cur_piece.distance(chk_offset) > offset_dist:
                            continue

                        offset_lines = [chk_offset]
                        split_lines_cur = self._multi_line_split(cur_piece, offset_lines)
                        num_splits = len(split_lines_cur)

                        # make sure this isn't from just where two arcs meet
                        if num_splits < 2:
                            continue

                        # found a real overlap
                        has_intersection = True
                        set_intersections.add(chk_data.id)

                        # we are going to split both offsets on the cur arc so that they always match
                        i_cur_offset_opposite = 0
                        if i_cur_offset == 0:
                            i_cur_offset_opposite = 1
                        cur_offset_opposite = cur_offsets[i_cur_offset_opposite]

                        new_pieces_cur = []
                        new_pieces_cur_opposite_side = []

                        # is the first section part of the overlap? If so, remove is True
                        # determine this by using an average max_dist since the first or last point may be slightly
                        # overlapping or not
                        tot_d = 0
                        for tmp_pt in split_lines_cur[0].coords:
                            tot_d += chk_orig_arc.distance(Point(tmp_pt))
                        avg_d = tot_d / len(split_lines_cur[0].coords)
                        remove = avg_d <= offset_dist

                        updated_cur_pts_left = self._pts_from_linestring(cur_piece)
                        for i_split, split_line in enumerate(split_lines_cur):
                            split_opposite = cur_offset_opposite[i_piece]
                            is_line = split_opposite is not None and split_opposite.geom_type == 'LineString'
                            if is_line and i_split != num_splits:
                                # get the intersection point on the opposite side
                                # if it's not the last one, the last point is an intersection pt
                                cur_pts = self._pts_from_linestring(split_line)
                                to_trim_pts = self._pts_from_linestring(split_opposite)
                                trim_opposite, trim_leftover = self._trim_arc_to_match(updated_cur_pts_left, cur_pts,
                                                                                       to_trim_pts)
                                split_opposite = LineString(trim_opposite)
                                cur_offset_opposite[i_piece] = None
                                if len(trim_leftover) > 1:
                                    cur_offset_opposite[i_piece] = LineString(trim_leftover)

                                # remove this section from the points we are working on
                                updated_cur_pts_left = [cur_pts[-1]] + updated_cur_pts_left[len(cur_pts) - 1:]

                            new_pieces_cur_opposite_side.append(split_opposite)

                            if remove:
                                # this will be an empty list for the current offset
                                if i_piece == len(cur_offset) - 1 and i_split == num_splits - 1:
                                    # if the end of the arc is not within the offset_dist, don't get rid of it
                                    shp_end_pt = self._get_linestring_endpoint(split_line, False)
                                    d = chk_orig_arc.distance(shp_end_pt)
                                    remove = d <= offset_dist
                                new_pieces_cur.append(None) if remove is True else new_pieces_cur.append(split_line)
                            else:
                                new_pieces_cur.append(split_line)

                            remove = not remove

                        # replace the old piece with the new pieces
                        del cur_offset[i_piece]
                        del cur_offset_opposite[i_piece]
                        for i_insert in range(len(new_pieces_cur)):
                            cur_offset.insert(i_piece + i_insert, new_pieces_cur[i_insert])
                            cur_offset_opposite.insert(i_piece + i_insert, new_pieces_cur_opposite_side[i_insert])

        # store that we have processed this
        self._processed[self._arc_data.id] = set_intersections

        if has_intersection is False:
            return False

        # check on segment length
        for side in [0, 1]:
            opposite_side = 0 if side == 1 else 1
            for i_piece, piece in enumerate(cur_offsets[side]):
                if piece is None:
                    continue
                elif piece.geom_type == 'LineString':
                    pts = self._pts_from_linestring(piece)
                    opposite_piece = cur_offsets[opposite_side][i_piece]
                    opposite_pts = None
                    if opposite_piece is not None and opposite_piece.geom_type == 'LineString':
                        opposite_pts = self._pts_from_linestring(opposite_piece)
                    self._check_last_segment_length(pts, opposite_pts, True)
                    self._check_last_segment_length(pts, opposite_pts, False)

                    # replace with the checked points
                    cur_offsets[side][i_piece] = LineString(pts)
                    if opposite_pts is not None:
                        cur_offsets[opposite_side][i_piece] = LineString(opposite_pts)

        # add the new arc pieces
        for i_piece in range(len(cur_offsets[0])):
            # create LineStrings like normal
            types = []
            offset_pts = []
            for offset_side in cur_offsets:
                if offset_side[i_piece] is None:
                    offset_pts.append(None)
                    types.append('None')
                elif offset_side[i_piece].geom_type == 'LineString':
                    pts = self._pts_from_linestring(offset_side[i_piece])
                    offset_pts.append(pts)
                    piece_type = 'None'
                    if len(pts) > 0:
                        self._poly_arcs.append(pts)
                        piece_type = 'LineString'
                    types.append(piece_type)

            # create the end lines
            num_elem = self._arc_data.number_elements
            if types[0] == 'LineString' and types[1] == 'LineString':
                create_end = True
                if i_piece == 0:
                    create_end = (self._arc_data.id, True) not in self._shared_ends

                if create_end:
                    self._poly_arcs.append(self._calc_end_channel_pts(offset_pts[0][0], offset_pts[1][0],
                                                                      num_elem))

                create_end = True
                if i_piece == len(cur_offsets[0]) - 1:
                    create_end = (self._arc_data.id, False) not in self._shared_ends
                if create_end:
                    self._poly_arcs.append(self._calc_end_channel_pts(offset_pts[0][-1], offset_pts[1][-1],
                                                                      num_elem))

        return True

    def _split_lines(self, line1, line2, int_loc):
        """Creates polygons by offsetting arcs.

        Args:
            line1 (list): first line to be split
            line2 (list): first line to be split
            int_loc (Point): the intersection location

        Returns:
            (list), (list), (list), (list): split arcs
        """
        split1_a = []
        split1_b = []
        split2_a = []
        split2_b = []

        if int_loc is not None:
            i_1 = i_2 = 0
            intersected = False
            loc = None
            while i_1 < len(line1) - 1:
                split1_a.append(line1[i_1])
                tmp_1 = LineString([line1[i_1], line1[i_1 + 1]])
                i_2 = 0
                while intersected is False and i_2 < len(line2) - 1:
                    tmp_2 = LineString([line2[i_2], line2[i_2 + 1]])
                    if tmp_1.intersects(tmp_2):
                        tmp_loc = tmp_1.intersection(tmp_2)
                        if tmp_loc == int_loc:
                            intersected = True
                            loc = [tmp_loc.x, tmp_loc.y, tmp_loc.z]
                            break
                    i_2 += 1
                if intersected is True:
                    break

                i_1 += 1

            if intersected is True:
                split1_a.append(loc)
                split1_b.append(loc)
                split1_b.extend(line1[i_1 + 1:])

                split2_a.extend(line2[:i_2 + 1])
                split2_a.append(list(loc))
                split2_b.append(list(loc))
                split2_b.extend(line2[i_2 + 1:])

        return split1_a, split1_b, split2_a, split2_b

    def _split_lines_ls(self, line1_ls, line2_ls, int_loc):
        """Creates polygons by offsetting arcs.

        Args:
            line_to_be_split (LineString): first line to be split
            int_lines (LineString): first line to be split
            int_loc (Point): the intersection location

        Returns:
            (list), (list), (list), (list): split arcs
        """
        line1 = self._pts_from_linestring(line1_ls)
        line2 = self._pts_from_linestring(line2_ls)
        return self._split_lines(line1, line2, int_loc)

    def _multi_line_split(self, line_to_be_split, int_lines):
        """Creates polygons by offsetting arcs.

        Args:
            line_to_be_split (LineString): first line to be split
            int_lines (LineString): second line to split with

        Returns:
            (GEOMETRYCOLLECTION): split arcs
        """
        dist_to_pt = dict()
        for line in int_lines:
            intersect = line_to_be_split.intersection(line)
            if intersect.geom_type == 'MultiPoint':
                for pt in intersect.geoms:
                    dist_to_pt[line_to_be_split.project(pt)] = pt
            elif intersect.geom_type == 'Point':
                dist_to_pt[line_to_be_split.project(intersect)] = intersect

        if len(dist_to_pt.keys()) == 0:
            return []

        split_lines = []
        dists = sorted(dist_to_pt.keys())
        shapely_pts = self._shapely_pts_from_linestring(line_to_be_split)
        locs = self._pts_from_linestring(line_to_be_split)
        cur_dist = 0.0
        idx = 1
        node = locs[0]
        for dist in dists:
            if dist == 0.0:
                continue
            if idx != len(locs):
                split_line = [node]
                cur_dist = line_to_be_split.project(shapely_pts[idx])
                while (cur_dist < dist and idx < len(locs) - 1):
                    split_line.append(locs[idx])
                    idx += 1
                    cur_dist = line_to_be_split.project(shapely_pts[idx])
                node = shapely_pt_to_loc(dist_to_pt[dist])
                split_line.append(node)
                split_lines.append(LineString(split_line))

        if dists[-1] != line_to_be_split.length and idx < len(locs):
            # create the last group
            split_line = [node] + locs[idx:]
            split_lines.append(LineString(split_line))

        return split_lines

    def _split_ls_at_pt(self, line_to_be_split, pt):
        """Creates polygons by offsetting arcs.

        Args:
            line_to_be_split (LineString): first line to be split
            pt (Point): point on line to split

        Returns:
            (GEOMETRYCOLLECTION): split arcs
        """
        split_lines = []

        if pt is not None:
            shapely_pts = self._shapely_pts_from_linestring(line_to_be_split)
            locs = self._pts_from_linestring(line_to_be_split)
            if pt == shapely_pts[0] or pt == shapely_pts[-1]:
                return [line_to_be_split]

            if pt in shapely_pts:
                idx = shapely_pts.index(pt)
                return [LineString(locs[:idx + 1]), LineString(locs[idx:])]

            dist = line_to_be_split.project(pt)
            if dist == 0.0 or dist == line_to_be_split.length:
                return [line_to_be_split]

            cur_dist = 0.0
            idx = 1
            split_line = [locs[0]]
            cur_dist = line_to_be_split.project(shapely_pts[idx])
            while (cur_dist < dist and idx < len(locs) - 1):
                split_line.append(locs[idx])
                idx += 1
                cur_dist = line_to_be_split.project(shapely_pts[idx])
            split_line.append(shapely_pt_to_loc(pt))
            split_lines.append(LineString(split_line))

            # create the rest of the arc
            split_line = [shapely_pt_to_loc(pt)] + locs[idx:]
            split_lines.append(LineString(split_line))

        return split_lines

    def merge_node_intersections(self, node_group, fake_arcs):
        """Creates polygons by offsetting arcs.

        Args:
            node_group (set): id's of the nodes with the intersecting arcs
            fake_arcs (list): arcs added to connect the group (must be deleted after)
        """
        orig_set_of_arcs = []
        ordered_groups = []
        tot_arcs = 0
        orig_ordered_arcs = []
        for node_id in node_group:
            arc_list = self._node_to_arc[node_id]
            if len(arc_list) > 0:
                ordered_groups.append(self._get_ordered_list(arc_list, node_id))
                tot_arcs += len(arc_list)
                orig_set_of_arcs.extend(arc_list)
                orig_ordered_arcs.extend(ordered_groups[-1])

        all_ordered_arcs = []
        if len(ordered_groups) > 1:
            # get the clockwise order for each node in the group
            starting_group_index = 0
            for i, group in enumerate(ordered_groups):
                if len(group) > 2:
                    starting_group_index = i
                    break

            all_ordered_arcs = ordered_groups[starting_group_index]
            # now ignore the first list because it is put in the "all" list
            ordered_groups.remove(ordered_groups[starting_group_index])

            # if there are multiple lists, merge them into one master list
            changed = True
            while changed is True:
                changed = False
                revisit = []
                for cur_list in ordered_groups:
                    merged, all_ordered_arcs = self.merge_arc_lists(all_ordered_arcs, cur_list, fake_arcs)
                    if merged is False:
                        revisit.append(cur_list)
                    else:
                        changed = True
                ordered_groups = revisit

            if len(ordered_groups) != 0:
                # we need to work in the groups that are not touching, but are close enough that offsets intersect
                # find the closest node and connect them, then rerun
                unmatched_id = -1
                group_index = 0
                while unmatched_id not in node_group and group_index < len(ordered_groups):
                    arc_index = 0
                    while unmatched_id not in node_group and arc_index < len(ordered_groups[group_index]):
                        unmatched_data = self._get_arc_data_from_arc(ordered_groups[group_index][arc_index][0])
                        unmatched_loc = unmatched_data.get_arc_end_loc(ordered_groups[group_index][arc_index][1])
                        unmatched_pt = loc_to_shapely_pt(unmatched_loc)
                        unmatched_id = unmatched_data.start_node
                        if ordered_groups[group_index][arc_index][1] is False:
                            unmatched_id = unmatched_data.end_node
                        arc_index += 1
                    group_index += 1

                min_dist = unmatched_data.number_elements * unmatched_data.element_width
                closest_node = None
                node_id = -1
                for arc in orig_ordered_arcs:
                    arc_data = self._get_arc_data_from_arc(arc[0])
                    node_loc = arc_data.get_arc_end_loc(arc[1])
                    node_pt = loc_to_shapely_pt(node_loc)
                    dist = unmatched_pt.distance(node_pt)
                    if dist < min_dist and dist > 0.0:
                        closest_node = node_pt
                        min_dist = dist
                        node_id = arc_data.start_node
                        if arc[1] is False:
                            node_id = arc_data.end_node

                if closest_node is not None:
                    # create an arc
                    node_loc = shapely_pt_to_loc(closest_node)
                    # get the id
                    id = len(self._arc_to_data)
                    while id in self._arc_to_data.keys():
                        id += 1
                    data = {'id': id,
                            'arc_pts': [unmatched_loc, node_loc],
                            'cov_geom': None,
                            'start_node': unmatched_id,
                            'end_node': node_id,
                            'element_width': arc_data.element_width,
                            'number_elements': arc_data.number_elements,
                            'bias': arc_data.bias}

                    arc_data = ArcData(data)

                    # add it to arc data
                    self._arc_to_data[id] = arc_data

                    # add it to nodes
                    self._node_to_arc[unmatched_id].append(data)
                    self._node_to_arc[node_id].append(data)

                    # call it again now with them connected
                    fake_arcs.append(id)
                    return self.merge_node_intersections(node_group, fake_arcs)
        else:
            all_ordered_arcs = ordered_groups[0]

        # only use valid arcs
        arcs_to_use = []
        for arc_pair in all_ordered_arcs:
            arc_data = self._get_arc_data_from_arc(arc_pair[0])
            if arc_data.has_valid_offsets():
                arcs_to_use.append(arc_pair)

        # check if any arcs are contained by another arc's offsets
        min_dist = (self._arc_data.element_width * self._arc_data.number_elements) / 2.0
        done = False
        while done is False and len(arcs_to_use) > 1:
            done = True
            for i, arc_pair in enumerate(arcs_to_use):
                arc_data = self._get_arc_data_from_arc(arc_pair[0])
                if arc_data.has_valid_offsets() is False:
                    arcs_to_use.remove(arc_pair)
                    done = False
                    break
                cur_arc_ls = LineString(arc_data.arc_pts)
                if cur_arc_ls.length <= min_dist:
                    arcs_to_use.remove(arcs_to_use[i])
                    self._remove_arc_from_list(arc_data.id)
                    self._short_arcs.add(arc_data.id)
                    done = False
                    break

                tmp_arc_pts = arc_data.arc_pts.copy()
                tmp_l_pts = arc_data.l_chan.copy()
                tmp_r_pts = arc_data.r_chan.copy()
                arc_is_up = arc_pair[1]
                if arc_is_up:
                    tmp_arc_pts.remove(tmp_arc_pts[0])
                    tmp_l_pts.remove(tmp_l_pts[0])
                    tmp_r_pts.remove(tmp_r_pts[0])
                else:
                    tmp_arc_pts.remove(tmp_arc_pts[-1])
                    tmp_l_pts.remove(tmp_l_pts[-1])
                    tmp_r_pts.remove(tmp_r_pts[-1])

                prev_i = i - 1 if i > 0 else len(arcs_to_use) - 1
                next_i = i + 1 if i < len(arcs_to_use) - 1 else 0

                next_arc_data = self._get_arc_data_from_arc(arcs_to_use[next_i][0])
                next_is_up = arcs_to_use[next_i][1]
                prev_arc_data = self._get_arc_data_from_arc(arcs_to_use[prev_i][0])
                prev_is_up = arcs_to_use[prev_i][1]

                prev_poly = next_poly = None
                if prev_arc_data.id != arc_data.id:
                    prev_poly = self._shapely_poly_from_arc_data(prev_arc_data)
                if next_arc_data.id != arc_data.id:
                    next_poly = self._shapely_poly_from_arc_data(next_arc_data)

                lines_to_check = [tmp_l_pts, tmp_arc_pts, tmp_r_pts]
                cur_other_node = arc_data.end_node if arc_is_up else arc_data.start_node
                cur_is_dangling = len(self._node_to_arc[cur_other_node]) == 1
                for line in lines_to_check:
                    shp_geom = None
                    if len(line) > 1:
                        shp_geom = LineString(line)
                    else:
                        shp_geom = loc_to_shapely_pt(line[0])

                    if prev_poly and shp_geom.covered_by(prev_poly):
                        # is one of these a dangling arc?
                        prev_other_node = prev_arc_data.end_node if prev_is_up else prev_arc_data.start_node
                        prev_is_dangling = len(self._node_to_arc[prev_other_node]) == 1
                        if prev_is_dangling and not cur_is_dangling:
                            self._remove_arc_from_list(prev_arc_data.id)
                            arcs_to_use.remove(arcs_to_use[prev_i])
                        elif cur_is_dangling and not prev_is_dangling:
                            self._remove_arc_from_list(arc_data.id)
                            arcs_to_use.remove(arcs_to_use[i])
                        else:
                            cur_ls = LineString(arc_data.arc_pts)
                            prev_ls = LineString(prev_arc_data.arc_pts)
                            if cur_ls.length > prev_ls.length:
                                self._remove_arc_from_list(prev_arc_data.id)
                                arcs_to_use.remove(arcs_to_use[prev_i])
                            else:
                                self._remove_arc_from_list(arc_data.id)
                                arcs_to_use.remove(arcs_to_use[i])
                        self._overlapping_arcs.add(arc_data.id)
                        self._overlapping_arcs.add(prev_arc_data.id)
                        done = False
                        break
                    elif next_poly and shp_geom.covered_by(next_poly):
                        next_other_node = next_arc_data.end_node if next_is_up else next_arc_data.start_node
                        next_is_dangling = len(self._node_to_arc[next_other_node]) == 1
                        if next_is_dangling and not cur_is_dangling:
                            self._remove_arc_from_list(next_arc_data.id)
                            arcs_to_use.remove(arcs_to_use[next_i])
                        elif cur_is_dangling and not next_is_dangling:
                            self._remove_arc_from_list(arc_data.id)
                            arcs_to_use.remove(arcs_to_use[i])
                        else:
                            cur_ls = LineString(arc_data.arc_pts)
                            next_ls = LineString(next_arc_data.arc_pts)
                            if cur_ls.length > next_ls.length:
                                self._remove_arc_from_list(next_arc_data.id)
                                arcs_to_use.remove(arcs_to_use[next_i])
                            else:
                                self._remove_arc_from_list(arc_data.id)
                                arcs_to_use.remove(arcs_to_use[i])
                        self._overlapping_arcs.add(arc_data.id)
                        self._overlapping_arcs.add(next_arc_data.id)
                        done = False
                        break
                if done is False:
                    break

        if len(arcs_to_use) < 2:
            return

        # now we have the list in order of how to connect, do intersections
        self._confluence_nodes.update(node_group)
        cur_data = self._get_arc_data_from_arc(arcs_to_use[0][0])
        cur_is_up = arcs_to_use[0][1]
        for i in range(len(arcs_to_use)):
            if i == len(arcs_to_use) - 1:
                next_arc = arcs_to_use[0][0]
                next_is_up = arcs_to_use[0][1]
            else:
                next_arc = arcs_to_use[i + 1][0]
                next_is_up = arcs_to_use[i + 1][1]
            next_data = self._get_arc_data_from_arc(next_arc)
            if next_data.has_valid_offsets() is True:
                self._intersect_offsets(cur_data, cur_is_up, next_data, next_is_up)
                cur_data = next_data
                cur_is_up = next_is_up

        # now that we have all of the intersection points, trim the offsets
        for i in range(len(arcs_to_use)):
            cur_data = self._get_arc_data_from_arc(arcs_to_use[i][0])
            is_lefts = [True, False]
            is_up = arcs_to_use[i][1]
            for is_left in is_lefts:
                if cur_data.has_valid_offsets():
                    int_pt = cur_data.get_trim_offset_pt(is_left, is_up)
                    if int_pt is None:
                        continue
                    offset_locs = cur_data.get_offset_pts(is_left)
                    offset_ls = LineString(offset_locs)

                    # handle differently if it is a loop intersecting with only one other arc
                    splits = []
                    if cur_data.is_loop() and cur_data.intersects_both_ends(is_left) and len(arcs_to_use) == 3:
                        up_length = 0.0
                        down_length = 0.0
                        # find out which cutoff point leaves the most offset left
                        int_up = cur_data.get_trim_offset_pt(is_left, True)
                        int_down = cur_data.get_trim_offset_pt(is_left, False)
                        splits_up = self._split_ls_at_pt(offset_ls, int_up)
                        if len(splits_up) > 1:
                            up_length = splits_up[1].length
                        splits_down = self._split_ls_at_pt(offset_ls, int_down)
                        down_length = splits_down[0].length
                        if len(splits_up) > 1 and len(splits_down) > 1:
                            if up_length > down_length:
                                splits = splits_up
                                is_up = True
                            elif down_length > up_length:
                                splits = splits_down
                                is_up = False
                        elif len(splits_up) > 1:
                            splits = splits_up
                            is_up = True
                        elif len(splits_down) > 1:
                            splits = splits_down
                            is_up = False
                    else:
                        splits = self._split_ls_at_pt(offset_ls, int_pt)

                    if len(splits) != 2:
                        continue

                    if is_up is True:
                        trim_pts = self._pts_from_linestring(splits[1])
                    else:
                        trim_pts = self._pts_from_linestring(splits[0])

                    self._trim_offsets_from_intersection(cur_data, is_left, is_up, trim_pts)

        # calculate the spacing that we will use for extra arcs - this is the average of the average spacing of all
        # arcs involved in the intersection
        extra_spacing = 0.0
        for arc in arcs_to_use:
            cur_data = self._get_arc_data_from_arc(arc[0])
            extra_spacing += (LineString(cur_data.arc_pts).length / (len(cur_data.arc_pts) - 1))
        extra_spacing = extra_spacing / (len(arcs_to_use))

        # add extra arcs for where it didn't connect by intersection
        cur_data = self._get_arc_data_from_arc(arcs_to_use[0][0])
        cur_is_up = arcs_to_use[0][1]

        for i in range(len(arcs_to_use)):
            if i == len(arcs_to_use) - 1:
                next_data = self._get_arc_data_from_arc(arcs_to_use[0][0])
                next_is_up = arcs_to_use[0][1]
            else:
                next_data = self._get_arc_data_from_arc(arcs_to_use[i + 1][0])
                next_is_up = arcs_to_use[i + 1][1]

            force_close = False
            if len(arcs_to_use) == 2:
                force_close = cur_data.get_arc_end_node(cur_is_up) == next_data.get_arc_end_node(next_is_up)
                # save this so that we know not to create duplicates of the ends when we create polygons
                # the end will be created with the arc that has the smaller id, so when we get to the bigger id, we
                # want to ignore that end
                if force_close:
                    if cur_data.id == next_data.id:
                        # it's a loop - we want to add the same thing when we come around here on the "next arc"
                        # or else neither end will be created
                        self._shared_ends.add((cur_data.id, True))
                    elif cur_data.id > next_data.id:
                        self._shared_ends.add((cur_data.id, cur_is_up))
                    else:
                        self._shared_ends.add((next_data.id, next_is_up))

            self._handle_offset_intersections(cur_data, next_data, cur_is_up, next_is_up, force_close, orig_set_of_arcs,
                                              extra_spacing)
            cur_data = next_data
            cur_is_up = next_is_up

        for added_id in fake_arcs:
            self._remove_arc_from_list(added_id)

    def merge_arc_lists(self, all, to_add, fake_arcs):
        """Creates polygons by offsetting arcs.

        Args:
            all (list): list that will contain all arcs in CCW order
            to_add (list): arc list to add
            fake_arcs (list): list of arcs that were added to help with connections

        Returns:
            (bool): was it added to the list?
            (list): the merged list
        """
        # if there are multiple lists, merge them into one master list
        dist_to_matched_opposite_pair = dict()
        for arc_pair in to_add:
            opposite_pair = (arc_pair[0], not arc_pair[1])
            if opposite_pair in all:
                arc_length = LineString(opposite_pair[0]['arc_pts']).length
                dist_to_matched_opposite_pair[arc_length] = opposite_pair

        if len(dist_to_matched_opposite_pair) > 0:
            dists = sorted(dist_to_matched_opposite_pair.keys())
            opposite_pair = dist_to_matched_opposite_pair[dists[0]]
            arc_pair = (opposite_pair[0], not opposite_pair[1])
            # reorder the list and add it in place of the shared arc (the shared arc will be removed)
            index_cur = to_add.index(arc_pair)
            reordered = to_add[index_cur + 1:] + to_add[: index_cur]
            index_all = all.index(opposite_pair)

            to_remove = self._get_arc_data_from_arc(opposite_pair[0])

            if to_remove.id not in fake_arcs and to_remove.has_valid_offsets():
                # trim the adjacent arcs on both ends before deleting the arc (if it was an original arc)
                to_remove_is_up = opposite_pair[1]

                prev_arc = index_cur - 1 if index_cur > 0 else len(to_add) - 1
                prev_data = self._get_arc_data_from_arc(to_add[prev_arc][0])
                prev_is_up = to_add[prev_arc][1]
                next_arc = index_cur + 1 if index_cur < len(to_add) - 1 else 0
                next_data = self._get_arc_data_from_arc(to_add[next_arc][0])
                next_is_up = to_add[next_arc][1]

                if prev_data.has_valid_offsets():
                    self._intersect_offsets(prev_data, prev_is_up, to_remove, not to_remove_is_up)
                if next_data.has_valid_offsets():
                    self._intersect_offsets(to_remove, not to_remove_is_up, next_data, next_is_up)

                prev_arc = index_all - 1 if index_all > 0 else len(all) - 1
                prev_data = self._get_arc_data_from_arc(all[prev_arc][0])
                prev_is_up = all[prev_arc][1]
                next_arc = index_all + 1 if index_all < len(all) - 1 else 0
                next_data = self._get_arc_data_from_arc(all[next_arc][0])
                next_is_up = all[next_arc][1]

                if prev_data.has_valid_offsets():
                    self._intersect_offsets(prev_data, prev_is_up, to_remove, to_remove_is_up)
                if next_data.has_valid_offsets():
                    self._intersect_offsets(to_remove, to_remove_is_up, next_data, next_is_up)

            # insert the current list where the shared arc is in the "all" list
            all = all[: index_all] + reordered + all[index_all + 1:]

            # remove offsets for this arc so that we don't create them later
            to_remove.clear_offsets()
            return True, all

        return False, all

    def _intersect_offsets(self, arc1_data, arc1_is_up, arc2_data, arc2_is_up):
        """Creates polygons by offsetting arcs.

        Args:
            arc1_data (ArcData): arc data for first arc
            arc1_is_up (bool): is the shared node upstream
            arc2_data (ArcData): adjacent arc in the CCW direction
            arc2_is_up (bool): is the shared node upstream
        Returns:
            (Point): intersect location
        """
        ls_1 = None
        ls_1_is_left = True
        pts1 = []
        if arc1_is_up is True:
            pts1 = arc1_data.r_chan.copy()
            ls_1 = LineString(arc1_data.r_chan)
            ls_1_is_left = False
        else:
            pts1 = arc1_data.l_chan.copy()
            ls_1 = LineString(arc1_data.l_chan)

        ls_2 = None
        ls_2_is_left = True
        pts2 = []
        if arc2_is_up is True:
            pts2 = arc2_data.l_chan.copy()
            ls_2 = LineString(arc2_data.l_chan)
        else:
            pts2 = arc2_data.r_chan.copy()
            ls_2 = LineString(arc2_data.r_chan)
            ls_2_is_left = False

        # do we already have intersect points from an arc that we deleted?
        if arc1_data.get_trim_offset_pt(ls_1_is_left, arc1_is_up) is not None:
            if arc2_data.get_trim_offset_pt(ls_2_is_left, arc2_is_up) is not None:
                return None

        # do they intersect? if they do, cut them off
        intersect = self._get_ls_intersect(ls_1, ls_2, arc1_is_up)

        if intersect is None:
            tol = 0.05 * (arc1_data.number_elements * arc1_data.element_width / 2.0)
            dist = ls_1.distance(ls_2)
            offset_dist = (self._arc_data.element_width * self._arc_data.number_elements) / 2.0
            if dist <= tol:
                new_ls_1 = LineString(self._extend_arc(pts1, arc1_is_up, tol))
                new_ls_2 = LineString(self._extend_arc(pts2, arc2_is_up, tol))
                intersect = self._get_ls_intersect(new_ls_1, new_ls_2, arc1_is_up)
                if intersect is not None:
                    int_loc = shapely_pt_to_loc(intersect)
                    # is the intersection on the original line for ls1?
                    dist = ls_1.project(intersect)
                    length = ls_1.length
                    if dist == length:
                        # go to the extended point
                        arc1_data.set_offset_end_loc(ls_1_is_left, arc1_is_up, int_loc)

                    # is the intersection on the original line?
                    dist = ls_2.project(intersect)
                    length = ls_2.length
                    if dist == length:
                        # go to the extended point
                        arc2_data.set_offset_end_loc(ls_2_is_left, arc2_is_up, int_loc)
            elif dist <= offset_dist:
                pass

            return None

        arc1_data.set_trim_offset_pt(ls_1_is_left, arc1_is_up, intersect)
        arc2_data.set_trim_offset_pt(ls_2_is_left, arc2_is_up, intersect)
        return intersect

    def _get_ls_intersect(self, ls_1, ls_2, ls_1_is_up):
        """Creates polygons by offsetting arcs.

        Args:
            ls_1 (LineString): arc data for first arc
            ls_2 (LineString): is the shared node upstream
            ls_1_is_up (bool): is the shared node upstream
        Returns:
            (Point): intersect point
        """
        # do they intersect?
        intersect = ls_1.intersection(ls_2)
        if intersect is not None:
            if intersect.geom_type == 'MultiPoint':
                # what are the intersection points
                dist_to_int_dict = dict()
                for pt in intersect.geoms:
                    dist_to_int_dict[ls_1.project(pt)] = pt

                sorted_dists = sorted(dist_to_int_dict.keys())

                if ls_1_is_up:
                    intersect = dist_to_int_dict[sorted_dists[0]]
                else:
                    intersect = dist_to_int_dict[sorted_dists[-1]]

            if intersect.geom_type == 'Point':
                return intersect

        return None

    def _shapely_poly_from_arc_data(self, arc_data):
        """Orders the arcs in CCW order around the shared node.

        Args:
            arc_data (ArcData): arc data to use for building polygon
        Returns:
            (Polygon): shapely polygon formed from offsets in the arc data
        """
        arc_data.r_chan.reverse()
        all_pts = arc_data.r_chan + arc_data.l_chan
        arc_data.r_chan.reverse()
        return Polygon(all_pts)

    def _merge_two_way(self, arc_list, node_id):
        """Merges two arcs that share a node.

        Args:
            arc_list (list): list of arcs that share node
            node_id (int): id of shared node
        """
        # skip out if this is an arc looping on itself
        if arc_list[0] == arc_list[1]:
            return False

        first_arc_data = self._get_arc_data_from_arc(arc_list[0])
        first_arc_pts = first_arc_data.arc_pts
        first_node_is_up = first_arc_data.is_node_upstream(node_id)
        second_arc_data = self._get_arc_data_from_arc(arc_list[1])
        second_arc_pts = second_arc_data.arc_pts
        second_node_is_up = second_arc_data.is_node_upstream(node_id)

        # merge the points and reset the end nodes for the arc data, and change the "node_to_arc" dict
        second_node_list = []
        if first_node_is_up:
            if second_node_is_up:
                first_arc_pts.reverse()
                first_arc_data.arc_pts = first_arc_pts + second_arc_pts[1:]
                first_arc_data.start_node = first_arc_data.end_node
                first_arc_data.end_node = second_arc_data.end_node

                second_node_list = self._node_to_arc[second_arc_data.end_node]
                index = second_node_list.index(arc_list[1])
                second_node_list[index] = arc_list[0]
            else:
                first_arc_data.arc_pts = second_arc_pts + first_arc_pts[1:]
                first_arc_data.start_node = second_arc_data.start_node

                second_node_list = self._node_to_arc[second_arc_data.start_node]
                index = second_node_list.index(arc_list[1])
                second_node_list[index] = arc_list[0]
        else:
            if second_node_is_up:
                first_arc_data.arc_pts = first_arc_pts + second_arc_pts[1:]
                first_arc_data.end_node = second_arc_data.end_node

                second_node_list = self._node_to_arc[second_arc_data.end_node]
                index = second_node_list.index(arc_list[1])
                second_node_list[index] = arc_list[0]
            else:
                second_arc_pts.reverse()
                first_arc_data.arc_pts = first_arc_pts + second_arc_pts[1:]
                first_arc_data.end_node = second_arc_data.start_node

                second_node_list = self._node_to_arc[second_arc_data.start_node]
                index = second_node_list.index(arc_list[1])
                second_node_list[index] = arc_list[0]

        if first_arc_data.is_loop() is True:
            pass

        self._remove_arc_from_list(arc_list[1]['id'])
        return True

    def _trim_offsets_from_intersection(self, arc_data, is_left, is_up, new_pts):
        """Orders the arcs in CCW order around the shared node.

        Args:
            arc_data (ArcData): arc with offsets to be trimmed
            is_left (bool): is the trimming data for the left offset
            is_up (bool): is the trimming data for upstream or downstream
            new_pts (list): list of trimmed points
        """
        to_replace_pts = []
        opposite_pts = []
        arc_pts = arc_data.arc_pts
        if is_left is True:
            to_replace_pts = arc_data.l_chan
            opposite_pts = arc_data.r_chan
        else:
            to_replace_pts = arc_data.r_chan
            opposite_pts = arc_data.l_chan

        reversed = False
        dist = loc_to_shapely_pt(new_pts[0]).distance(loc_to_shapely_pt(to_replace_pts[0]))
        if dist != 0.0:
            new_pts.reverse()
            to_replace_pts.reverse()
            opposite_pts.reverse()
            arc_pts.reverse()
            reversed = True

        opposite_leftovers = []
        if len(new_pts) > 1:
            opposite_pts, opposite_leftovers = self._trim_arc_to_match(to_replace_pts, new_pts, opposite_pts)
            arc_pts, _ = self._trim_arc_to_match(to_replace_pts, new_pts, arc_pts)

        if reversed is True:
            new_pts.reverse()
            to_replace_pts.reverse()
            opposite_pts.reverse()
            opposite_leftovers.reverse()
            arc_pts.reverse()

        # set the offsets now
        if is_left is True:
            arc_data.l_chan = new_pts
            arc_data.r_chan = opposite_pts
        else:
            arc_data.r_chan = new_pts
            arc_data.l_chan = opposite_pts

        # see if the other trim point was in the leftovers
        opposite_leftovers_ls = LineString(opposite_leftovers)
        splits = self._split_ls_at_pt(opposite_leftovers_ls, arc_data.get_trim_offset_pt(not is_left, is_up))
        if len(splits) > 1:
            pts_1 = self._pts_from_linestring(splits[0])
            pts_2 = self._pts_from_linestring(splits[1])

            if pts_1[0] == opposite_pts[0] or pts_1[0] == opposite_pts[-1]:
                opposite_leftovers = pts_1
            else:
                opposite_leftovers = pts_2
        arc_data.set_trim_leftovers(not is_left, is_up, opposite_leftovers)

        # make sure that the last segment isn't too small
        if is_left:
            self._check_last_segment_length(arc_data.l_chan, arc_data.r_chan, is_up)
        else:
            self._check_last_segment_length(arc_data.r_chan, arc_data.l_chan, is_up)

        # set the orig arc points
        arc_data.arc_pts = arc_pts

    def _check_last_segment_length(self, check_points, opposite_points, is_up):
        """Orders the arcs in CCW order around the shared node.

        Args:
            check_points (list): locations to check
            opposite_points (list): locations of opposite arc
            is_up (bool): is the end in question up or down
        """
        # make sure that the last segment isn't too small
        # if it is less than 10% of the length of the adjacent segment, it will be removed
        # if it is less than 80% of the length of the adjacent segment, it will be relocated to 45% between
        # the confluence and the end of the adjacent segment (see Alan's remarks in bug 0015250)
        if len(check_points) < 3:
            return

        if is_up:
            adj_dist = distance_between_pts(check_points[1], check_points[2])
            check_dist = distance_between_pts(check_points[0], check_points[1])
            if check_dist < 0.1 * adj_dist:
                check_points.remove(check_points[1])
                if opposite_points is not None:
                    opposite_points.remove(opposite_points[1])
            elif check_dist < 0.8 * adj_dist:
                tmp_ls = LineString(check_points[0:3])
                moved_pt = tmp_ls.interpolate(0.45, True)
                check_points[1] = shapely_pt_to_loc(moved_pt)

                if opposite_points is not None:
                    tmp_ls = LineString(opposite_points[0:3])
                    moved_pt = tmp_ls.interpolate(0.45, True)
                    opposite_points[1] = shapely_pt_to_loc(moved_pt)
        else:
            adj_dist = distance_between_pts(check_points[-2], check_points[-3])
            check_dist = distance_between_pts(check_points[-1], check_points[-2])
            if check_dist < 0.1 * adj_dist:
                check_points.remove(check_points[-2])
                if opposite_points is not None:
                    opposite_points.remove(opposite_points[-2])
            elif check_dist < 0.8 * adj_dist:
                tmp_ls = LineString(check_points[-3:])
                moved_pt = tmp_ls.interpolate(0.55, True)
                check_points[-2] = shapely_pt_to_loc(moved_pt)

                if opposite_points is not None:
                    tmp_ls = LineString(opposite_points[-3:])
                    moved_pt = tmp_ls.interpolate(0.55, True)
                    opposite_points[-2] = shapely_pt_to_loc(moved_pt)

    def _trim_arc_to_match(self, orig_arc_1, trimmed_arc_1, arc_to_be_trimmed):
        """Trim "arc_to_be_trimmed" in the same way "trimmed_arc_1" was trimmed.

        Args:
            orig_arc_1 (list): locations of untrimmed arc 1
            trimmed_arc_1 (list): locations of trimmed arc 1
            arc_to_be_trimmed (list): locations of arc to be trimmed to match arc1

        Returns:
            (list): points of trim to match arc
            (list): the other portion of the arc that is trimmed off (not always needed)
        """
        num_new_pts = len(trimmed_arc_1)

        # find out the segment of the arc where the trim is
        trim_loc = trimmed_arc_1[-1]
        pt_before_trim = orig_arc_1[num_new_pts - 2]
        pt_after_trim = orig_arc_1[num_new_pts - 1]
        trim_ls = LineString([pt_before_trim, trim_loc, pt_after_trim])
        trim_pt = loc_to_shapely_pt(trim_loc)
        normal_dist = trim_ls.project(trim_pt, True)

        # now make the arc match
        new_trimmed_arc = arc_to_be_trimmed[:num_new_pts]
        trim_ls = LineString([new_trimmed_arc[-2], new_trimmed_arc[-1]])
        new_trim_pt = trim_ls.interpolate(normal_dist, True)
        new_trim_loc = shapely_pt_to_loc(new_trim_pt)
        new_trimmed_arc[-1] = new_trim_loc

        # put the trimmed off points in a list
        trimmed_off = [new_trim_loc]
        trimmed_off += arc_to_be_trimmed[num_new_pts - 1:]

        return new_trimmed_arc, trimmed_off

    def _get_ordered_list(self, arc_list, node_id):
        """Orders the arcs in CW order around the shared node.

        Args:
            arc_list (list): list of arcs that share the node
            node_id (int): node id that the arcs share

        Returns:
            (list): ordered list of arcs, paired with if the node is upstream or downstream
        """
        dir_to_arc = dict()
        dirs = []
        visited_arcs = []
        for arc in arc_list:
            arc_data = self._get_arc_data_from_arc(arc)
            is_up = arc_data.is_node_upstream(node_id)
            if arc in visited_arcs:
                # this happens if we have an arc that is looping on itself at the intersection (north_threeway case)
                is_up = not is_up
            pt_1 = None
            pt_2 = None
            if is_up is True:
                pt_1 = arc_data.arc_pts[1]
                pt_2 = arc_data.arc_pts[0]
            else:
                pt_1 = arc_data.arc_pts[-2]
                pt_2 = arc_data.arc_pts[-1]

            dx = pt_2[0] - pt_1[0]
            dy = pt_2[1] - pt_1[1]
            dir = math.atan2(dy, dx)
            dirs.append(dir)
            dir_to_arc[dir] = (arc, is_up)
            visited_arcs.append(arc)

        sorted_dirs = sorted(dirs, reverse=True)
        return [dir_to_arc[cur_dir] for cur_dir in sorted_dirs]

    def _handle_offset_intersections(self, cur_arc_data, next_arc_data, cur_is_up, next_is_up, force_close, arc_list,
                                     extra_spacing):
        """Creates polygons by offsetting arcs.

        Args:
            cur_arc_data (ArcData): arc data of the arc we are checking on
            next_arc_data (ArcData): arc data for adjacent arc that we should be intersecting with
            cur_is_up (bool): shared node is upstream on current arc
            next_is_up (bool): shared node is upstream on next arc
            force_close (bool): are we only intersecting two channels that share a node
            arc_list (list): all arcs in the intersection
            extra_spacing (float): spacing for vertices in "extra" arcs
        """
        cur_end_pt = None
        cur_end_loc = []
        if cur_is_up is True:
            cur_end_loc = cur_arc_data.r_chan[0]
            cur_end_pt = loc_to_shapely_pt(cur_end_loc)
        else:
            cur_end_loc = cur_arc_data.l_chan[-1]
            cur_end_pt = loc_to_shapely_pt(cur_end_loc)

        next_end_pt = None
        next_end_loc = []
        if next_is_up is True:
            next_end_loc = next_arc_data.l_chan[0]
            next_end_pt = loc_to_shapely_pt(next_end_loc)
        else:
            next_end_loc = next_arc_data.r_chan[-1]
            next_end_pt = loc_to_shapely_pt(next_end_loc)

        if next_end_pt == cur_end_pt:
            # nothing to do
            return

        done = False
        if cur_arc_data.has_valid_offsets() and next_arc_data.has_valid_offsets():
            tol = 6.0 * min(cur_arc_data.min_allowed_seg_length, next_arc_data.min_allowed_seg_length)
            dist = cur_end_pt.distance(next_end_pt)
            if dist < tol or force_close:
                # move the points so that they match and we don't make tiny pointless arcs
                new_pt = [(cur_end_loc[0] + next_end_loc[0]) / 2.0, (cur_end_loc[1] + next_end_loc[1]) / 2.0, 0.0]
                if cur_is_up is True:
                    cur_arc_data.r_chan[0] = new_pt
                else:
                    cur_arc_data.l_chan[-1] = new_pt
                if next_is_up is True:
                    next_arc_data.l_chan[0] = new_pt
                else:
                    next_arc_data.r_chan[-1] = new_pt
                done = True

        if done is False:
            # too far apart to merge, so we will create an arc to connect them
            # use trimmed leftovers if possible
            cur_is_left = not cur_is_up
            next_is_left = next_is_up

            cur_trimmed = cur_arc_data.get_trim_leftovers(cur_is_left, cur_is_up)
            next_trimmed = next_arc_data.get_trim_leftovers(next_is_left, next_is_up)

            extra_arc = []
            if len(cur_trimmed) > 0:
                if cur_trimmed[-1] == cur_end_loc:
                    cur_trimmed.reverse()
                    extra_arc = cur_trimmed
                elif cur_trimmed[0] == cur_end_loc:
                    extra_arc = cur_trimmed
                else:
                    extra_arc.append(cur_end_loc)
            else:
                extra_arc.append(cur_end_loc)

            if len(next_trimmed) > 0:
                if next_trimmed[0] == next_end_loc:
                    next_trimmed.reverse()
                    extra_arc.extend(next_trimmed)
                elif next_trimmed[-1] == next_end_loc:
                    extra_arc.extend(next_trimmed)
                else:
                    extra_arc.append(next_end_loc)
            else:
                tol = min(cur_arc_data.min_allowed_seg_length, next_arc_data.min_allowed_seg_length)
                if cur_end_pt.distance(next_end_pt) > tol:
                    extra_arc.append(next_end_loc)

            # make sure that the extra arc doesn't intersect channel arcs
            # if len(extra_arc) > 2:
            #     for arc in arc_list:
            #         data = self._get_arc_data_from_arc(arc)
            #         ls = LineString(data.arc_pts)
            #         extra_ls = LineString(extra_arc)
            #         if extra_ls.intersects(ls):
            #             i = 0
            #             while i < len(extra_arc) - 2 and len(extra_arc) > 2:
            #                 ls_seg = LineString([extra_arc[i], extra_arc[i + 1]])
            #                 removed = False
            #                 if ls_seg.intersects(ls):
            #                     ls_seg = LineString([extra_arc[i], extra_arc[i + 2]])
            #                     if ls_seg.intersects(ls) is False:
            #                         extra_arc.remove(extra_arc[i + 1])
            #                         i -= 1
            #                         removed = True
            #                 if removed is False:
            #                     i += 1

            # redistribute vertices on this "extra" arc to match the end arcs for polygons (adjusted for length)
            extra_arc = self._redist_extra_arc(extra_arc, extra_spacing)
            if cur_arc_data.id not in self._extra_arcs_for_node_merge.keys():
                self._extra_arcs_for_node_merge[cur_arc_data.id] = []
            self._extra_arcs_for_node_merge[cur_arc_data.id].append((next_arc_data.id, extra_arc))

    def _redist_extra_arc(self, extra_arc, extra_spacing):
        """Creates polygons by offsetting arcs.

        Args:
            extra_arc (list): locations in the extra arc
            extra_spacing (float): spacing
        Returns:
            redist_arc (list): redistributed locations
        """
        ls = LineString(extra_arc)
        extra_length = ls.length
        extra_num_segs = math.ceil(extra_length / extra_spacing)
        redist_arc = [shapely_pt_to_loc(ls.interpolate(i / extra_num_segs, True)) for i in range(extra_num_segs + 1)]
        return redist_arc

    def _check_node_for_intersections(self, node_id, node_group, max_dist):
        """Checks if node has other arcs that intersect.

        Args:
            node_id (int): id of the node with the intersecting arcs
            node_group (set): group of nodes that intersect
            max_dist (float): maximum distance for intersecting
        """
        if node_id in node_group or node_id in self._checked_nodes:
            return

        if len(self._node_to_arc[node_id]) > 1:
            node_group.add(node_id)

        cur_pt = self._node_to_shapely_pt[node_id]
        arcs_from_node = self._node_to_arc[node_id]
        cur_arcs = []
        for arc in arcs_from_node:
            arc_data = self._get_arc_data_from_arc(arc)
            if arc_data.has_valid_offsets():
                cur_arcs.append(arc_data)

        for chk_node_id in self._node_to_arc.keys():
            if chk_node_id == node_id or chk_node_id in node_group or chk_node_id in self._checked_nodes:
                continue
            chk_arcs = self._node_to_arc[chk_node_id]

            # if this is one short arc that is already part of node_group, don't add the second node - it screws it up
            if len(chk_arcs) == 1:
                chk_arc_data = self._get_arc_data_from_arc(chk_arcs[0])
                chk_ls = LineString(chk_arc_data.arc_pts)
                if chk_ls.length < max_dist:
                    continue
            chk_pt = self._node_to_shapely_pt[chk_node_id]
            dist = cur_pt.distance(chk_pt)
            if dist == 0.0:
                raise RuntimeError(f'Geometry error: Overlapping nodes. Node {node_id}. '
                                   'Try executing the "Clean" command on the coverage and try again.')
            elif dist <= max_dist:
                if len(chk_arcs) > 1:
                    node_group.add(node_id)
                    self._check_node_for_intersections(chk_node_id, node_group, max_dist)
                    node_group.add(chk_node_id)
                else:
                    # check if any offsets will intersect
                    intersected = False
                    for cur_data in cur_arcs:
                        cur_end_ls = []
                        pt_index = 0 if cur_data.is_node_upstream(node_id) is True else -1
                        cur_end_ls = LineString([cur_data.l_chan[pt_index], cur_data.r_chan[pt_index]])

                        for chk_arc in chk_arcs:
                            chk_data = self._get_arc_data_from_arc(chk_arc)
                            if chk_data in cur_arcs or not chk_data.has_valid_offsets():
                                continue
                            add = False
                            poly = self._shapely_poly_from_arc_data(chk_data)
                            if cur_end_ls.intersects(poly):
                                intersection = cur_end_ls.intersection(poly)
                                if intersection:
                                    add = True

                            if add is True:
                                node_group.add(node_id)
                                self._check_node_for_intersections(chk_node_id, node_group, max_dist)
                                node_group.add(chk_node_id)
                                intersected = True
                                break

                        if intersected is True:
                            break

        self._checked_nodes.add(node_id)

    def _get_linestring_endpoint(self, ls, is_first):
        """Creates polygons by offsetting arcs.

        Args:
            ls (LineString): the LineString
            is_first (bool): getting the first endpoint
        Returns:
            (shapely_point): shapely point from desired endpoint
        """
        return Point(ls.xy[0][0], ls.xy[1][0], 0.0) if is_first is True else Point(ls.xy[0][-1], ls.xy[1][-2], 0.0)

    def _pts_from_linestring(self, ls):
        """Creates polygons by offsetting arcs.

        Args:
            ls (LineString): the LineString

        Returns:
            (list): locations
        """
        return [[p[0], p[1], 0.0] for p in zip(*ls.coords.xy)]

    def _shapely_pts_from_linestring(self, ls):
        """Creates polygons by offsetting arcs.

        Args:
            ls (LineString): the LineString

        Returns:
            (list): shapely points
        """
        return [Point(p[0], p[1], 0.0) for p in zip(*ls.coords.xy)]

    def _get_arc_data_from_arc(self, arc):
        """Creates polygons by offsetting arcs.

        Args:
            arc (dict): arc info from the coverage

        Returns:
            data (ArcData): ArcData class
        """
        if arc['id'] not in self._arc_to_data.keys():
            self._arc_to_data[arc['id']] = ArcData(arc)

        return self._arc_to_data[arc['id']]

    def _remove_arc_from_list(self, arc_id):
        """Creates polygons by offsetting arcs.

        Args:
            arc (dict): arc info from the coverage

        Returns:
            data (ArcData): ArcData class
        """
        to_delete = None
        for arc in self._arc_data_list:
            if arc['id'] == arc_id:
                if arc in self._node_to_arc[arc['start_node']]:
                    self._node_to_arc[arc['start_node']].remove(arc)
                if arc in self._node_to_arc[arc['end_node']]:
                    self._node_to_arc[arc['end_node']].remove(arc)
                to_delete = arc
                break

        if to_delete is not None:
            del self._arc_to_data[to_delete['id']]
            self._arc_data_list.remove(to_delete)

    def _detect_potential_offset_intersections(self):
        """See if the new polygon will intersect with other polygons."""
        cur_arc_ls = LineString(self._arc_data.arc_pts)
        cur_intersecting_data = []
        offset_dist = (self._arc_data.element_width * self._arc_data.number_elements) / 2.0
        if cur_arc_ls.length < offset_dist * 3.0:
            self._checked_for_intersections.add(self._arc_data.id)
            return

        # find out which arcs have the potential to intersect
        for arc in self._arc_data_list:
            if arc['id'] in self._checked_for_intersections or arc['id'] == self._arc_data.id:
                continue
            chk_data = self._get_arc_data_from_arc(arc)
            chk_ls = LineString(chk_data.arc_pts)
            if chk_ls.length < offset_dist:
                continue

            if cur_arc_ls.distance(chk_ls) < (offset_dist * 2.0):
                cur_intersecting_data.append(chk_data)

        if len(cur_intersecting_data) == 0:
            self._checked_for_intersections.add(self._arc_data.id)
            return

        for chk_data in cur_intersecting_data:
            # if the nodes are shared, this will get handled with node groups
            chk_ls = LineString(chk_data.arc_pts)
            if chk_ls.length < offset_dist * 3.0:
                self._checked_for_intersections.add(chk_data.id)
                continue

            # find the two closest points to each other so that we can split the arcs
            near = nearest_points(cur_arc_ls, chk_ls)
            split_done = False
            try_itr = 0
            while split_done is False and try_itr < 9:
                cur_splits = []
                chk_splits = []

                # make sure we aren't just getting a node or too close to a node to be valid
                cur_min = offset_dist * 2.0
                cur_max = cur_arc_ls.length - offset_dist * 2.0
                cur_dist = cur_arc_ls.project(near[0])
                if cur_dist > cur_min and cur_dist < cur_max:
                    cur_splits = self._split_ls_at_pt(cur_arc_ls, near[0])

                chk_min = offset_dist * 2.0
                chk_max = chk_ls.length - offset_dist * 2.0
                chk_dist = chk_ls.project(near[1])
                if chk_dist > chk_min and chk_dist < chk_max:
                    chk_splits = self._split_ls_at_pt(chk_ls, near[1])

                if len(cur_splits) == 0 and len(chk_splits) == 0:
                    try_itr += 1
                    # check the endpoints
                    if try_itr == 1:
                        chk_end = loc_to_shapely_pt(chk_data.get_arc_end_loc(True))
                        dist = cur_arc_ls.distance(chk_end)
                        if dist < (offset_dist * 2.0) and dist > 0.0:
                            near = nearest_points(cur_arc_ls, chk_end)
                        else:
                            try_itr = 2
                    if try_itr == 2:
                        chk_end = loc_to_shapely_pt(chk_data.get_arc_end_loc(False))
                        dist = cur_arc_ls.distance(chk_end)
                        if dist < (offset_dist * 2.0) and dist > 0.0:
                            near = nearest_points(cur_arc_ls, chk_end)
                        else:
                            try_itr = 3
                    if try_itr == 3:
                        cur_end = loc_to_shapely_pt(self._arc_data.get_arc_end_loc(True))
                        dist = chk_ls.distance(cur_end)
                        if dist < (offset_dist * 2.0) and dist > 0.0:
                            near = nearest_points(cur_end, chk_ls)
                        else:
                            try_itr = 4
                    if try_itr == 4:
                        cur_end = loc_to_shapely_pt(self._arc_data.get_arc_end_loc(False))
                        dist = chk_ls.distance(cur_end)
                        if dist < (offset_dist * 2.0) and dist > 0.0:
                            near = nearest_points(cur_end, chk_ls)
                        else:
                            try_itr = 5
                    if try_itr == 5:
                        if len(self._arc_data.arc_pts) > 3:
                            num_pts = int(len(self._arc_data.arc_pts) / 2)
                            tmp_cur_arc_ls = LineString(self._arc_data.arc_pts[:num_pts])
                            dist = tmp_cur_arc_ls.distance(chk_ls)
                            if dist < (offset_dist * 2.0) and dist > 0.0:
                                near = nearest_points(tmp_cur_arc_ls, chk_ls)
                            else:
                                try_itr = 6
                        else:
                            try_itr = 7
                    if try_itr == 6:
                        num_pts = int(len(self._arc_data.arc_pts) / 2)
                        tmp_cur_arc_ls = LineString(self._arc_data.arc_pts[num_pts:])
                        dist = tmp_cur_arc_ls.distance(chk_ls)
                        if dist < (offset_dist * 2.0) and dist > 0.0:
                            near = nearest_points(tmp_cur_arc_ls, chk_ls)
                        else:
                            try_itr = 7
                    if try_itr == 7:
                        if len(chk_data.arc_pts) > 3:
                            num_pts = int(len(chk_data.arc_pts) / 2)
                            tmp_chk_arc_ls = LineString(chk_data.arc_pts[:num_pts])
                            dist = tmp_chk_arc_ls.distance(cur_arc_ls)
                            if dist < (offset_dist * 2.0) and dist > 0.0:
                                near = nearest_points(cur_arc_ls, tmp_chk_arc_ls)
                            else:
                                try_itr = 8
                        else:
                            try_itr = 9
                    if try_itr == 8:
                        num_pts = int(len(chk_data.arc_pts) / 2)
                        tmp_chk_arc_ls = LineString(chk_data.arc_pts[num_pts:])
                        dist = tmp_chk_arc_ls.distance(cur_arc_ls)
                        if dist < (offset_dist * 2.0) and dist > 0.0:
                            near = nearest_points(cur_arc_ls, tmp_chk_arc_ls)
                        else:
                            try_itr = 9
                else:
                    split_done = True

            run_again = False
            if split_done is True:
                # remove the existing arcs and add the split arcs if we have valid point and did the split
                splits_list = [cur_splits, chk_splits]
                data_list = [self._arc_data, chk_data]
                for i, splits in enumerate(splits_list):
                    data = data_list[i]
                    if self._split_arc(splits, data) is True:
                        self._overlapping_arcs.add(self._arc_data.id)
                        self._overlapping_arcs.add(chk_data.id)
                        # run this again
                        if i == 0:
                            run_again = True

            if run_again is True:
                self._detect_potential_offset_intersections()
                return

    def _split_loop(self):
        # just split the current arc halfway
        split_pt = loc_to_shapely_pt(self._arc_data.arc_pts[int(len(self._arc_data.arc_pts) / 2)])
        ls = LineString(self._arc_data.arc_pts)
        splits = self._split_ls_at_pt(ls, split_pt)

        self._split_arc(splits, self._arc_data)

    def _split_arc(self, splits, data):
        """Creates polygons by offsetting arcs.

        Args:
            splits (list): list of split segments
            data (ArcData): data of arc to be split
        Returns:
            (bool): was the arc split
        """
        # remove the existing arc and add the split arcs if we have valid point and did the split
        offset_dist = (self._arc_data.element_width * self._arc_data.number_elements) / 2.0
        did_split = False
        if len(splits) > 1 and splits[0].length > offset_dist * 2.5 and splits[1].length > offset_dist * 2.5:
            orig_end_node = data.end_node
            orig_start_node = data.start_node
            pts1 = self._pts_from_linestring(splits[0])
            pts2 = self._pts_from_linestring(splits[1])

            # check on the segment distance on the first and last points
            if data.min_allowed_seg_length is not None:
                if distance_between_pts(pts2[0], pts2[1]) < data.min_allowed_seg_length:
                    del pts2[1]
                if distance_between_pts(pts1[-1], pts1[-2]) < data.min_allowed_seg_length:
                    del pts1[-2]

            self._node_to_shapely_pt[self._max_node_id] = loc_to_shapely_pt(pts1[-1])

            cur_data = {'id': data.id,
                        'arc_pts': pts1,
                        'cov_geom': None,
                        'start_node': orig_start_node,
                        'end_node': self._max_node_id,
                        'element_width': data.element_width,
                        'number_elements': data.number_elements,
                        'bias': data.bias}

            # update the arc data list
            for i, arc_data in enumerate(self._arc_data_list):
                if arc_data['id'] == data.id:
                    self._arc_data_list[i] = cur_data
                    break

            # replace the arc data with the updated
            updated_data = ArcData(cur_data)
            updated_data.min_allowed_seg_length = data.min_allowed_seg_length
            self._arc_to_data[data.id] = self._arc_data = updated_data

            # add the new arc - the second half of the arc we just split
            # get the id
            id = len(self._arc_to_data)
            while id in self._arc_to_data.keys():
                id += 1

            new_data = {'id': id,
                        'arc_pts': pts2,
                        'cov_geom': None,
                        'start_node': self._max_node_id,
                        'end_node': orig_end_node,
                        'element_width': data.element_width,
                        'number_elements': data.number_elements,
                        'bias': data.bias}

            # add it to arc data
            self._arc_data_list.append(new_data)
            new_arc_data = ArcData(new_data)
            new_arc_data.min_allowed_seg_length = data.min_allowed_seg_length
            self._arc_to_data[id] = new_arc_data

            # add it to nodes
            self._node_to_arc[self._max_node_id] = [cur_data, new_data]

            # update old end node
            list_arcs = self._node_to_arc[orig_end_node]
            do_first_replace = True
            replace_twice = data.is_loop()
            for i, arc in enumerate(list_arcs):
                if arc['id'] == cur_data['id']:
                    if do_first_replace:
                        self._node_to_arc[orig_end_node][i] = new_data
                        if replace_twice is False:
                            break
                        else:
                            do_first_replace = False
                    else:
                        self._node_to_arc[orig_end_node][i] = cur_data
                        break

            # update old start node
            list_arcs = self._node_to_arc[orig_start_node]
            for i, arc in enumerate(list_arcs):
                if arc['id'] == cur_data['id']:
                    self._node_to_arc[orig_start_node][i] = cur_data
                    break

            self._overlapping_arcs.add(data.id)
            self._max_node_id += 1
            did_split = True

        return did_split


def shapely_pt_to_loc(shp):
    """Calculates the dot product between 2D vectors.

    Args:
        shp (shapely_pt): shapely point

    Returns:
        (list): x, y, z location
    """
    return [shp.x, shp.y, 0.0]


def loc_to_shapely_pt(pt):
    """Calculates the dot product between 2D vectors.

    Args:
        pt (list): x,y,z location

    Returns:
        (shapely_point): shapely point
    """
    return Point(pt[0], pt[1], 0.0)


def distance_between_pts(pt1, pt2):
    """Calculates the distance between two points.

    Args:
        pt1 (list): x,y,z location
        pt2 (list): x,y,z location

    Returns:
        (float): distance between the points
    """
    return loc_to_shapely_pt(pt1).distance(loc_to_shapely_pt(pt2))


def add_vertices_to_arc(pts, new_spacing):
    """Calculates the dot product between 2D vectors.

    Args:
        pts (list): list of points
        new_spacing (float): new spacing for vertices

    Returns:
        new_pts (list): new list of points
    """
    tmp_arc = LineString(pts)
    arc_length = tmp_arc.length
    new_pts = []
    new_pts.append(pts[0])

    distance = new_spacing
    redist_pt = tmp_arc.interpolate(distance)
    while redist_pt is not None and distance <= arc_length:
        new_pts.append(shapely_pt_to_loc(redist_pt))

        distance += new_spacing
        redist_pt = tmp_arc.interpolate(distance)

    new_pts.append(pts[-1])
    return new_pts


def polygons_from_arcs(input_coverage: GeoDataFrame, element_width: float, number_of_elements: int, bias: float,
                       output_coverage_name: str, logger: logging.Logger, pinch_ends: bool,
                       min_seg_length: float, wkt: str) -> GeoDataFrame:
    """
    Converts arcs to polygons using a buffer max_dist.

    Args:
        input_coverage: The coverage to convert
        element_width: Average element/cell width
        number_of_elements: Number of elements/cells (must be even)
        bias: Bias (0.01-100.0)
        output_coverage_name: The name for the output coverage
        logger: The logger for user output
        pinch_ends: Pinch the polygon ends if the arc end is free
        min_seg_length: If not None, this value overrides the calculated minimum segment length
        wkt (string): projection info

    Returns:
        The resulting coverage
    """
    if input_coverage is not None:
        arc_data = []
        arcs = input_coverage[input_coverage['geometry_types'] == 'Arc']
        if len(arcs) == 0:
            raise RuntimeError('No arcs found in the input coverage.')

        for arc in arcs.itertuples():
            arc_points = arc.geometry.coords
            pts = [[pt[0], pt[1], pt[2]] for pt in arc_points]
            arc_data.append({'id': arc.id,
                             'arc_pts': pts,
                             'cov_geom': input_coverage,
                             'start_node': arc.start_node,
                             'end_node': arc.end_node,
                             'element_width': element_width,
                             'number_elements': number_of_elements,
                             'bias': bias})
        arcs_to_polys = PolygonsFromArcs(arc_data, output_coverage_name, logger, pinch_ends, min_seg_length, wkt)
        output_cov = arcs_to_polys.generate_coverage()

        return output_cov
