"""ManningsNFromLandUseRasterTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas

# 3. Aquaveo modules
from xms.tool_core import ALLOW_ONLY_POINT_MAPPED, ALLOW_ONLY_SCALARS, IoDirection, Tool

# 4. Local modules
from xms.tool.utilities.landuse_codes import get_ccap_mannings_n_values, get_default_mannings_n_values
from xms.tool.utilities.landuse_mapper import LanduseMapper

ARG_INPUT_LANDUSE = 0
ARG_INPUT_LANDUSE_TYPE = 1
ARG_INPUT_GRID = 2
ARG_INPUT_CSV = 3
ARG_INPUT_DEFAULT_OPTION = 4
ARG_INPUT_DEFAULT = 5
ARG_INPUT_DEFAULT_DSET = 6
ARG_INPUT_LND = 7
ARG_OUTPUT_DATASET = 8


class ManningsNFromLandUseRasterTool(Tool):
    """Tool to convert an NLCD raster to a Mannings N dataset."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Manning\'s N from Land Use Raster')

        self._grid = None
        self._grid_uuid = None
        self._builder = None
        self._dataset_vals = None
        self._mapping_table = {}
        self._default_value = 0.025
        self._default_dset = None
        self._lnd = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.raster_argument(name='landuse_raster', description='Input landuse raster'),
            self.string_argument(name='landuse_type', description='Landuse raster type',
                                 choices=['NLCD', 'C-CAP', 'Other'], value='NLCD'),
            self.grid_argument(name='grid', description='Target grid'),
            self.file_argument(name='mapping_csv', description='Landuse to Mannings N mapping table', optional=True),
            self.string_argument(name='default_value_option', description='Default Mannings N option',
                                 choices=['Constant', 'Dataset'], value='Constant'),
            self.float_argument(name='default_value', description='Default Mannings N value', value=0.025,
                                min_value=0.0),
            self.dataset_argument(name='default_value_dataset', description='Default Mannings N dataset',
                                  optional=True, filters=[ALLOW_ONLY_SCALARS, ALLOW_ONLY_POINT_MAPPED]),
            self.dataset_argument(name='locked_nodes_dataset', description='Locked nodes dataset (optional)',
                                  optional=True, filters=[ALLOW_ONLY_SCALARS, ALLOW_ONLY_POINT_MAPPED]),
            self.dataset_argument(name='mannings_dataset', description='Output Mannings N dataset', value='Land Use',
                                  io_direction=IoDirection.OUTPUT),
        ]
        self.enable_arguments(arguments)
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        # Validate input data
        land_use_type = arguments[ARG_INPUT_LANDUSE_TYPE].text_value if \
            arguments[ARG_INPUT_LANDUSE_TYPE].text_value is not None else ''
        csv_file = arguments[ARG_INPUT_CSV].text_value if arguments[ARG_INPUT_CSV].text_value is not None else ''

        # Target geometry
        grid = self.get_input_grid(arguments[ARG_INPUT_GRID].text_value)
        if not grid:
            errors[arguments[ARG_INPUT_GRID].name] = 'Could not open target grid.'
        else:
            self._grid_uuid = grid.uuid
            self._grid = grid.ugrid

        # Default Mannings N
        if arguments[ARG_INPUT_DEFAULT_OPTION].text_value == 'Dataset':
            self._default_dset = self._validate_input_dataset(arguments[ARG_INPUT_DEFAULT_DSET], errors)
            if self._default_dset and self._grid and self._default_dset.num_values != self._grid.point_count:
                errors[arguments[ARG_INPUT_DEFAULT_DSET].name] = \
                    'Number of values in default Mannings N dataset must match number of points in target geometry.'
        elif arguments[ARG_INPUT_DEFAULT].text_value == '':
            errors[arguments[ARG_INPUT_DEFAULT].name] = 'Default Mannings N value not specified.'

        # Locked nodes dataset
        if arguments[ARG_INPUT_LND].text_value:
            self._lnd = self._validate_input_dataset(arguments[ARG_INPUT_LND], errors)
            if self._lnd and self._grid and self._lnd.num_values != self._grid.point_count:
                errors[arguments[ARG_INPUT_LND].name] = \
                    'Number of values in locked nodes dataset must match number of points in target geometry.'
        # Custom mapping table CSV
        if len(csv_file) > 0:
            if not os.path.exists(csv_file):
                errors[arguments[ARG_INPUT_CSV].name] = 'Could not open Landuse to Mannings N mapping table.'
        elif land_use_type == 'Other':  # Must specify a mapping table if using custom land use type
            errors[arguments[ARG_INPUT_CSV].name] = \
                'Must select a Mannings N mapping table CSV file when land use type is "Other".'
        return errors

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        default_dset = arguments[ARG_INPUT_DEFAULT_OPTION].text_value == 'Dataset'
        arguments[ARG_INPUT_DEFAULT].hide = default_dset  # hide constant edit field if option is dataset
        arguments[ARG_INPUT_DEFAULT_DSET].hide = not default_dset  # hide dataset field if option is constant

        hide_file_sel = arguments[ARG_INPUT_LANDUSE_TYPE].text_value != 'Other'
        arguments[ARG_INPUT_CSV].hide = hide_file_sel

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        # import cProfile, pstats, io
        # pr = cProfile.Profile()
        # pr.enable()

        self.run_bounding_box_version(arguments)

        # pr.disable()
        # s = io.StringIO()
        # sortby = 'cumulative'
        # ps = pstats.Stats(pr, stream=s).sort_stats(sortby)
        # ps.print_stats()
        # self.logger.debug(f'\n************PROFILE*************\n{s.getvalue()}\n***************************\n')

    def run_bounding_box_version(self, arguments):
        """Override to run the tool.  Uses code similar to SMS to generate the Mannings N values.

        However, this version uses a simple rectangular bounding box around each node, and the dataset toolbox in
        SMS appears to be making a polygon with the surrounding adjacent nodes, so the results will differ slightly
        due to the raster pixels being read (rectangle vs. polygon).  A future implementation of this might be to
        change the bounding box code below, and get a polygon area of influence instead.

        Args:
            arguments (list): The tool arguments.
        """
        # Set up the Manning's N mapping from the built in table or text file passed in.
        self.logger.info('Reading land use mapping table...')
        self._default_value = float(arguments[ARG_INPUT_DEFAULT].text_value)
        self._setup_mapping_dataframe(arguments)

        # Calculate the weighted N values similar to SMS
        self.calculate_weighted_n_zonal(arguments)

        # Write out the dataset
        self._setup_output_dataset_builder(arguments)
        self._add_output_datasets()

    def calculate_weighted_n_zonal(self, arguments):
        """Calculates the weighted Mannings values for each point on the grid.

        Args:
            arguments (list): The tool arguments.
        """
        # Get the raster, its size, geo transform, and band
        self.logger.info('Retrieving input raster...')
        raster_file = self.get_input_raster(arguments[ARG_INPUT_LANDUSE].text_value)

        # Calculate the size function and set the dataset values default
        mannings_creator = LanduseMapper(self.logger, raster_file, self._grid, self._default_value,
                                         list(self._mapping_table.keys()), list(self._mapping_table.values()),
                                         self._default_dset, self._lnd, self.default_wkt)
        self._dataset_vals = mannings_creator.process_points()

    def _setup_output_dataset_builder(self, arguments):
        """Set up dataset builders for selected tool outputs.

        Args:
            arguments (list): The tool arguments.
        """
        # Create a place for the output dataset file
        dataset_name = arguments[ARG_OUTPUT_DATASET].text_value
        self._builder = self.get_output_dataset_writer(
            name=dataset_name,
            geom_uuid=self._grid_uuid,
        )

    def _add_output_datasets(self):
        """Add datasets created by the tool to be sent back to XMS."""
        self.logger.info('Adding output dataset...')
        if self._builder is not None:
            self.logger.info('Writing output Mannings dataset to XMDF file...')
            self._builder.write_xmdf_dataset([0.0], [self._dataset_vals])
            # Send the dataset back to XMS
            self.set_output_dataset(self._builder)

    def _setup_mapping_dataframe(self, arguments):
        """Set up dataframe holding the mapping from NLCD values to Mannings N.

        Args:
            arguments (list): The tool arguments.

        """
        if arguments[ARG_INPUT_CSV].text_value.strip("\"'") == '':
            # The optional CSV file was not entered, so use the default Mannings N values
            if arguments[ARG_INPUT_LANDUSE_TYPE].text_value == 'C-CAP':
                code_vals, descriptions, mannings_val = get_ccap_mannings_n_values()
            else:
                code_vals, descriptions, mannings_val = get_default_mannings_n_values()
            mapping_df = pandas.DataFrame(data={'Description': descriptions, 'Mannings': mannings_val}, index=code_vals)
            mapping_df.loc[0, 'Mannings'] = self._default_value  # Background
            mapping_df.loc[1, 'Mannings'] = self._default_value  # Unclassified (really only exists in C-CAP)
        else:
            # The user chose a .csv file.  Read it as a dataframe, and rename the columns.
            try:
                mapping_df = pandas.read_csv(arguments[ARG_INPUT_CSV].text_value.strip("\"'"), index_col=0)
                mapping_df.columns = ['Description', 'Mannings']
            except ValueError:
                self.logger.info('Error reading Landuse to Mannings N mapping file. Aborting.')
                raise

        self._mapping_table = mapping_df.to_dict()['Mannings']  # Faster to access dict than DataFrame for this
