"""MapActivityTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import ALLOW_ONLY_SCALARS, IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.datasets.map_activity import map_activity


class MapActivityTool(Tool):
    """Tool to convert scalar datasets to a vector dataset."""
    ARG_VALUE_DATASET = 0
    ARG_ACTIVITY_DATASET = 1
    ARG_OUTPUT_DATASET = 2

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Map Activity')
        self._value_reader = None
        self._activity_reader = None
        self._builder = None
        self._value_ug = None
        self._activity_ug = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.dataset_argument(name='value_dataset', description='Value dataset', filters=ALLOW_ONLY_SCALARS),
            self.dataset_argument(name='activity_dataset', description='Activity dataset', filters=ALLOW_ONLY_SCALARS),
            self.dataset_argument(name='output_dataset', description='Output dataset', io_direction=IoDirection.OUTPUT)
        ]
        self.enable_arguments(arguments)
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        # Make sure output name prefix specified
        arg_value = arguments[self.ARG_VALUE_DATASET]
        arg_activity = arguments[self.ARG_ACTIVITY_DATASET]

        # Validate input datasets
        self._value_reader = self._validate_input_dataset(arg_value, errors)
        self._activity_reader = self._validate_input_dataset(arg_activity, errors)
        if self._value_reader is None or self._activity_reader is None:
            return errors

        # Check number of values in first timestep. Jagged arrays not allowed.
        if len(self._value_reader.values[0]) != len(self._activity_reader.values[0]):
            errors[arg_value.name] = 'Mismatched number of dataset values in the input datasets.'

        # Check number value and activity time steps are equal
        if len(self._value_reader.times) != len(self._activity_reader.times):
            errors[arg_activity.name] = 'Activity and value datasets must have the same number of time steps.'

        self._value_ug = self.get_input_dataset_grid(arg_value.text_value)
        self._activity_ug = self.get_input_dataset_grid(arg_activity.text_value)
        if self._activity_ug.uuid != self._value_ug.uuid:
            errors[arg_activity.name] = 'Activity and value datasets must come from the same mesh/grid.'

        return errors

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        pass

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        self._builder = map_activity(arguments[self.ARG_OUTPUT_DATASET].text_value, self._value_reader,
                                     self._activity_reader, self._activity_ug, self.logger)
        self._add_output_datasets()

    def _add_output_datasets(self):
        """Add datasets created by the tool to be sent back to XMS."""
        if self._builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._builder)

# def main():
#     """Main function, for testing."""
#     from xms.tool_gui.tool_dialog import ToolDialog
#     from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists
#
#     qapp = ensure_qapplication_exists()
#     tool = MapActivityTool()
#     arguments = tool.initial_arguments()
#     tool_dialog = ToolDialog(None, arguments, tool.name, tool=tool)
#     if tool_dialog.exec():
#         tool.run_tool(tool_dialog.tool_arguments)
#
#
# if __name__ == "__main__":
#     main()
