"""ScalarsFromVectorsTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import ALLOW_ONLY_VECTORS, IoDirection, Tool
from xms.tool_core.argument import contains_invalid_characters

# 4. Local modules
from xms.tool.algorithms.datasets.scalars_from_vectors import scalars_from_vectors


class ScalarsFromVectorsTool(Tool):
    """Tool to convert a vector dataset to datasets of scalars."""
    ARG_INPUT_DSET = 0
    ARG_OUTPUT_PREFIX = 1
    ARG_MAG_BOOL = 2
    ARG_DIR_BOOL = 3
    ARG_VX_BOOL = 4
    ARG_VY_BOOL = 5

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Scalars from Vectors')
        self._dataset_reader = None
        self._mag_builder = None
        self._dir_builder = None
        self._vx_builder = None
        self._vy_builder = None
        self._ugrid = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.dataset_argument(name='input_dataset', description='Input vector dataset',
                                  filters=ALLOW_ONLY_VECTORS),
            self.string_argument(name='default_name', description='Dataset name prefix', value="new dataset"),
            self.dataset_argument(name='magnitude_bool', description='Magnitude', io_direction=IoDirection.OUTPUT,
                                  optional=True, value='Mag'),
            self.dataset_argument(name='direction_bool', description='Direction', io_direction=IoDirection.OUTPUT,
                                  optional=True, value='Dir'),
            self.dataset_argument(name='vx_bool', description='Vx', io_direction=IoDirection.OUTPUT,
                                  optional=True, value='Vx'),
            self.dataset_argument(name='vy_bool', description='Vy', io_direction=IoDirection.OUTPUT,
                                  optional=True, value='Vy'),
        ]
        self.enable_arguments(arguments)
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}

        # Validate input datasets
        self._dataset_reader = self._validate_input_dataset(arguments[self.ARG_INPUT_DSET], errors)
        self._ugrid = self.get_input_dataset_grid(arguments[self.ARG_INPUT_DSET].text_value)

        # Make sure the output prefix does not contain any illegal tree item name characters.
        err = contains_invalid_characters(arguments[self.ARG_OUTPUT_PREFIX], allow_forward_slash=True,
                                          only_check_output=False)
        if err:
            errors[arguments[self.ARG_OUTPUT_PREFIX].name] = err

        # Make sure at least one scalar output selected.
        if (not arguments[self.ARG_MAG_BOOL].value and not arguments[self.ARG_DIR_BOOL].value and not
                arguments[self.ARG_VX_BOOL].value and not arguments[self.ARG_VY_BOOL].value):
            errors[arguments[self.ARG_MAG_BOOL].name] = 'No output scalar datasets selected.'
            errors[arguments[self.ARG_DIR_BOOL].name] = 'No output scalar datasets selected.'
            errors[arguments[self.ARG_VX_BOOL].name] = 'No output scalar datasets selected.'
            errors[arguments[self.ARG_VY_BOOL].name] = 'No output scalar datasets selected.'

        return errors

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        pass

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        # Get algorithm inputs
        input_dataset_grid = self.get_input_dataset_grid(arguments[self.ARG_INPUT_DSET].text_value)
        dataset_prefix = arguments[self.ARG_OUTPUT_PREFIX].text_value
        mag_name = arguments[self.ARG_MAG_BOOL].value
        dir_name = arguments[self.ARG_DIR_BOOL].value
        vx_name = arguments[self.ARG_VX_BOOL].value
        vy_name = arguments[self.ARG_VY_BOOL].value

        # Run algorithm
        self._mag_builder, self._dir_builder, self._vx_builder, self._vy_builder = scalars_from_vectors(
            self._dataset_reader, input_dataset_grid, dataset_prefix, mag_name, dir_name, vx_name, vy_name, self.logger)

        # Send data back to xms
        self._add_output_datasets()

    def _add_output_datasets(self):
        """Add datasets created by the tool to be sent back to XMS."""
        if self._mag_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._mag_builder)
        if self._dir_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._dir_builder)
        if self._vx_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._vx_builder)
        if self._vy_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._vy_builder)


# def main():
#     """Main function, for testing."""
#     from xms.tool_gui.tool_dialog import ToolDialog
#     from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists
#
#     ensure_qapplication_exists()
#     tool = ScalarsFromVectorsTool()
#     tool.set_gui_data_folder(os.path.join(get_test_files_path(), 'scalars_from_vectors_tool'))
#     arguments = tool.initial_arguments()
#     arguments[ARG_INPUT_DSET].value = 'simple'
#     arguments[ARG_OUTPUT_PREFIX].value = 'blabber'
#     arguments[ARG_MAG_BOOL].value = True
#     arguments[ARG_DIR_BOOL].value = True
#     arguments[ARG_VX_BOOL].value = True
#     arguments[ARG_VY_BOOL].value = True
#     tool_dialog = ToolDialog(None, tool, arguments, 'Scalars from Vectors')
#     if tool_dialog.exec():
#         tool.run_tool(tool_dialog.tool_arguments)
#
#
# if __name__ == "__main__":
#     main()
