"""ExportLandXmlTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.gdal.utilities import gdal_utils as gu
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.ugrids.landxml_exporter import LandXmlExporter


class ExportLandXmlTool(Tool):
    """Tool to export a grid to various file formats."""

    ARG_INPUT_GRID = 0  # Ugrid
    ARG_INPUT_COVERAGE = 1  # Coverage
    ARG_OUTPUT_LANDXML = 2  # Output file

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Export LandXML')
        self._input_ugrid = None
        self._input_coverage = None
        self._horizontal_units = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        if os.name == 'nt':  # Windows
            documents_path = os.path.join(os.environ.get('USERPROFILE', ''), 'Documents')
        else:  # Linux/macOS (Testing)
            documents_path = os.path.join(os.environ.get('HOME', ''), 'Documents')  # pragma no cover
        path = os.path.join(documents_path, 'land.xml')
        arguments = [
            self.grid_argument(name='input_ugrid', description='UGrid', io_direction=IoDirection.INPUT),
            self.coverage_argument(name='input_arcs_coverage',
                                   description='(Optional) Input coverage containing arcs to determine breaklines',
                                   optional=True),
            self.file_argument(name='output_file', description='Output file', file_filter='XML file (*.xml)',
                               default_suffix='xml', io_direction=IoDirection.OUTPUT, value=path),
        ]
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        self._input_ugrid = self.get_input_grid(arguments[self.ARG_INPUT_GRID].text_value)
        if self._input_ugrid is None:
            msg = 'Unable to load UGrid. Aborting'
            errors[arguments[self.ARG_INPUT_GRID].name] = msg
            return errors
        elif not self._input_ugrid.ugrid.cell_count > 0:
            msg = 'UGrid does not have any cells. Aborting'
            errors[arguments[self.ARG_INPUT_GRID].name] = msg
            return errors
        elif not self._input_ugrid.check_all_cells_2d():
            grid_name = arguments[self.ARG_INPUT_GRID].name
            grid_value = arguments[self.ARG_INPUT_GRID].value
            errors[grid_name] = f'The "{grid_value}" is not a 2D UGrid'
            return errors

        self._input_coverage = self.get_input_coverage(arguments[self.ARG_INPUT_COVERAGE].value)
        if self._input_coverage is not None and not (self._input_coverage['geometry_types'] == 'Arc').any():
            errors[arguments[self.ARG_OUTPUT_LANDXML].name] = (f'The "{arguments[self.ARG_INPUT_COVERAGE].value}" '
                                                               'coverage does not have any arcs.')

        self._horizontal_units = gu.get_horiz_unit_from_wkt(self.default_wkt)
        return errors

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        input_ugrid_name = arguments[self.ARG_INPUT_GRID].text_value.rsplit('/', 1)[-1]
        output_path = arguments[self.ARG_OUTPUT_LANDXML].value

        self.logger.info('Initializing UGrid and coverage data.')
        algorithm = LandXmlExporter(self._input_ugrid, input_ugrid_name, self._input_coverage,
                                    output_path, self._horizontal_units, self.logger)
        algorithm.export()
