"""StringArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from os import getcwd as os_getcwd
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection


class FileArgument(Argument):
    """File tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: str | None = None,
        hide: bool = False,
        file_filter: str = 'All files (*.*)',
        default_suffix: str = '',
        select_folder: bool = False
    ):
        """Construct a file argument.

        Args:
            name (str): Python friendly argument name.
            description (str): User friendly description of the argument.
            io_direction (IoDirection): IO Direction of the argument (input or output).
            optional (bool): Is the argument optional?
            value (Optional[string]): Default value.
            hide (bool): Should the argument be hidden (True) or visible?
            file_filter (str): Allowed file extensions
            default_suffix (str): Default file extension added if none is provided
            select_folder (bool): Should the argument select a folder instead of a file?
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        if self.io_direction == IoDirection.OUTPUT:
            self.optional = True  # output doesn't need to be specified
        self.file_filter = file_filter
        self.default_suffix = default_suffix
        self.select_folder = select_folder

    def _get_type(self):
        """Get a string representing the argument type (string).

        Returns:
            (str): The argument type.
        """
        return 'file'

    def _set_value(self, value):
        """Set the argument value.

        Args:
            value (obj): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        interface_info = {}
        if self.io_direction == IoDirection.INPUT:
            if self.select_folder:
                interface_info['type'] = 'SelectFolder'
            else:
                interface_info['type'] = 'SelectFile'
        else:
            if self.select_folder:
                interface_info['type'] = 'SaveFolder'
            else:
                interface_info['type'] = 'SaveFile'
                self.value = _resolved_full_filename(self.value)
        interface_info['file_filter'] = self.file_filter
        interface_info['default_suffix'] = self.default_suffix
        interface_info = interface_info | super().get_interface_info()

        if interface_info['value'] is None:
            interface_info['value'] = ''
        interface_info['file_exists'] = Path(interface_info['value']).is_file()

        return interface_info

    def to_dict(self):
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'FileArgument'}
        super_values = super(FileArgument, self).to_dict()
        values.update(super_values)
        self._add_key_value('file_filter', self.file_filter, values)
        self._add_key_value('default_suffix', self.default_suffix, values)
        self._add_key_value('select_folder', self.select_folder, values)
        return values

    def validate(self):
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        result = super().validate()
        if result is not None:
            return result

        if not self.optional and self.io_direction == IoDirection.INPUT:
            if not os.path.isfile(str(self.value)):
                return 'Could not find file.'
        return None

    def adjust_value_from_results(self) -> None:
        """Set the value for running from previous results."""
        if self.io_direction == IoDirection.INPUT:
            file_exists = os.path.isfile(str(self.value)) if self.value is not None else False
            if not file_exists:
                self.value = None


def _resolved_full_filename(curr_name) -> str:
    """Helper function to append the default settings directory to a filename that has an invalid path."""
    default_path = Path(os_getcwd())

    # curr_name is empty
    if not curr_name:
        return ''

    input_path = Path(curr_name)

    # curr_name is an existing directory
    if input_path.is_absolute() and input_path.is_dir():
        return ''

    # curr_name is a valid filename, but not a directory
    if input_path.is_absolute() and input_path.parent.is_dir():
        return str(input_path)

    # curr_name is not an absolute filname, and the filename
    # is a directory in the default path
    if (default_path / input_path.name).is_dir():
        return ''

    # curr_name is not an absolute filename, and the filename
    # is not a directory
    return str(default_path / input_path.name)
