"""Test utilities."""

# 1. Standard Python modules
import filecmp
import os

# 2. Third party modules
import geopandas as gpd

# 3. Aquaveo modules

# 4. Local modules
from xms.tool_core.coverage_writer import CoverageWriter


def remove_file(file_path: str) -> None:
    """Remove a file if it exists.

    Args:
        file_path (str): The file path.
    """
    if os.path.isfile(file_path):
        os.remove(file_path)


def get_test_files_path() -> str:
    """Returns the full path to the xmstool_core 'tests/files' directory.

    Returns:
        (str): See description.
    """
    file_dir = os.path.dirname(os.path.realpath(__file__))
    files_path = os.path.join(file_dir, '..', '..', 'tests', 'files')
    return os.path.abspath(files_path)


def write_geodataframe_to_h5(coverage: gpd.GeoDataFrame, file_path: str) -> None:
    """Write a GeoDataFrame coverage to an .h5 file.

    Args:
        coverage (gpd.GeoDataFrame): The coverage
        file_path (str): The file path.
    """
    writer = CoverageWriter(file_path)
    writer.write(coverage)


def files_equal(first: os.PathLike | str, second: os.PathLike | str) -> bool:
    """
    Compare two files and see if they have the same content.

    Similar to filecmp.cmp(), except it defaults to a deep comparison rather than trusting file metadata.

    Args:
        first: First file to compare.
        second: Second file to compare.

    Returns:
        Whether the two files have the same content.
    """
    return filecmp.cmp(first, second, shallow=False)
