"""Code for creating polygons from arcs."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import bisect
import math

# 2. Third party modules
import numpy as np
from shapely.geometry import LineString
from shapely.geometry.point import Point as shPoint

# 3. Aquaveo modules

# 4. Local modules


def calculate_arc_offsets(arc_pts, offset_dist, join_method, mitre_limit):
    """Offset the input arc to the left and right based on the offset_dist.

        Must not be a loop
        Shapely does not distribute the points to match the original arc, so you may want to redistribute after

    Args:
        arc_pts (list): the points of the arc to offset
        offset_dist (float): offset dist - positive for to the left, negative for to the right
        join_method (string): method to use for join ('round', 'beveled', 'mitre')
        mitre_limit (float): mitre limit to be used for shapely buffer

    Returns:
        (list): the points of the offset arc
    """
    arc_line = LineString(arc_pts)
    offset_curve = arc_line.offset_curve(offset_dist, join_style=join_method, mitre_limit=mitre_limit)
    return pts_from_ls(offset_curve)


def shapely_pt_to_loc(shp):
    """Calculates the dot product between 2D vectors.

    Args:
        shp (shapely_pt): shapely point

    Returns:
        (list): x, y, z location
    """
    return [shp.x, shp.y, 0.0]


def loc_to_shapely_pt(pt):
    """Calculates the dot product between 2D vectors.

    Args:
        pt (list): x,y,z location

    Returns:
        (shapely_point): shapely point
    """
    return shPoint(pt[0], pt[1], 0.0)


def pts_from_ls(ls):
    """Creates polygons by offsetting arcs.

    Args:
        ls (LineString): the LineString

    Returns:
        (list): locations
    """
    return [[p[0], p[1], 0.0] for p in zip(*ls.coords.xy)]


def pts_from_ls_tuple(ls):
    """Creates polygons by offsetting arcs.

    Args:
        ls (LineString): the LineString

    Returns:
        (list): locations
    """
    return [(p[0], p[1], 0.0) for p in zip(*ls.coords.xy)]


def pts_from_shapely_poly(poly):
    """Creates polygons by offsetting arcs.

    Args:
        poly (Polygon): the shapely polygon

    Returns:
        (list): locations
    """
    all_locs = [[[p[0], p[1], 0.0] for p in zip(*poly.exterior.coords.xy)]]
    hole_locs = [[[p[0], p[1], 0.0] for p in zip(*hole.coords.xy)] for hole in poly.interiors]
    return all_locs + hole_locs


def pts_from_shapely_poly_tuple(poly):
    """Creates polygons by offsetting arcs.

    Args:
        poly (Polygon): the shapely polygon

    Returns:
        (list): locations
    """
    all_locs = [[(p[0], p[1], 0.0) for p in zip(*poly.exterior.coords.xy)]]
    hole_locs = [[(p[0], p[1], 0.0) for p in zip(*hole.coords.xy)] for hole in poly.interiors]
    return all_locs + hole_locs


def perim_pts_from_sh_poly(poly):
    """Creates polygons by offsetting arcs.

    Args:
        poly (Polygon): the shapely polygon

    Returns:
        (list): locations
    """
    return [[p[0], p[1], 0.0] for p in zip(*poly.exterior.coords.xy)]


def perim_pts_from_sh_poly_tuple(poly):
    """Creates polygons by offsetting arcs.

    Args:
        poly (Polygon): the shapely polygon

    Returns:
        (list): locations
    """
    return [(p[0], p[1], 0.0) for p in zip(*poly.exterior.coords.xy)]


def shapely_pts_from_ls(ls):
    """Creates polygons by offsetting arcs.

    Args:
        ls (LineString): the LineString

    Returns:
        (list): shapely points
    """
    return [shPoint(p[0], p[1], 0.0) for p in zip(*ls.coords.xy)]


def split_ls_at_pt(ls, split_shpt):
    """Creates polygons by offsetting arcs.

    Args:
        ls (LineString): ls to split
        split_shpt (shPoint): point location of split
    Returns:
        (list), (list): split arcs
    """
    # distance along the linestring where we will split
    dist = ls.project(split_shpt)

    # get the locations and shapely points
    shpts = shapely_pts_from_ls(ls)
    locs = pts_from_ls(ls)

    # get distances at each point
    dists = [ls.project(pt) for pt in shpts]
    split_idx = bisect.bisect(dists, dist)

    split_loc = shapely_pt_to_loc(split_shpt)
    split_1 = locs[:split_idx]
    split_1.append(split_loc)

    split_2 = [split_loc]
    split_2.extend(locs[split_idx:])

    return split_1, split_2


def get_angle_of_segment(pt1, pt2):
    """Redistribute the target arc.

    Args:
        pt1 (list): xyz first point
        pt2 (list): xyz second point
        pt3 (list): xyz third point
        start_pt (list): starting location for arc
        length (float): length of arc

    Returns:
        (LineString): perpendicular line
    """
    dx = pt2[0] - pt1[0]
    dy = pt2[1] - pt1[1]
    angle = math.atan2(dy, dx)
    angle = angle = np.pi * 2.0 if angle > np.pi else angle
    angle = angle + np.pi * 2.0 if angle < -np.pi else angle
    return angle


def remove_small_segments(locs, min_seg):
    """Offset the input arc to the left and right based on the offset_dist.

    Args:
        locs (list): locations along the arc
        min_seg (float): smallest segment length allowed

    Returns:
        (list): list with small segments removed
    """
    new_pts = [locs[0]]
    last_good_pt = locs[0]

    for i in range(1, len(locs)):
        if math.dist(last_good_pt, locs[i]) >= min_seg:
            new_pts.append(locs[i])
            last_good_pt = locs[i]
        elif i == len(locs) - 1:
            new_pts.remove(new_pts[-1])
            new_pts.append(locs[i])

    return new_pts


def shapely_endpoints_from_linestring(ls):
    """Creates polygons by offsetting arcs.

    Args:
        ls (LineString): the LineString
    Returns:
        (list): first and second shapely endpoints as list
    """
    return [shPoint(ls.xy[0][0], ls.xy[1][0], 0.0), shPoint(ls.xy[0][-1], ls.xy[1][-1], 0.0)]


def endpt_locs_from_ls(ls):
    """Get the locations of the linestring endpoints.

    Args:
        ls (LineString): the LineString
    Returns:
        (list): first and second shapely endpoints as list
    """
    return [(ls.xy[0][0], ls.xy[1][0], 0.0), (ls.xy[0][-1], ls.xy[1][-1], 0.0)]


def distance_between_pts(pt_1, pt_2):
    """Get the distance between two locations.

    Args:
        pt_1 (list): location of point 1
        pt_2 (list): location of point 2
    Returns:
        (float): distance between the two points
    """
    return loc_to_shapely_pt(pt_1).distance(loc_to_shapely_pt(pt_2))
